/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.pagecache.impl.muninn;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.VarHandle;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.LockSupport;
import org.eclipse.collections.api.set.ImmutableSet;
import org.eclipse.collections.api.set.primitive.IntSet;
import org.neo4j.internal.helpers.Numbers;
import org.neo4j.internal.unsafe.UnsafeUtil;
import org.neo4j.io.mem.MemoryAllocator;
import org.neo4j.io.pagecache.IOController;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.io.pagecache.PageCacheOpenOptions;
import org.neo4j.io.pagecache.PageSwapperFactory;
import org.neo4j.io.pagecache.PagedFile;
import org.neo4j.io.pagecache.buffer.IOBufferFactory;
import org.neo4j.io.pagecache.buffer.NativeIOBuffer;
import org.neo4j.io.pagecache.impl.muninn.CacheLiveLockException;
import org.neo4j.io.pagecache.impl.muninn.CursorFactory;
import org.neo4j.io.pagecache.impl.muninn.EvictionBouncer;
import org.neo4j.io.pagecache.impl.muninn.EvictionTask;
import org.neo4j.io.pagecache.impl.muninn.FreePage;
import org.neo4j.io.pagecache.impl.muninn.LatchMap;
import org.neo4j.io.pagecache.impl.muninn.MuninnPageCursor;
import org.neo4j.io.pagecache.impl.muninn.MuninnPagedFile;
import org.neo4j.io.pagecache.impl.muninn.PageList;
import org.neo4j.io.pagecache.impl.muninn.PreFetcher;
import org.neo4j.io.pagecache.impl.muninn.SwapperSet;
import org.neo4j.io.pagecache.impl.muninn.VersionStorage;
import org.neo4j.io.pagecache.impl.muninn.VictimPageReference;
import org.neo4j.io.pagecache.tracing.DatabaseFlushEvent;
import org.neo4j.io.pagecache.tracing.EvictionRunEvent;
import org.neo4j.io.pagecache.tracing.FileFlushEvent;
import org.neo4j.io.pagecache.tracing.PageCacheTracer;
import org.neo4j.io.pagecache.tracing.PageFaultEvent;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.scheduler.Group;
import org.neo4j.scheduler.JobHandle;
import org.neo4j.scheduler.JobMonitoringParams;
import org.neo4j.scheduler.JobScheduler;
import org.neo4j.time.Clocks;
import org.neo4j.time.SystemNanoClock;
import org.neo4j.util.FeatureToggles;
import org.neo4j.util.Preconditions;
import org.neo4j.util.VisibleForTesting;

public class MuninnPageCache
implements PageCache {
    public static final byte ZERO_BYTE = (byte)(FeatureToggles.flag(MuninnPageCache.class, (String)"brandedZeroByte", (boolean)false) ? 15 : 0);
    private static final int MEMORY_USE_PER_PAGE = 8224;
    private static final int percentPagesToKeepFree = FeatureToggles.getInteger(MuninnPageCache.class, (String)"percentPagesToKeepFree", (int)5);
    private static final int cooperativeEvictionLiveLockThreshold = FeatureToggles.getInteger(MuninnPageCache.class, (String)"cooperativeEvictionLiveLockThreshold", (int)100);
    private static final IOException oomException = new IOException("OutOfMemoryError encountered in the page cache background eviction thread");
    private static final FreePage shutdownSignal = new FreePage(0L);
    private static final AtomicInteger pageCacheIdCounter = new AtomicInteger();
    private final JobScheduler scheduler;
    private final SystemNanoClock clock;
    private static final List<OpenOption> ignoredOpenOptions = Arrays.asList(StandardOpenOption.APPEND, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.SPARSE);
    private static final int UNKNOWN_PAGES_TO_EVICT = -1;
    private final int pageCacheId;
    private final PageSwapperFactory swapperFactory;
    private final int cachePageSize;
    private final int pageReservedBytes;
    private final int keepFree;
    private final PageCacheTracer pageCacheTracer;
    private final IOBufferFactory bufferFactory;
    private final int faultLockStriping;
    private final boolean preallocateStoreFiles;
    private final boolean enableEvictionThread;
    private final MemoryAllocator memoryAllocator;
    private final boolean closeAllocatorOnShutdown;
    final PageList pages;
    final long victimPage;
    private volatile Object freelist;
    private static final VarHandle FREE_LIST;
    private final ConcurrentHashMap<String, MuninnPagedFile> mappedFiles;
    private volatile Thread evictionThread;
    private volatile boolean evictorParked;
    private volatile IOException evictorException;
    private volatile boolean closed;
    private boolean threadsInitialised;
    private boolean printExceptionsOnClose;

    public static long memoryRequiredForPages(long pageCount) {
        return pageCount * 8224L;
    }

    public static Configuration config(int maxPages) {
        return MuninnPageCache.config(MemoryAllocator.createAllocator(MuninnPageCache.memoryRequiredForPages(maxPages), (MemoryTracker)EmptyMemoryTracker.INSTANCE));
    }

    public static Configuration config(MemoryAllocator memoryAllocator) {
        return new Configuration(memoryAllocator, Clocks.nanoClock(), (MemoryTracker)EmptyMemoryTracker.INSTANCE, PageCacheTracer.NULL, 8192, IOBufferFactory.DISABLED_BUFFER_FACTORY, LatchMap.faultLockStriping, true, true, 16, false);
    }

    public MuninnPageCache(PageSwapperFactory swapperFactory, JobScheduler jobScheduler, Configuration configuration) {
        MuninnPageCache.verifyHacks();
        MuninnPageCache.verifyCachePageSizeIsPowerOfTwo(configuration.pageSize);
        Objects.requireNonNull(jobScheduler);
        int maxPages = MuninnPageCache.calculatePageCount(configuration.memoryAllocator, configuration.pageSize);
        this.pageCacheId = pageCacheIdCounter.incrementAndGet();
        this.swapperFactory = swapperFactory;
        this.cachePageSize = configuration.pageSize;
        this.pageReservedBytes = Preconditions.requireNonNegative((int)configuration.reservedPageSize);
        this.keepFree = MuninnPageCache.calculatePagesToKeepFree(maxPages);
        this.pageCacheTracer = configuration.pageCacheTracer;
        this.printExceptionsOnClose = true;
        this.bufferFactory = configuration.bufferFactory;
        this.victimPage = VictimPageReference.getVictimPage(this.cachePageSize, configuration.memoryTracker);
        this.pages = new PageList(maxPages, this.cachePageSize, configuration.memoryAllocator, new SwapperSet(), this.victimPage, MuninnPageCache.getBufferAlignment(this.cachePageSize));
        this.scheduler = jobScheduler;
        this.clock = configuration.clock;
        this.faultLockStriping = configuration.faultLockStriping;
        this.enableEvictionThread = configuration.enableEvictionThread;
        this.preallocateStoreFiles = configuration.preallocateStoreFiles;
        this.memoryAllocator = configuration.memoryAllocator;
        this.closeAllocatorOnShutdown = configuration.closeAllocatorOnShutdown;
        this.setFreelistHead(new AtomicInteger());
        this.pageCacheTracer.maxPages(maxPages, this.cachePageSize);
        this.mappedFiles = new ConcurrentHashMap();
    }

    private static int getBufferAlignment(int cachePageSize) {
        return Math.min(UnsafeUtil.pageSize(), cachePageSize);
    }

    private static int calculatePagesToKeepFree(int maxPages) {
        int freePages = (int)((float)maxPages * ((float)Math.min(percentPagesToKeepFree, 50) / 100.0f));
        int lowerBound = Math.min(maxPages / 2, 30);
        return Math.max(lowerBound, Math.min(freePages, 100000));
    }

    private static void verifyHacks() {
        UnsafeUtil.assertHasUnsafe();
    }

    private static void verifyCachePageSizeIsPowerOfTwo(int cachePageSize) {
        if (!Numbers.isPowerOfTwo((long)cachePageSize)) {
            throw new IllegalArgumentException("Cache page size must be a power of two, but was " + cachePageSize);
        }
    }

    private static int calculatePageCount(MemoryAllocator memoryAllocator, int cachePageSize) {
        int minimumPageCount;
        long memoryPerPage = cachePageSize + 32;
        long maxPages = memoryAllocator.availableMemory() / memoryPerPage;
        if (maxPages < (long)(minimumPageCount = 2)) {
            throw new IllegalArgumentException(String.format("Page cache must have at least %s pages (%s bytes of memory), but was given %s pages.", minimumPageCount, (long)minimumPageCount * memoryPerPage, maxPages));
        }
        maxPages = Math.min(maxPages, Integer.MAX_VALUE);
        return Math.toIntExact(maxPages);
    }

    @Override
    public synchronized PagedFile map(Path path, int filePageSize, String databaseName, ImmutableSet<OpenOption> openOptions, IOController ioController, EvictionBouncer evictionBouncer, VersionStorage versionStorage) throws IOException {
        this.assertHealthy();
        this.ensureThreadsInitialised();
        if (filePageSize > this.cachePageSize) {
            throw new IllegalArgumentException("Cannot map files with a filePageSize (" + filePageSize + ") that is greater than the cachePageSize (" + this.cachePageSize + ")");
        }
        path = path.normalize();
        boolean createIfNotExists = false;
        boolean truncateExisting = false;
        boolean deleteOnClose = false;
        boolean anyPageSize = false;
        boolean useDirectIO = false;
        boolean littleEndian = true;
        boolean multiVersioned = false;
        for (OpenOption option : openOptions) {
            if (option.equals(StandardOpenOption.CREATE)) {
                createIfNotExists = true;
                continue;
            }
            if (option.equals(StandardOpenOption.TRUNCATE_EXISTING)) {
                truncateExisting = true;
                continue;
            }
            if (option.equals(StandardOpenOption.DELETE_ON_CLOSE)) {
                deleteOnClose = true;
                continue;
            }
            if (option.equals(PageCacheOpenOptions.ANY_PAGE_SIZE)) {
                anyPageSize = true;
                continue;
            }
            if (option.equals(PageCacheOpenOptions.DIRECT)) {
                useDirectIO = true;
                continue;
            }
            if (option.equals(PageCacheOpenOptions.BIG_ENDIAN)) {
                littleEndian = false;
                continue;
            }
            if (option.equals(PageCacheOpenOptions.MULTI_VERSIONED)) {
                multiVersioned = true;
                continue;
            }
            if (ignoredOpenOptions.contains(option)) continue;
            throw new UnsupportedOperationException("Unsupported OpenOption: " + option);
        }
        String filePath = path.toString();
        MuninnPagedFile current = this.mappedFiles.get(filePath);
        if (current != null) {
            if (current.pageSize() != filePageSize && !anyPageSize) {
                String msg = "Cannot map file " + path + " with filePageSize " + filePageSize + " bytes, because it has already been mapped with a filePageSize of " + current.pageSize() + " bytes.";
                throw new IllegalArgumentException(msg);
            }
            if (current.littleEndian != littleEndian) {
                throw new IllegalArgumentException("Cannot map file " + path + " with littleEndian " + littleEndian + ", because it has already been mapped with a littleEndian " + current.littleEndian);
            }
            if (current.multiVersioned != multiVersioned) {
                throw new IllegalArgumentException("Cannot map file " + path + " with multiVersioned " + multiVersioned + ", because it has already been mapped with a multiVersioned " + current.multiVersioned);
            }
            if (truncateExisting) {
                throw new UnsupportedOperationException("Cannot truncate a file that is already mapped");
            }
            current.incrementRefCount();
            current.setDeleteOnClose(deleteOnClose);
            return current;
        }
        if (filePageSize < 8) {
            throw new IllegalArgumentException("Cannot map files with a filePageSize (" + filePageSize + ") that is less than 8 bytes");
        }
        MuninnPagedFile pagedFile = new MuninnPagedFile(path, this, filePageSize, this.swapperFactory, this.pageCacheTracer, createIfNotExists, truncateExisting, useDirectIO, this.preallocateStoreFiles, databaseName, this.faultLockStriping, ioController, evictionBouncer, multiVersioned, multiVersioned ? this.pageReservedBytes : 0, versionStorage, littleEndian);
        pagedFile.incrementRefCount();
        pagedFile.setDeleteOnClose(deleteOnClose);
        this.mappedFiles.put(filePath, pagedFile);
        this.pageCacheTracer.mappedFile(pagedFile.swapperId, pagedFile);
        return pagedFile;
    }

    @Override
    public synchronized Optional<PagedFile> getExistingMapping(Path path) throws IOException {
        this.assertHealthy();
        this.ensureThreadsInitialised();
        path = path.normalize();
        MuninnPagedFile pagedFile = this.tryGetMappingOrNull(path);
        if (pagedFile != null) {
            pagedFile.incrementRefCount();
            return Optional.of(pagedFile);
        }
        return Optional.empty();
    }

    private MuninnPagedFile tryGetMappingOrNull(Path path) {
        return this.mappedFiles.get(path.toString());
    }

    @Override
    public synchronized List<PagedFile> listExistingMappings() throws IOException {
        this.assertNotClosed();
        this.ensureThreadsInitialised();
        return this.mappedFiles.values().stream().map(PagedFile.class::cast).toList();
    }

    private void ensureThreadsInitialised() throws IOException {
        if (this.threadsInitialised) {
            return;
        }
        this.threadsInitialised = true;
        try {
            if (this.enableEvictionThread) {
                JobMonitoringParams monitoringParams = JobMonitoringParams.systemJob((String)"Eviction of pages from the page cache");
                this.scheduler.schedule(Group.PAGE_CACHE_EVICTION, monitoringParams, (Runnable)new EvictionTask(this));
            }
        }
        catch (Exception e) {
            IOException exception = new IOException(e);
            try {
                this.close();
            }
            catch (Exception closeException) {
                exception.addSuppressed(closeException);
            }
            throw exception;
        }
    }

    synchronized void unmap(MuninnPagedFile file) {
        String filePath;
        MuninnPagedFile current;
        if (file.decrementRefCount() && (current = this.mappedFiles.remove(filePath = file.path().toString())) != null) {
            this.pageCacheTracer.unmappedFile(file.swapperId, file);
            this.flushAndCloseWithoutFail(file);
        }
    }

    private void flushAndCloseWithoutFail(MuninnPagedFile file) {
        boolean flushedAndClosed = false;
        boolean printedFirstException = false;
        do {
            try {
                file.flushAndForceForClose();
                file.closeSwapper();
                flushedAndClosed = true;
            }
            catch (IOException e) {
                if (!this.printExceptionsOnClose || printedFirstException) continue;
                printedFirstException = true;
                try {
                    e.printStackTrace();
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        } while (!flushedAndClosed);
    }

    public void setPrintExceptionsOnClose(boolean enabled) {
        this.printExceptionsOnClose = enabled;
    }

    @Override
    public void flushAndForce(DatabaseFlushEvent flushEvent) throws IOException {
        List<PagedFile> files = this.listExistingMappings();
        try (FileFlushEvent ignored = flushEvent.beginFileFlush();){
            this.flushAllPagesParallel(files, IOController.DISABLED);
        }
        this.clearEvictorException();
    }

    private void flushAllPagesParallel(List<? extends PagedFile> files, IOController limiter) throws IOException {
        ArrayList<JobHandle> flushes = new ArrayList<JobHandle>(files.size());
        for (PagedFile pagedFile : files) {
            flushes.add(this.scheduler.schedule(Group.FILE_IO_HELPER, JobMonitoringParams.systemJob((String)pagedFile.getDatabaseName(), (String)("Flushing changes to file '" + pagedFile.path().getFileName() + "'")), () -> {
                try {
                    this.flushFile((MuninnPagedFile)file, limiter);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }));
        }
        for (JobHandle jobHandle : flushes) {
            try {
                jobHandle.waitTermination();
            }
            catch (InterruptedException | ExecutionException e) {
                throw new IOException(e);
            }
        }
    }

    private void flushFile(MuninnPagedFile muninnPagedFile, IOController limiter) throws IOException {
        try (FileFlushEvent flushEvent = this.pageCacheTracer.beginFileFlush(muninnPagedFile.swapper);
             NativeIOBuffer buffer = this.bufferFactory.createBuffer();){
            muninnPagedFile.flushAndForceInternal(flushEvent, false, limiter, buffer);
        }
    }

    @Override
    public synchronized void close() {
        if (this.closed) {
            return;
        }
        ConcurrentHashMap<String, MuninnPagedFile> files = this.mappedFiles;
        if (!files.isEmpty()) {
            StringBuilder msg = new StringBuilder("Cannot close the PageCache while files are still mapped:");
            for (MuninnPagedFile file : files.values()) {
                int refCount = file.getRefCount();
                msg.append("\n\t");
                msg.append(file.path());
                msg.append(" (").append(refCount);
                msg.append(refCount == 1 ? " mapping)" : " mappings)");
            }
            throw new IllegalStateException(msg.toString());
        }
        this.closed = true;
        MuninnPageCache.interrupt(this.evictionThread);
        this.evictionThread = null;
        if (this.closeAllocatorOnShutdown) {
            this.memoryAllocator.close();
        }
    }

    private static void interrupt(Thread thread) {
        if (thread != null) {
            thread.interrupt();
        }
    }

    private void assertHealthy() throws IOException {
        this.assertNotClosed();
        IOException exception = this.evictorException;
        if (exception != null) {
            throw new IOException("Exception in the page eviction thread", exception);
        }
    }

    private void assertNotClosed() {
        if (this.closed) {
            throw new IllegalStateException("The PageCache has been shut down");
        }
    }

    @Override
    public int pageSize() {
        return this.cachePageSize;
    }

    @Override
    public int pageReservedBytes(ImmutableSet<OpenOption> openOptions) {
        return openOptions.contains((Object)PageCacheOpenOptions.MULTI_VERSIONED) ? this.pageReservedBytes : 0;
    }

    @Override
    public long maxCachedPages() {
        return this.pages.getPageCount();
    }

    @Override
    public long freePages() {
        return MuninnPageCache.getFreeListSize(this.pages, this.getFreelistHead());
    }

    @Override
    public IOBufferFactory getBufferFactory() {
        return this.bufferFactory;
    }

    int getPageCacheId() {
        return this.pageCacheId;
    }

    long grabFreeAndExclusivelyLockedPage(PageFaultEvent faultEvent) throws IOException {
        Object nextPage;
        FreePage freePage;
        while (true) {
            this.assertHealthy();
            Object current = this.getFreelistHead();
            if (current == null) {
                this.unparkEvictor();
                long pageRef = this.cooperativelyEvict(faultEvent);
                if (pageRef == 0L) continue;
                return pageRef;
            }
            if (current instanceof AtomicInteger) {
                AtomicInteger counter = (AtomicInteger)current;
                int pageCount = this.pages.getPageCount();
                int pageId = counter.get();
                if (pageId < pageCount && counter.compareAndSet(pageId, pageId + 1)) {
                    faultEvent.freeListSize(pageCount - counter.get());
                    return this.pages.deref(pageId);
                }
                if (pageId < pageCount) continue;
                this.compareAndSetFreelistHead(current, null);
                continue;
            }
            if (!(current instanceof FreePage)) continue;
            freePage = (FreePage)current;
            if (freePage == shutdownSignal) {
                throw new IllegalStateException("The PageCache has been shut down.");
            }
            nextPage = freePage.next;
            if (this.compareAndSetFreelistHead(freePage, nextPage)) break;
        }
        faultEvent.freeListSize(MuninnPageCache.getFreeListSize(this.pages, nextPage));
        return freePage.pageRef;
    }

    private static int getFreeListSize(PageList pageList, Object next) {
        if (next instanceof FreePage) {
            return ((FreePage)next).count;
        }
        if (next instanceof AtomicInteger) {
            return pageList.getPageCount() - ((AtomicInteger)next).get();
        }
        return 0;
    }

    private long cooperativelyEvict(PageFaultEvent faultEvent) throws IOException {
        long pageRef;
        int iterations = 0;
        int pageCount = this.pages.getPageCount();
        int clockArm = ThreadLocalRandom.current().nextInt(pageCount);
        boolean evicted = false;
        do {
            this.assertHealthy();
            if (this.getFreelistHead() != null) {
                return 0L;
            }
            if (clockArm == pageCount) {
                if (iterations == cooperativeEvictionLiveLockThreshold) {
                    throw MuninnPageCache.cooperativeEvictionLiveLock();
                }
                ++iterations;
                clockArm = 0;
            }
            if (PageList.isLoaded(pageRef = this.pages.deref(clockArm)) && PageList.decrementUsage(pageRef)) {
                evicted = this.pages.tryEvict(pageRef, faultEvent);
            }
            ++clockArm;
        } while (!evicted);
        return pageRef;
    }

    private static CacheLiveLockException cooperativeEvictionLiveLock() {
        return new CacheLiveLockException("Live-lock encountered when trying to cooperatively evict a page during page fault. This happens when we want to access a page that is not in memory, so it has to be faulted in, but there are no free memory pages available to accept the page fault, so we have to evict an existing page, but all the in-memory pages are currently locked by other accesses. If those other access are waiting for our page fault to make progress, then we have a live-lock, and the only way we can get out of it is by throwing this exception. This should be extremely rare, but can happen if the page cache size is tiny and the number of concurrently running transactions is very high. You should be able to get around this problem by increasing the amount of memory allocated to the page cache with the `server.memory.pagecache.size` setting. Please contact Neo4j support if you need help tuning your database.");
    }

    private void unparkEvictor() {
        if (this.evictorParked) {
            this.evictorParked = false;
            LockSupport.unpark(this.evictionThread);
        }
    }

    private void parkEvictor(long parkNanos) {
        this.evictorParked = true;
        LockSupport.parkNanos(this, parkNanos);
        this.evictorParked = false;
    }

    private Object getFreelistHead() {
        return FREE_LIST.getVolatile(this);
    }

    private boolean compareAndSetFreelistHead(Object expected, Object update) {
        return FREE_LIST.compareAndSet(this, expected, update);
    }

    private void setFreelistHead(Object newFreelistHead) {
        FREE_LIST.setVolatile(this, newFreelistHead);
    }

    void continuouslySweepPages() {
        this.evictionThread = Thread.currentThread();
        int clockArm = 0;
        while (!this.closed) {
            int pageCountToEvict = this.parkUntilEvictionRequired(this.keepFree);
            EvictionRunEvent evictionRunEvent = this.pageCacheTracer.beginPageEvictions(pageCountToEvict);
            try {
                clockArm = this.evictPages(pageCountToEvict, clockArm, evictionRunEvent);
            }
            finally {
                if (evictionRunEvent == null) continue;
                evictionRunEvent.close();
            }
        }
        this.setFreelistHead(shutdownSignal);
    }

    private int parkUntilEvictionRequired(int keepFree) {
        int numberOfPagesToEvict;
        long parkNanos = TimeUnit.MILLISECONDS.toNanos(10L);
        do {
            this.parkEvictor(parkNanos);
            if (!Thread.interrupted() && !this.closed) continue;
            return 0;
        } while ((numberOfPagesToEvict = this.tryGetNumberOfPagesToEvict(keepFree)) == -1);
        return numberOfPagesToEvict;
    }

    @VisibleForTesting
    int tryGetNumberOfPagesToEvict(int keepFree) {
        Object freelistHead = this.getFreelistHead();
        if (freelistHead == null) {
            return keepFree;
        }
        if (freelistHead.getClass() == FreePage.class) {
            int availablePages = ((FreePage)freelistHead).count;
            if (availablePages < keepFree) {
                return keepFree - availablePages;
            }
        } else if (freelistHead.getClass() == AtomicInteger.class) {
            AtomicInteger counter = (AtomicInteger)freelistHead;
            long count = this.pages.getPageCount() - counter.get();
            if (count < (long)keepFree) {
                return count < 0L ? keepFree : (int)((long)keepFree - count);
            }
        }
        return -1;
    }

    int evictPages(int pageEvictionAttempts, int clockArm, EvictionRunEvent evictionRunEvent) {
        while (pageEvictionAttempts > 0 && !this.closed) {
            if (clockArm == this.pages.getPageCount()) {
                clockArm = 0;
            }
            if (this.closed) {
                return 0;
            }
            long pageRef = this.pages.deref(clockArm);
            if (PageList.isLoaded(pageRef) && PageList.decrementUsage(pageRef)) {
                try {
                    --pageEvictionAttempts;
                    if (this.pages.tryEvict(pageRef, evictionRunEvent)) {
                        this.clearEvictorException();
                        this.addFreePageToFreelist(pageRef, evictionRunEvent);
                    }
                }
                catch (IOException e) {
                    this.evictorException = e;
                }
                catch (OutOfMemoryError oom) {
                    this.evictorException = oomException;
                }
                catch (Throwable th) {
                    this.evictorException = new IOException("Eviction thread encountered a problem", th);
                }
            }
            ++clockArm;
        }
        return clockArm;
    }

    @VisibleForTesting
    String describePages() {
        StringBuilder result = new StringBuilder();
        int pageCount = this.pages.getPageCount();
        for (int pageCachePageId = 0; pageCachePageId < pageCount; ++pageCachePageId) {
            long pageRef = this.pages.deref(pageCachePageId);
            result.append("[");
            result.append("PageCachePageId: ").append(pageCachePageId).append(", ");
            result.append("PageRef: ").append(Long.toHexString(pageRef)).append(", ");
            result.append("LockWord: ").append(Long.toBinaryString(UnsafeUtil.getLongVolatile((long)pageRef))).append(", ");
            result.append("Address: ").append(Long.toHexString(UnsafeUtil.getLongVolatile((long)(pageRef + 8L)))).append(", ");
            result.append("LastTxId: ").append(Long.toHexString(UnsafeUtil.getLongVolatile((long)(pageRef + 16L)))).append(", ");
            result.append("PageBinding: ").append(Long.toHexString(UnsafeUtil.getLongVolatile((long)(pageRef + 24L))));
            result.append("]\n");
        }
        return result.toString();
    }

    void addFreePageToFreelist(long pageRef, EvictionRunEvent evictions) {
        Object current;
        assert (PageList.getPageHorizon(pageRef) == 0L);
        FreePage freePage = new FreePage(pageRef);
        int pageCount = this.pages.getPageCount();
        do {
            if ((current = this.getFreelistHead()) instanceof AtomicInteger && ((AtomicInteger)current).get() > pageCount) {
                current = null;
            }
            freePage.setNext(pageCount, current);
        } while (!this.compareAndSetFreelistHead(current, freePage));
        evictions.freeListSize(freePage.count);
    }

    void clearEvictorException() {
        if (this.evictorException != null) {
            this.evictorException = null;
        }
    }

    public String toString() {
        int pagesToEvict = this.tryGetNumberOfPagesToEvict(this.keepFree);
        return String.format("%s[pageCacheId:%d, pageSize:%d, pages:%d, pagesToEvict:%s]", this.getClass().getSimpleName(), this.pageCacheId, this.cachePageSize, this.pages.getPageCount(), pagesToEvict != -1 ? String.valueOf(pagesToEvict) : "N/A");
    }

    void sweep(SwapperSet swappers) {
        swappers.sweep((IntSet swapperIds) -> {
            int pageCount = this.pages.getPageCount();
            try (EvictionRunEvent evictionEvent = this.pageCacheTracer.beginEviction();){
                block7: for (int i = 0; i < pageCount; ++i) {
                    long pageRef = this.pages.deref(i);
                    while (swapperIds.contains(PageList.getSwapperId(pageRef))) {
                        if (!this.pages.tryEvict(pageRef, evictionEvent)) continue;
                        this.addFreePageToFreelist(pageRef, evictionEvent);
                        continue block7;
                    }
                }
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        });
    }

    void startPreFetching(MuninnPageCursor cursor, CursorFactory cursorFactory) {
        PreFetcher preFetcher = new PreFetcher(cursor, cursorFactory, this.clock);
        MuninnPagedFile pagedFile = cursor.pagedFile;
        Path fileName = pagedFile.swapper.path().getFileName();
        JobMonitoringParams monitoringParams = JobMonitoringParams.systemJob((String)pagedFile.databaseName, (String)("Pre-fetching of file '" + fileName + "'"));
        cursor.preFetcher = this.scheduler.schedule(Group.PAGE_CACHE_PRE_FETCHER, monitoringParams, (Runnable)preFetcher);
    }

    @VisibleForTesting
    int getKeepFree() {
        return this.keepFree;
    }

    static {
        try {
            MethodHandles.Lookup l = MethodHandles.lookup();
            FREE_LIST = l.findVarHandle(MuninnPageCache.class, "freelist", Object.class);
        }
        catch (ReflectiveOperationException e) {
            throw new ExceptionInInitializerError(e);
        }
    }

    public static class Configuration {
        private final MemoryAllocator memoryAllocator;
        private final SystemNanoClock clock;
        private final MemoryTracker memoryTracker;
        private final PageCacheTracer pageCacheTracer;
        private final int pageSize;
        private final IOBufferFactory bufferFactory;
        private final int faultLockStriping;
        private final boolean enableEvictionThread;
        private final boolean preallocateStoreFiles;
        private final int reservedPageSize;
        private final boolean closeAllocatorOnShutdown;

        private Configuration(MemoryAllocator memoryAllocator, SystemNanoClock clock, MemoryTracker memoryTracker, PageCacheTracer pageCacheTracer, int pageSize, IOBufferFactory bufferFactory, int faultLockStriping, boolean enableEvictionThread, boolean preallocateStoreFiles, int reservedPageSize, boolean closeAllocatorOnShutdown) {
            this.memoryAllocator = memoryAllocator;
            this.clock = clock;
            this.memoryTracker = memoryTracker;
            this.pageCacheTracer = pageCacheTracer;
            this.pageSize = pageSize;
            this.reservedPageSize = reservedPageSize;
            this.bufferFactory = bufferFactory;
            this.faultLockStriping = faultLockStriping;
            this.enableEvictionThread = enableEvictionThread;
            this.preallocateStoreFiles = preallocateStoreFiles;
            this.closeAllocatorOnShutdown = closeAllocatorOnShutdown;
        }

        public Configuration memoryAllocator(MemoryAllocator memoryAllocator) {
            return new Configuration(memoryAllocator, this.clock, this.memoryTracker, this.pageCacheTracer, this.pageSize, this.bufferFactory, this.faultLockStriping, this.enableEvictionThread, this.preallocateStoreFiles, this.reservedPageSize, this.closeAllocatorOnShutdown);
        }

        public Configuration clock(SystemNanoClock clock) {
            return new Configuration(this.memoryAllocator, clock, this.memoryTracker, this.pageCacheTracer, this.pageSize, this.bufferFactory, this.faultLockStriping, this.enableEvictionThread, this.preallocateStoreFiles, this.reservedPageSize, this.closeAllocatorOnShutdown);
        }

        public Configuration memoryTracker(MemoryTracker memoryTracker) {
            return new Configuration(this.memoryAllocator, this.clock, memoryTracker, this.pageCacheTracer, this.pageSize, this.bufferFactory, this.faultLockStriping, this.enableEvictionThread, this.preallocateStoreFiles, this.reservedPageSize, this.closeAllocatorOnShutdown);
        }

        public Configuration pageCacheTracer(PageCacheTracer pageCacheTracer) {
            return new Configuration(this.memoryAllocator, this.clock, this.memoryTracker, pageCacheTracer, this.pageSize, this.bufferFactory, this.faultLockStriping, this.enableEvictionThread, this.preallocateStoreFiles, this.reservedPageSize, this.closeAllocatorOnShutdown);
        }

        public Configuration pageSize(int pageSize) {
            return new Configuration(this.memoryAllocator, this.clock, this.memoryTracker, this.pageCacheTracer, pageSize, this.bufferFactory, this.faultLockStriping, this.enableEvictionThread, this.preallocateStoreFiles, this.reservedPageSize, this.closeAllocatorOnShutdown);
        }

        public Configuration bufferFactory(IOBufferFactory bufferFactory) {
            return new Configuration(this.memoryAllocator, this.clock, this.memoryTracker, this.pageCacheTracer, this.pageSize, bufferFactory, this.faultLockStriping, this.enableEvictionThread, this.preallocateStoreFiles, this.reservedPageSize, this.closeAllocatorOnShutdown);
        }

        public Configuration reservedPageBytes(int reservedPageBytes) {
            return new Configuration(this.memoryAllocator, this.clock, this.memoryTracker, this.pageCacheTracer, this.pageSize, this.bufferFactory, this.faultLockStriping, this.enableEvictionThread, this.preallocateStoreFiles, reservedPageBytes, this.closeAllocatorOnShutdown);
        }

        public Configuration faultLockStriping(int faultLockStriping) {
            return new Configuration(this.memoryAllocator, this.clock, this.memoryTracker, this.pageCacheTracer, this.pageSize, this.bufferFactory, faultLockStriping, this.enableEvictionThread, this.preallocateStoreFiles, this.reservedPageSize, this.closeAllocatorOnShutdown);
        }

        public Configuration disableEvictionThread() {
            return new Configuration(this.memoryAllocator, this.clock, this.memoryTracker, this.pageCacheTracer, this.pageSize, this.bufferFactory, this.faultLockStriping, false, this.preallocateStoreFiles, this.reservedPageSize, this.closeAllocatorOnShutdown);
        }

        public Configuration preallocateStoreFiles(boolean preallocateStoreFiles) {
            return new Configuration(this.memoryAllocator, this.clock, this.memoryTracker, this.pageCacheTracer, this.pageSize, this.bufferFactory, this.faultLockStriping, this.enableEvictionThread, preallocateStoreFiles, this.reservedPageSize, this.closeAllocatorOnShutdown);
        }

        public Configuration closeAllocatorOnShutdown(boolean closeAllocatorOnShutdown) {
            return new Configuration(this.memoryAllocator, this.clock, this.memoryTracker, this.pageCacheTracer, this.pageSize, this.bufferFactory, this.faultLockStriping, this.enableEvictionThread, this.preallocateStoreFiles, this.reservedPageSize, closeAllocatorOnShutdown);
        }
    }
}

