/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.io.fs;

import java.io.DataOutputStream;
import java.io.Flushable;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;

public class OutputStreamWritableChannel implements FlushableChannel {
    private final DataOutputStream dataOutputStream;

    public OutputStreamWritableChannel(OutputStream outputStream) {
        this.dataOutputStream = new DataOutputStream(outputStream);
    }

    @Override
    public Flushable prepareForFlush() throws IOException {
        return dataOutputStream;
    }

    @Override
    public FlushableChannel put(byte value) throws IOException {
        dataOutputStream.writeByte(value);
        return this;
    }

    @Override
    public FlushableChannel putShort(short value) throws IOException {
        dataOutputStream.writeShort(value);
        return this;
    }

    @Override
    public FlushableChannel putInt(int value) throws IOException {
        dataOutputStream.writeInt(value);
        return this;
    }

    @Override
    public FlushableChannel putLong(long value) throws IOException {
        dataOutputStream.writeLong(value);
        return this;
    }

    @Override
    public FlushableChannel putFloat(float value) throws IOException {
        dataOutputStream.writeFloat(value);
        return this;
    }

    @Override
    public FlushableChannel putDouble(double value) throws IOException {
        dataOutputStream.writeDouble(value);
        return this;
    }

    @Override
    public FlushableChannel put(byte[] value, int length) throws IOException {
        dataOutputStream.write(value, 0, length);
        return this;
    }

    @Override
    public FlushableChannel put(byte[] value, int offset, int length) throws IOException {
        dataOutputStream.write(value, offset, length);
        return this;
    }

    @Override
    public FlushableChannel putAll(ByteBuffer src) throws IOException {
        if (src.hasArray()) {
            dataOutputStream.write(src.array());
        } else {
            while (src.hasRemaining()) {
                dataOutputStream.writeByte(src.get());
            }
        }
        return this;
    }

    @Override
    public void close() throws IOException {
        dataOutputStream.close();
    }
}
