/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.jdbc;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.UnaryOperator;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.neo4j.jdbc.Neo4jStatement;
import org.neo4j.jdbc.Neo4jTransaction;
import org.neo4j.jdbc.Neo4jTransactionSupplier;
import org.neo4j.jdbc.ResultSetImpl;
import org.neo4j.jdbc.Warnings;
import org.neo4j.jdbc.internal.shaded.schema_name.SchemaNames;
import org.neo4j.jdbc.values.Values;

class StatementImpl
implements Neo4jStatement {
    private static final Pattern PATTERN_ENFORCE_CYPHER = Pattern.compile("(['`\"])?[^'`\"]*/\\*\\+ NEO4J FORCE_CYPHER \\*/[^'`\"]*(['`\"])?");
    private static final Logger LOGGER = Logger.getLogger("org.neo4j.jdbc.statement");
    static final int DEFAULT_BUFFER_SIZE_FOR_INCOMING_STREAMS = 4096;
    static final Charset DEFAULT_ASCII_CHARSET_FOR_INCOMING_STREAM = StandardCharsets.ISO_8859_1;
    private final Connection connection;
    private final Neo4jTransactionSupplier transactionSupplier;
    private int fetchSize = 1000;
    private int maxRows;
    private int maxFieldSize;
    protected ResultSet resultSet;
    private int updateCount = -1;
    private boolean multipleResultsApi;
    private int queryTimeout;
    protected boolean poolable;
    private boolean closeOnCompletion;
    private boolean closed;
    private final UnaryOperator<String> sqlProcessor;
    private final Warnings warnings;
    private final AtomicBoolean resultSetAcquired = new AtomicBoolean(false);
    private final Map<String, Object> transactionMetadata = new ConcurrentHashMap<String, Object>();

    StatementImpl(Connection connection, Neo4jTransactionSupplier transactionSupplier, UnaryOperator<String> sqlProcessor, Warnings localWarnings) {
        this.connection = Objects.requireNonNull(connection);
        this.transactionSupplier = Objects.requireNonNull(transactionSupplier);
        this.sqlProcessor = Objects.requireNonNullElseGet(sqlProcessor, UnaryOperator::identity);
        this.warnings = Objects.requireNonNullElseGet(localWarnings, Warnings::new);
    }

    StatementImpl() {
        this.connection = null;
        this.transactionSupplier = null;
        this.sqlProcessor = UnaryOperator.identity();
        this.warnings = new Warnings();
    }

    @Override
    public ResultSet executeQuery(String sql) throws SQLException {
        return this.executeQuery0(sql, true, Map.of());
    }

    protected final ResultSet executeQuery0(String sql, boolean applyProcessor, Map<String, Object> parameters) throws SQLException {
        this.assertIsOpen();
        this.closeResultSet();
        this.updateCount = -1;
        this.multipleResultsApi = false;
        if (applyProcessor) {
            sql = this.processSQL(sql);
        }
        Neo4jTransaction transaction = this.transactionSupplier.getTransaction(this.transactionMetadata);
        int fetchSize = this.maxRows > 0 ? Math.min(this.maxRows, this.fetchSize) : this.fetchSize;
        Neo4jTransaction.RunAndPullResponses runAndPull = transaction.runAndPull(sql, StatementImpl.getParameters(parameters), fetchSize, this.queryTimeout);
        this.resultSet = new ResultSetImpl(this, transaction, runAndPull.runResponse(), runAndPull.pullResponse(), this.fetchSize, this.maxRows, this.maxFieldSize);
        this.resultSetAcquired.set(false);
        return this.resultSet;
    }

    @Override
    public int executeUpdate(String sql) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Executing update `%s`".formatted(sql));
        return this.executeUpdate0(sql, true, Map.of());
    }

    protected final int executeUpdate0(String sql, boolean applyProcessor, Map<String, Object> parameters) throws SQLException {
        this.assertIsOpen();
        this.closeResultSet();
        this.updateCount = -1;
        this.multipleResultsApi = false;
        if (applyProcessor) {
            sql = this.processSQL(sql);
        }
        Neo4jTransaction transaction = this.transactionSupplier.getTransaction(this.transactionMetadata);
        return transaction.runAndDiscard(sql, StatementImpl.getParameters(parameters), this.queryTimeout, transaction.isAutoCommit()).resultSummary().map(Neo4jTransaction.ResultSummary::counters).map(c -> {
            int rowCount = c.nodesCreated() + c.nodesDeleted() + c.relationshipsCreated() + c.relationshipsDeleted();
            if (rowCount == 0 && c.containsUpdates()) {
                int labelsAndProperties = c.labelsAdded() + c.labelsRemoved() + c.propertiesSet();
                rowCount = labelsAndProperties > 0 ? 1 : 0;
            }
            return rowCount;
        }).orElse(0);
    }

    @Override
    public void close() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Closing");
        if (this.closed) {
            return;
        }
        this.closeResultSet();
        this.closed = true;
    }

    @Override
    public int getMaxFieldSize() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting max field size");
        this.assertIsOpen();
        return this.maxFieldSize;
    }

    @Override
    public void setMaxFieldSize(int max) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting max field size to %d".formatted(max));
        this.assertIsOpen();
        if (max < 0) {
            throw new SQLException("Max field size can not be negative");
        }
        this.maxFieldSize = max;
    }

    @Override
    public int getMaxRows() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting max rows");
        this.assertIsOpen();
        return this.maxRows;
    }

    @Override
    public void setMaxRows(int max) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting max rows to %d".formatted(max));
        this.assertIsOpen();
        if (max < 0) {
            throw new SQLException("Max rows can not be negative");
        }
        this.maxRows = max;
    }

    @Override
    public void setEscapeProcessing(boolean ignored) throws SQLException {
        LOGGER.log(Level.WARNING, () -> "Setting escape processing to %s (ignored)".formatted(ignored));
        this.assertIsOpen();
    }

    @Override
    public int getQueryTimeout() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting query timeout");
        this.assertIsOpen();
        return this.queryTimeout;
    }

    @Override
    public void setQueryTimeout(int seconds) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting query timeout to %d seconds".formatted(seconds));
        this.assertIsOpen();
        if (seconds < 0) {
            throw new SQLException("Query timeout can not be negative");
        }
        this.queryTimeout = seconds;
    }

    @Override
    public void cancel() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting warnings");
        this.assertIsOpen();
        return this.warnings.get();
    }

    @Override
    public void clearWarnings() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Clearing warnings");
        this.assertIsOpen();
        this.warnings.clear();
    }

    @Override
    public void setCursorName(String name) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public boolean execute(String sql) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Executing `%s`".formatted(sql));
        return this.execute0(sql, Map.of());
    }

    protected final boolean execute0(String sql, Map<String, Object> parameters) throws SQLException {
        this.assertIsOpen();
        this.closeResultSet();
        this.updateCount = -1;
        this.multipleResultsApi = true;
        Neo4jTransaction transaction = this.transactionSupplier.getTransaction(this.transactionMetadata);
        int fetchSize = this.maxRows > 0 ? Math.min(this.maxRows, this.fetchSize) : this.fetchSize;
        Neo4jTransaction.RunAndPullResponses runAndPull = transaction.runAndPull(this.processSQL(sql), StatementImpl.getParameters(parameters), fetchSize, this.queryTimeout);
        Neo4jTransaction.PullResponse pullResponse = runAndPull.pullResponse();
        this.resultSet = new ResultSetImpl(this, transaction, runAndPull.runResponse(), pullResponse, this.fetchSize, this.maxRows, this.maxFieldSize);
        this.updateCount = pullResponse.resultSummary().map(summary -> summary.counters().totalCount()).filter(count -> count > 0).orElse(-1);
        return this.updateCount == -1;
    }

    private static Map<String, Object> getParameters(Map<String, Object> parameters) throws SQLException {
        Map result = Objects.requireNonNullElseGet(parameters, Map::of);
        for (Map.Entry entry : result.entrySet()) {
            Object object = entry.getValue();
            if (object instanceof Reader) {
                Reader reader = (Reader)object;
                try {
                    object = reader;
                    try {
                        int len;
                        StringBuilder buf = new StringBuilder();
                        char[] buffer = new char[4096];
                        while ((len = reader.read(buffer)) != -1) {
                            buf.append(buffer, 0, len);
                        }
                        entry.setValue(Values.value(buf.toString()));
                        continue;
                    }
                    finally {
                        if (object == null) continue;
                        ((Reader)object).close();
                        continue;
                    }
                }
                catch (IOException ex) {
                    throw new SQLException(ex);
                }
            }
            Object ex = entry.getValue();
            if (!(ex instanceof InputStream)) continue;
            InputStream inputStream = (InputStream)ex;
            try (BufferedInputStream in = new BufferedInputStream(inputStream);
                 ByteArrayOutputStream out = new ByteArrayOutputStream();){
                in.transferTo(out);
                entry.setValue(Values.value(out.toByteArray()));
            }
            catch (IOException ex2) {
                throw new SQLException(ex2);
            }
        }
        return result;
    }

    @Override
    public ResultSet getResultSet() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting result set");
        this.assertIsOpen();
        if (!this.resultSetAcquired.compareAndSet(false, true)) {
            throw new SQLException("Result set has already been acquired");
        }
        return this.multipleResultsApi && this.updateCount == -1 ? this.resultSet : null;
    }

    @Override
    public int getUpdateCount() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting update count");
        this.assertIsOpen();
        return this.multipleResultsApi ? this.updateCount : -1;
    }

    @Override
    public boolean getMoreResults() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting more results state");
        this.assertIsOpen();
        if (this.multipleResultsApi) {
            this.closeResultSet();
            this.updateCount = -1;
        }
        return false;
    }

    @Override
    public void setFetchDirection(int direction) throws SQLException {
        LOGGER.log(Level.WARNING, () -> "Setting fetch direction to %d (ignored)".formatted(direction));
        this.assertIsOpen();
    }

    @Override
    public int getFetchDirection() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting fetch direction");
        this.assertIsOpen();
        return 1000;
    }

    @Override
    public void setFetchSize(int rows) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting fetch size to %d".formatted(rows));
        this.assertIsOpen();
        if (rows < 0) {
            throw new SQLException("Fetch size can not be negative");
        }
        this.fetchSize = rows > 0 ? rows : 1000;
    }

    @Override
    public int getFetchSize() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting fetch size");
        this.assertIsOpen();
        return this.fetchSize;
    }

    @Override
    public int getResultSetConcurrency() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting result set concurrency");
        this.assertIsOpen();
        return 1007;
    }

    @Override
    public int getResultSetType() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting result set type");
        this.assertIsOpen();
        return 1003;
    }

    @Override
    public void addBatch(String sql) throws SQLException {
        throw new SQLException("Not supported");
    }

    @Override
    public void clearBatch() throws SQLException {
        throw new SQLException("Not supported");
    }

    @Override
    public int[] executeBatch() throws SQLException {
        throw new SQLException("Not supported");
    }

    @Override
    public Connection getConnection() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting connection");
        this.assertIsOpen();
        return this.connection;
    }

    @Override
    public boolean getMoreResults(int current) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public ResultSet getGeneratedKeys() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Trying to prepare update with auto generated keys set to %d".formatted(autoGeneratedKeys));
        if (autoGeneratedKeys != 2) {
            throw new SQLFeatureNotSupportedException();
        }
        return this.executeUpdate(sql);
    }

    @Override
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Trying to prepare execution with auto generated keys set to %d".formatted(autoGeneratedKeys));
        if (autoGeneratedKeys != 2) {
            throw new SQLFeatureNotSupportedException();
        }
        return this.execute(sql);
    }

    @Override
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public boolean execute(String sql, String[] columnNames) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public int getResultSetHoldability() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting result set holdability");
        this.assertIsOpen();
        return 2;
    }

    @Override
    public boolean isClosed() {
        LOGGER.log(Level.FINER, () -> "Getting closed state");
        return this.closed;
    }

    @Override
    public void setPoolable(boolean poolable) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting poolable to %s".formatted(poolable));
        this.assertIsOpen();
        this.poolable = poolable;
    }

    @Override
    public boolean isPoolable() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting poolable state");
        this.assertIsOpen();
        return this.poolable;
    }

    @Override
    public void closeOnCompletion() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Setting close on completion to %s".formatted(true));
        this.assertIsOpen();
        this.closeOnCompletion = true;
    }

    @Override
    public boolean isCloseOnCompletion() throws SQLException {
        LOGGER.log(Level.FINER, () -> "Getting close on completion state");
        this.assertIsOpen();
        return this.closeOnCompletion;
    }

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Unwrapping `%s` into `%s`".formatted(this.getClass().getCanonicalName(), iface.getCanonicalName()));
        if (iface.isAssignableFrom(this.getClass())) {
            return iface.cast(this);
        }
        throw new SQLException("This object does not implement the given interface");
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) {
        return iface.isAssignableFrom(this.getClass());
    }

    @Override
    public String enquoteIdentifier(String identifier, boolean alwaysQuote) throws SQLException {
        LOGGER.log(Level.FINER, () -> "Enquoting identifier `%s` with always quoting set to %s".formatted(identifier, alwaysQuote));
        return SchemaNames.sanitize(identifier, alwaysQuote).orElseThrow(() -> new SQLException("Cannot quote identifier " + identifier));
    }

    protected void assertIsOpen() throws SQLException {
        if (this.closed) {
            throw new SQLException("The statement set is closed");
        }
    }

    private void closeResultSet() throws SQLException {
        if (this.resultSet != null) {
            this.resultSet.close();
            this.resultSet = null;
            this.resultSetAcquired.set(false);
        }
    }

    protected final String processSQL(String sql) throws SQLException {
        try {
            UnaryOperator<Object> processor = StatementImpl.forceCypher(sql) ? UnaryOperator.identity() : this.sqlProcessor;
            String processedSQL = (String)processor.apply(sql);
            if (LOGGER.isLoggable(Level.FINE) && !processedSQL.equals(sql)) {
                LOGGER.log(Level.FINE, "Processed ''{0}'' into ''{1}''", new Object[]{sql, processedSQL});
            }
            return processedSQL;
        }
        catch (IllegalArgumentException | IllegalStateException ex) {
            throw new SQLException(Optional.ofNullable(ex.getCause()).orElse(ex));
        }
    }

    static boolean forceCypher(String sql) {
        Matcher matcher = PATTERN_ENFORCE_CYPHER.matcher(sql);
        while (matcher.find()) {
            if (matcher.group(1) != null && matcher.group(1).equals(matcher.group(2))) continue;
            return true;
        }
        return false;
    }

    @Override
    public Neo4jStatement withMetadata(Map<String, Object> metadata) {
        LOGGER.log(Level.FINER, () -> "Adding new transaction metadata");
        if (metadata != null) {
            this.transactionMetadata.putAll(metadata);
        }
        return this;
    }
}

