/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.storageengine.api;

import static org.neo4j.token.api.TokenConstants.ANY_RELATIONSHIP_TYPE;

import java.util.Arrays;
import org.apache.commons.lang3.ArrayUtils;
import org.eclipse.collections.api.iterator.LongIterator;
import org.eclipse.collections.impl.iterator.ImmutableEmptyLongIterator;
import org.neo4j.collection.PrimitiveLongCollections;
import org.neo4j.graphdb.Direction;
import org.neo4j.storageengine.api.txstate.NodeState;

/**
 * Used to specify a selection of relationships to get from a node.
 */
public abstract class RelationshipSelection {
    /**
     * Tests whether a relationship of a certain type should be part of this selection.
     *
     * @param type the relationship type id of the relationship to test.
     * @return whether or not this relationship type is part of this selection.
     */
    public abstract boolean test(int type);

    /**
     * Tests whether a relationship of a certain direction should be part of this selection.
     * @param direction {@link RelationshipDirection} of the relationship to test.
     * @return whether or not this relationship is part of this selection.
     */
    public abstract boolean test(RelationshipDirection direction);

    /**
     * @return the {@link Direction} of relationships in this selection.
     */
    public abstract Direction direction();

    /**
     * Tests whether a relationship of a certain type and direction should be part of this selection.
     *
     * @param type      the relationship type id of the relationship to test.
     * @param direction {@link RelationshipDirection} of the relationship to test.
     * @return whether or not this relationship type is part of this selection.
     */
    public abstract boolean test(int type, RelationshipDirection direction);

    /**
     * @return the number of criteria in this selection. One criterion is a type and {@link Direction}.
     */
    public abstract int numberOfCriteria();

    /**
     * @param index which criterion to access.
     * @return the type for the given index, must be between 0 and {@link #numberOfCriteria()}.
     */
    public abstract int criterionType(int index);

    /**
     * @param index which criterion to access.
     * @return the {@link Direction} for the given index, must be between 0 and {@link #numberOfCriteria()}.
     */
    public abstract Direction criterionDirection(int index);

    /**
     * @return {@code true} if this selection is limited on type, i.e. if number of criteria matches number of selected types,
     * otherwise {@code false}.
     */
    public abstract boolean isTypeLimited();

    /**
     * @return {@code true} if this selection is limited in any way, otherwise {@code false} where all relationships should be selected.
     */
    public boolean isLimited() {
        return true;
    }

    /**
     * Selects the correct set of added relationships from transaction state, based on the selection criteria.
     *
     * @param transactionState the {@link NodeState} to select added relationships from.
     * @return a {@link LongIterator} of added relationships matching the selection criteria from transaction state.
     */
    public abstract LongIterator addedRelationship(NodeState transactionState);

    public abstract RelationshipSelection reverse();

    public static RelationshipSelection selection(int[] types, Direction direction) {
        if (types == null) {
            return selection(direction);
        } else if (types.length == 0) {
            return NO_RELATIONSHIPS;
        } else if (types.length == 1) {
            return new DirectionalSingleType(types[0], direction);
        }
        return new DirectionalMultipleTypes(types, direction);
    }

    public static RelationshipSelection selection(int type, Direction direction) {
        return new DirectionalSingleType(type, direction);
    }

    public static RelationshipSelection selection(Direction direction) {
        return direction == Direction.BOTH ? ALL_RELATIONSHIPS : new DirectionalAllTypes(direction);
    }

    private abstract static class Directional extends RelationshipSelection {
        protected final Direction direction;

        Directional(Direction direction) {
            this.direction = direction;
        }

        @Override
        public boolean test(RelationshipDirection direction) {
            return direction.matches(this.direction);
        }

        @Override
        public Direction direction() {
            return direction;
        }
    }

    private abstract static class DirectionalSingleCriterion extends Directional {
        protected final int type;

        DirectionalSingleCriterion(int type, Direction direction) {
            super(direction);
            this.type = type;
        }

        @Override
        public boolean test(RelationshipDirection direction) {
            return direction.matches(this.direction);
        }

        @Override
        public int numberOfCriteria() {
            return 1;
        }

        @Override
        public Direction criterionDirection(int index) {
            assert index == 0;
            return direction;
        }

        @Override
        public int criterionType(int index) {
            assert index == 0;
            return type;
        }

        @Override
        public Direction direction() {
            return direction;
        }
    }

    private static class DirectionalSingleType extends DirectionalSingleCriterion {
        DirectionalSingleType(int type, Direction direction) {
            super(type, direction);
        }

        @Override
        public boolean test(int type) {
            return this.type == type;
        }

        @Override
        public boolean test(int type, RelationshipDirection direction) {
            return this.type == type && direction.matches(this.direction);
        }

        @Override
        public boolean isTypeLimited() {
            return true;
        }

        @Override
        public LongIterator addedRelationship(NodeState transactionState) {
            return transactionState.getAddedRelationships(direction, type);
        }

        @Override
        public RelationshipSelection reverse() {
            return Direction.BOTH.equals(direction) ? this : new DirectionalSingleType(type, direction.reverse());
        }

        @Override
        public String toString() {
            return "RelationshipSelection[" + "type=" + type + ", " + direction + "]";
        }
    }

    private static class DirectionalMultipleTypes extends Directional {
        private final int[] types;

        DirectionalMultipleTypes(int[] types, Direction direction) {
            super(direction);
            this.types = types;
        }

        @Override
        public boolean test(int type) {
            return ArrayUtils.contains(types, type);
        }

        @Override
        public boolean test(int type, RelationshipDirection direction) {
            return test(type) && direction.matches(this.direction);
        }

        @Override
        public int numberOfCriteria() {
            return types.length;
        }

        @Override
        public boolean isTypeLimited() {
            return true;
        }

        @Override
        public Direction criterionDirection(int index) {
            assert index < types.length;
            return direction;
        }

        @Override
        public int criterionType(int index) {
            assert index < types.length;
            return types[index];
        }

        @Override
        public LongIterator addedRelationship(NodeState transactionState) {
            LongIterator[] all = new LongIterator[types.length];
            int index = 0;
            for (int i = 0; i < types.length; i++) {
                // We have to avoid duplication here, so check backwards if this type exists earlier in the array
                if (!existsEarlier(types, i)) {
                    all[index++] = transactionState.getAddedRelationships(direction, types[i]);
                }
            }
            if (index != types.length) {
                all = Arrays.copyOf(all, index);
            }
            return PrimitiveLongCollections.concat(all);
        }

        @Override
        public RelationshipSelection reverse() {
            return Direction.BOTH.equals(direction) ? this : new DirectionalMultipleTypes(types, direction.reverse());
        }

        @Override
        public String toString() {
            return "RelationshipSelection[" + "types=" + Arrays.toString(types) + ", " + direction + "]";
        }

        private static boolean existsEarlier(int[] types, int i) {
            int candidateType = types[i];
            for (int j = i - 1; j >= 0; j--) {
                if (candidateType == types[j]) {
                    return true;
                }
            }
            return false;
        }
    }

    private static class DirectionalAllTypes extends DirectionalSingleCriterion {
        DirectionalAllTypes(Direction direction) {
            super(ANY_RELATIONSHIP_TYPE, direction);
        }

        @Override
        public boolean test(int type) {
            return true;
        }

        @Override
        public boolean test(int type, RelationshipDirection direction) {
            return direction.matches(this.direction);
        }

        @Override
        public boolean isTypeLimited() {
            return false;
        }

        @Override
        public LongIterator addedRelationship(NodeState transactionState) {
            return transactionState.getAddedRelationships(direction);
        }

        @Override
        public RelationshipSelection reverse() {
            return Direction.BOTH.equals(direction) ? this : new DirectionalAllTypes(direction.reverse());
        }

        @Override
        public String toString() {
            return "RelationshipSelection[" + direction + "]";
        }
    }

    public static final RelationshipSelection ALL_RELATIONSHIPS = new RelationshipSelection() {
        @Override
        public boolean test(int type) {
            return true;
        }

        @Override
        public boolean test(RelationshipDirection direction) {
            return true;
        }

        @Override
        public Direction direction() {
            return Direction.BOTH;
        }

        @Override
        public boolean test(int type, RelationshipDirection direction) {
            return true;
        }

        @Override
        public RelationshipSelection reverse() {
            return this;
        }

        @Override
        public int numberOfCriteria() {
            return 1;
        }

        @Override
        public Direction criterionDirection(int index) {
            assert index == 0;
            return Direction.BOTH;
        }

        @Override
        public int criterionType(int index) {
            assert index == 0;
            return ANY_RELATIONSHIP_TYPE;
        }

        @Override
        public boolean isTypeLimited() {
            return false;
        }

        @Override
        public boolean isLimited() {
            return false;
        }

        @Override
        public LongIterator addedRelationship(NodeState transactionState) {
            return transactionState.getAddedRelationships();
        }

        @Override
        public String toString() {
            return "RelationshipSelection[*]";
        }
    };

    public static final RelationshipSelection NO_RELATIONSHIPS = new RelationshipSelection() {
        @Override
        public boolean test(int type) {
            return false;
        }

        @Override
        public boolean test(RelationshipDirection direction) {
            return false;
        }

        @Override
        public Direction direction() {
            return Direction.BOTH;
        }

        @Override
        public boolean test(int type, RelationshipDirection direction) {
            return false;
        }

        @Override
        public int numberOfCriteria() {
            return 0;
        }

        @Override
        public Direction criterionDirection(int index) {
            throw new IllegalArgumentException("Unknown criterion index " + index);
        }

        @Override
        public int criterionType(int index) {
            throw new IllegalArgumentException("Unknown criterion index " + index);
        }

        @Override
        public boolean isTypeLimited() {
            return true;
        }

        @Override
        public LongIterator addedRelationship(NodeState transactionState) {
            return ImmutableEmptyLongIterator.INSTANCE;
        }

        @Override
        public RelationshipSelection reverse() {
            return this;
        }

        @Override
        public String toString() {
            return "RelationshipSelection[NOTHING]";
        }
    };
}
