/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.api.index;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.neo4j.graphdb.DependencyResolver;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.factory.GraphDatabaseBuilder;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.kernel.GraphDatabaseAPI;
import org.neo4j.kernel.NeoStoreDataSource;
import org.neo4j.kernel.api.Statement;
import org.neo4j.kernel.api.exceptions.EntityNotFoundException;
import org.neo4j.kernel.api.exceptions.KernelException;
import org.neo4j.kernel.api.exceptions.index.IndexNotFoundKernelException;
import org.neo4j.kernel.api.index.IndexDescriptor;
import org.neo4j.kernel.api.properties.Property;
import org.neo4j.kernel.impl.api.index.IndexingService;
import org.neo4j.kernel.impl.api.index.UpdatesTracker;
import org.neo4j.kernel.impl.core.ThreadToStatementContextBridge;
import org.neo4j.kernel.impl.store.NeoStores;
import org.neo4j.kernel.impl.store.counts.CountsTracker;
import org.neo4j.kernel.monitoring.Monitors;
import org.neo4j.register.Register;
import org.neo4j.register.Registers;
import org.neo4j.test.DatabaseRule;
import org.neo4j.test.EmbeddedDatabaseRule;

public class IndexStatisticsTest {
    private static final double UNIQUE_NAMES = 10.0;
    private static final String[] NAMES = new String[]{"Andres", "Davide", "Jakub", "Chris", "Tobias", "Stefan", "Petra", "Rickard", "Mattias", "Emil", "Chris", "Chris"};
    private static final int CREATION_MULTIPLIER = 10000;
    private static final int MISSED_UPDATES_TOLERANCE = NAMES.length;
    private static final double DOUBLE_ERROR_TOLERANCE = 1.0E-5;
    @Rule
    public DatabaseRule dbRule = new EmbeddedDatabaseRule(){

        @Override
        protected void configure(GraphDatabaseBuilder builder) {
            super.configure(builder);
            builder.setConfig(GraphDatabaseSettings.index_background_sampling_enabled, "false");
        }
    };
    private GraphDatabaseService db;
    private ThreadToStatementContextBridge bridge;
    private final IndexOnlineMonitor indexOnlineMonitor = new IndexOnlineMonitor();

    @Test
    public void shouldProvideIndexStatisticsForDataCreatedWhenPopulationBeforeTheIndexIsOnline() throws KernelException {
        this.createSomePersons();
        IndexDescriptor index = this.awaitOnline(this.createIndex("Person", "name"));
        Assert.assertEquals((double)0.75, (double)this.indexSelectivity(index), (double)1.0E-5);
        Assert.assertEquals((long)4L, (long)this.indexSize(index));
        Assert.assertEquals((long)0L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldNotSeeDataCreatedAfterPopulation() throws KernelException {
        IndexDescriptor index = this.awaitOnline(this.createIndex("Person", "name"));
        this.createSomePersons();
        Assert.assertEquals((double)1.0, (double)this.indexSelectivity(index), (double)1.0E-5);
        Assert.assertEquals((long)0L, (long)this.indexSize(index));
        Assert.assertEquals((long)4L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsForDataSeenDuringPopulationAndIgnoreDataCreatedAfterPopulation() throws KernelException {
        this.createSomePersons();
        IndexDescriptor index = this.awaitOnline(this.createIndex("Person", "name"));
        this.createSomePersons();
        Assert.assertEquals((double)0.75, (double)this.indexSelectivity(index), (double)1.0E-5);
        Assert.assertEquals((long)4L, (long)this.indexSize(index));
        Assert.assertEquals((long)4L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldRemoveIndexStatisticsAfterIndexIsDeleted() throws KernelException {
        this.createSomePersons();
        IndexDescriptor index = this.awaitOnline(this.createIndex("Person", "name"));
        this.dropIndex(index);
        try {
            this.indexSelectivity(index);
            Assert.fail((String)"Expected IndexNotFoundKernelException to be thrown");
        }
        catch (IndexNotFoundKernelException e) {
            Register.DoubleLongRegister actual = this.getTracker().indexSample(index.getLabelId(), index.getPropertyKeyId(), Registers.newDoubleLongRegister());
            this.assertDoubleLongEquals(0L, 0L, actual);
        }
        Register.DoubleLongRegister actual = this.getTracker().indexUpdatesAndSize(index.getLabelId(), index.getPropertyKeyId(), Registers.newDoubleLongRegister());
        this.assertDoubleLongEquals(0L, 0L, actual);
    }

    @Test
    public void shouldProvideIndexSelectivityWhenThereAreManyDuplicates() throws Exception {
        int created = this.repeatCreateNamedPeopleFor(NAMES.length * 10000).length;
        IndexDescriptor index = this.awaitOnline(this.createIndex("Person", "name"));
        double expectedSelectivity = 10.0 / (double)created;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(created, this.indexSize(index));
        Assert.assertEquals((long)0L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditions() throws Exception {
        int initialNodes = this.repeatCreateNamedPeopleFor(NAMES.length * 10000).length;
        IndexDescriptor index = this.createIndex("Person", "name");
        UpdatesTracker updatesTracker = this.executeCreations(index, 10000);
        this.awaitOnline(index);
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        IndexStatisticsTest.assertCorrectIndexUpdates(updatesTracker.createdAfterPopulation(), this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndDeletions() throws Exception {
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * 10000);
        int initialNodes = nodes.length;
        IndexDescriptor index = this.createIndex("Person", "name");
        UpdatesTracker updatesTracker = this.executeCreationsAndDeletions(nodes, index, 10000);
        this.awaitOnline(index);
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation() - updatesTracker.deletedDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = updatesTracker.deletedAfterPopulation() + updatesTracker.createdAfterPopulation();
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndChanges() throws Exception {
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * 10000);
        int initialNodes = nodes.length;
        IndexDescriptor index = this.createIndex("Person", "name");
        UpdatesTracker updatesTracker = this.executeCreationsAndUpdates(nodes, index, 10000);
        this.awaitOnline(index);
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        IndexStatisticsTest.assertCorrectIndexUpdates(updatesTracker.createdAfterPopulation(), this.indexUpdates(index));
    }

    @Test
    public void shouldWorkWhileHavingHeavyConcurrentUpdates() throws Exception {
        final long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * 10000);
        int initialNodes = nodes.length;
        int threads = 5;
        ExecutorService executorService = Executors.newFixedThreadPool(threads);
        final IndexDescriptor index = this.createIndex("Person", "name");
        ArrayList<1> jobs = new ArrayList<1>(threads);
        for (int i = 0; i < threads; ++i) {
            jobs.add(new Callable<UpdatesTracker>(){

                @Override
                public UpdatesTracker call() throws Exception {
                    return IndexStatisticsTest.this.executeCreationsDeletionsAndUpdates(nodes, index, 10000);
                }
            });
        }
        List futures = executorService.invokeAll(jobs);
        UpdatesTracker result = new UpdatesTracker();
        result.notifyPopulationCompleted();
        for (Future future : futures) {
            result.add((UpdatesTracker)future.get());
        }
        this.awaitOnline(index);
        executorService.awaitTermination(1L, TimeUnit.SECONDS);
        executorService.shutdown();
        int tolerance = MISSED_UPDATES_TOLERANCE * threads;
        double doubleTolerance = 1.0E-5 * (double)threads;
        int seenWhilePopulating = initialNodes + result.createdDuringPopulation() - result.deletedDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index), doubleTolerance);
        IndexStatisticsTest.assertCorrectIndexSize("Tracker had " + result, seenWhilePopulating, this.indexSize(index), tolerance);
        int expectedIndexUpdates = result.deletedAfterPopulation() + result.createdAfterPopulation();
        IndexStatisticsTest.assertCorrectIndexUpdates("Tracker had " + result, expectedIndexUpdates, this.indexUpdates(index), tolerance);
    }

    private void deleteNode(long nodeId) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            Statement statement = this.bridge.get();
            statement.dataWriteOperations().nodeDelete(nodeId);
            tx.success();
        }
    }

    private void changeName(long nodeId, String propertyKeyName, Object newValue) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            Statement statement = this.bridge.get();
            int propertyKeyId = statement.tokenWriteOperations().propertyKeyGetOrCreateForName(propertyKeyName);
            statement.dataWriteOperations().nodeSetProperty(nodeId, Property.property((int)propertyKeyId, (Object)newValue));
            tx.success();
        }
    }

    private int createNamedPeople(long[] nodes, int offset) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            Statement statement = this.bridge.get();
            for (String name : NAMES) {
                long nodeId = this.createNode(statement, "Person", "name", name);
                if (nodes == null) continue;
                nodes[offset++] = nodeId;
            }
            tx.success();
        }
        return NAMES.length;
    }

    private long[] repeatCreateNamedPeopleFor(int totalNumberOfPeople) throws Exception {
        final long[] nodes = new long[totalNumberOfPeople];
        int threads = 100;
        final int peoplePerThread = totalNumberOfPeople / 100;
        ExecutorService service = Executors.newFixedThreadPool(100);
        final AtomicReference exception = new AtomicReference();
        ArrayList<2> jobs = new ArrayList<2>(100);
        int i = 0;
        while (i < 100) {
            final int finalI = i++;
            jobs.add(new Callable<Void>(){

                @Override
                public Void call() throws Exception {
                    for (int offset = finalI * peoplePerThread; offset < (finalI + 1) * peoplePerThread; offset += IndexStatisticsTest.this.createNamedPeople(nodes, offset)) {
                        try {
                            continue;
                        }
                        catch (KernelException e) {
                            exception.compareAndSet(null, e);
                            throw new RuntimeException(e);
                        }
                    }
                    return null;
                }
            });
        }
        for (Future job : service.invokeAll(jobs)) {
            job.get();
        }
        service.awaitTermination(1L, TimeUnit.SECONDS);
        service.shutdown();
        Exception ex = (Exception)exception.get();
        if (ex != null) {
            throw ex;
        }
        return nodes;
    }

    private void dropIndex(IndexDescriptor index) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            Statement statement = this.bridge.get();
            statement.schemaWriteOperations().indexDrop(index);
            tx.success();
        }
    }

    private long indexSize(IndexDescriptor descriptor) throws KernelException {
        return ((NeoStoreDataSource)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(NeoStoreDataSource.class)).getIndexService().indexUpdatesAndSize(descriptor).readSecond();
    }

    private long indexUpdates(IndexDescriptor descriptor) throws KernelException {
        return ((NeoStoreDataSource)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(NeoStoreDataSource.class)).getIndexService().indexUpdatesAndSize(descriptor).readFirst();
    }

    private double indexSelectivity(IndexDescriptor descriptor) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            Statement statement = this.bridge.get();
            double selectivity = statement.readOperations().indexUniqueValuesSelectivity(descriptor);
            tx.success();
            double d = selectivity;
            return d;
        }
    }

    private CountsTracker getTracker() {
        return ((NeoStores)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(NeoStores.class)).getCounts();
    }

    private void createSomePersons() throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            Statement statement = this.bridge.get();
            this.createNode(statement, "Person", "name", "Davide");
            this.createNode(statement, "Person", "name", "Stefan");
            this.createNode(statement, "Person", "name", "John");
            this.createNode(statement, "Person", "name", "John");
            tx.success();
        }
    }

    private long createNode(Statement statement, String labelName, String propertyKeyName, Object value) throws KernelException {
        int labelId = statement.tokenWriteOperations().labelGetOrCreateForName(labelName);
        int propertyKeyId = statement.tokenWriteOperations().propertyKeyGetOrCreateForName(propertyKeyName);
        long nodeId = statement.dataWriteOperations().nodeCreate();
        statement.dataWriteOperations().nodeAddLabel(nodeId, labelId);
        statement.dataWriteOperations().nodeSetProperty(nodeId, Property.property((int)propertyKeyId, (Object)value));
        return nodeId;
    }

    private IndexDescriptor createIndex(String labelName, String propertyKeyName) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            Statement statement = this.bridge.get();
            int labelId = statement.tokenWriteOperations().labelGetOrCreateForName(labelName);
            int propertyKeyId = statement.tokenWriteOperations().propertyKeyGetOrCreateForName(propertyKeyName);
            IndexDescriptor index = statement.schemaWriteOperations().indexCreate(labelId, propertyKeyId);
            tx.success();
            IndexDescriptor indexDescriptor = index;
            return indexDescriptor;
        }
    }

    private IndexDescriptor awaitOnline(IndexDescriptor index) throws KernelException {
        long start = System.currentTimeMillis();
        long end = start + 20000L;
        while (System.currentTimeMillis() < end) {
            Transaction tx = this.db.beginTx();
            Throwable throwable = null;
            try {
                Statement statement = this.bridge.get();
                switch (statement.readOperations().indexGetState(index)) {
                    case ONLINE: {
                        IndexDescriptor indexDescriptor = index;
                        return indexDescriptor;
                    }
                    case FAILED: {
                        throw new IllegalStateException("Index failed instead of becoming ONLINE");
                    }
                }
                tx.success();
                try {
                    Thread.sleep(100L);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (tx == null) continue;
                if (throwable != null) {
                    try {
                        tx.close();
                    }
                    catch (Throwable x2) {
                        throwable.addSuppressed(x2);
                    }
                    continue;
                }
                tx.close();
            }
        }
        throw new IllegalStateException("Index did not become ONLINE within reasonable time");
    }

    private UpdatesTracker executeCreations(IndexDescriptor index, int numberOfCreations) throws KernelException {
        return this.internalExecuteCreationsDeletionsAndUpdates(null, index, numberOfCreations, false, false);
    }

    private UpdatesTracker executeCreationsAndDeletions(long[] nodes, IndexDescriptor index, int numberOfCreations) throws KernelException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, index, numberOfCreations, true, false);
    }

    private UpdatesTracker executeCreationsAndUpdates(long[] nodes, IndexDescriptor index, int numberOfCreations) throws KernelException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, index, numberOfCreations, false, true);
    }

    private UpdatesTracker executeCreationsDeletionsAndUpdates(long[] nodes, IndexDescriptor index, int numberOfCreations) throws KernelException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, index, numberOfCreations, true, true);
    }

    private UpdatesTracker internalExecuteCreationsDeletionsAndUpdates(long[] nodes, IndexDescriptor index, int numberOfCreations, boolean allowDeletions, boolean allowUpdates) throws KernelException {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        UpdatesTracker updatesTracker = new UpdatesTracker();
        int offset = 0;
        while (updatesTracker.created() < numberOfCreations) {
            int created = this.createNamedPeople(nodes, offset);
            offset += created;
            updatesTracker.increaseCreated(created);
            if (!updatesTracker.isPopulationCompleted() && this.indexOnlineMonitor.isIndexOnline(index)) {
                updatesTracker.notifyPopulationCompleted();
            }
            if (allowDeletions && updatesTracker.created() % 5 == 0) {
                long nodeId = nodes[((Random)random).nextInt(nodes.length)];
                try {
                    this.deleteNode(nodeId);
                    updatesTracker.increaseDeleted(1);
                }
                catch (EntityNotFoundException entityNotFoundException) {
                    // empty catch block
                }
                if (!updatesTracker.isPopulationCompleted() && this.indexOnlineMonitor.isIndexOnline(index)) {
                    updatesTracker.notifyPopulationCompleted();
                }
            }
            if (!allowUpdates || updatesTracker.created() % 5 != 0) continue;
            int randomIndex = ((Random)random).nextInt(nodes.length);
            try {
                this.changeName(nodes[randomIndex], "name", NAMES[randomIndex % NAMES.length]);
            }
            catch (EntityNotFoundException entityNotFoundException) {
                // empty catch block
            }
            if (updatesTracker.isPopulationCompleted() || !this.indexOnlineMonitor.isIndexOnline(index)) continue;
            updatesTracker.notifyPopulationCompleted();
        }
        updatesTracker.notifyPopulationCompleted();
        return updatesTracker;
    }

    private void assertDoubleLongEquals(long expectedUniqueValue, long expectedSampledSize, Register.DoubleLongRegister register) {
        Assert.assertEquals((long)expectedUniqueValue, (long)register.readFirst());
        Assert.assertEquals((long)expectedSampledSize, (long)register.readSecond());
    }

    private static void assertCorrectIndexSize(long expected, long actual) {
        IndexStatisticsTest.assertCorrectIndexSize("", expected, actual, MISSED_UPDATES_TOLERANCE);
    }

    private static void assertCorrectIndexSize(String info, long expected, long actual, int tolerance) {
        String message = String.format("Expected number of entries to not differ by more than %d (expected: %d actual: %d) %s", tolerance, expected, actual, info);
        Assert.assertTrue((String)message, (Math.abs(expected - actual) <= (long)tolerance ? 1 : 0) != 0);
    }

    private static void assertCorrectIndexUpdates(long expected, long actual) {
        IndexStatisticsTest.assertCorrectIndexUpdates("", expected, actual, MISSED_UPDATES_TOLERANCE);
    }

    private static void assertCorrectIndexUpdates(String info, long expected, long actual, int tolerance) {
        String message = String.format("Expected number of index updates to not differ by more than %d (expected: %d actual: %d). %s", tolerance, expected, actual, info);
        Assert.assertTrue((String)message, (Math.abs(expected - actual) <= (long)tolerance ? 1 : 0) != 0);
    }

    private static void assertCorrectIndexSelectivity(double expected, double actual) {
        IndexStatisticsTest.assertCorrectIndexSelectivity(expected, actual, 1.0E-5);
    }

    private static void assertCorrectIndexSelectivity(double expected, double actual, double tolerance) {
        String message = String.format("Expected number of entries to not differ by more than %f (expected: %f actual: %f)", tolerance, expected, actual);
        Assert.assertEquals((String)message, (double)expected, (double)actual, (double)tolerance);
    }

    @Before
    public void before() {
        GraphDatabaseAPI graphDatabaseAPI = this.dbRule.getGraphDatabaseAPI();
        this.db = graphDatabaseAPI;
        DependencyResolver dependencyResolver = graphDatabaseAPI.getDependencyResolver();
        this.bridge = (ThreadToStatementContextBridge)dependencyResolver.resolveDependency(ThreadToStatementContextBridge.class);
        ((Monitors)graphDatabaseAPI.getDependencyResolver().resolveDependency(Monitors.class)).addMonitorListener((Object)this.indexOnlineMonitor, new String[0]);
    }

    private static class IndexOnlineMonitor
    extends IndexingService.MonitorAdapter {
        private final Set<IndexDescriptor> onlineIndexes = new HashSet<IndexDescriptor>();

        private IndexOnlineMonitor() {
        }

        public void populationCompleteOn(IndexDescriptor descriptor) {
            this.onlineIndexes.add(descriptor);
        }

        public boolean isIndexOnline(IndexDescriptor descriptor) {
            return this.onlineIndexes.contains(descriptor);
        }
    }
}

