/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.storemigration;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import org.neo4j.helpers.Exceptions;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.api.index.SchemaIndexProvider;
import org.neo4j.kernel.impl.storemigration.MigrationStatus;
import org.neo4j.kernel.impl.storemigration.StoreMigrationParticipant;
import org.neo4j.kernel.impl.storemigration.UpgradableDatabase;
import org.neo4j.kernel.impl.storemigration.UpgradeConfiguration;
import org.neo4j.kernel.impl.storemigration.UpgradeNotAllowedException;
import org.neo4j.logging.Log;
import org.neo4j.logging.LogProvider;

public class StoreUpgrader {
    public static final String MIGRATION_DIRECTORY = "upgrade";
    public static final String MIGRATION_LEFT_OVERS_DIRECTORY = "upgrade_backup";
    private static final String MIGRATION_STATUS_FILE = "_status";
    public static final Monitor NO_MONITOR = new MonitorAdapter(){};
    private final List<StoreMigrationParticipant> participants = new ArrayList<StoreMigrationParticipant>();
    private final UpgradeConfiguration upgradeConfiguration;
    private final FileSystemAbstraction fileSystem;
    private final Monitor monitor;
    private final Log log;

    public StoreUpgrader(UpgradeConfiguration upgradeConfiguration, FileSystemAbstraction fileSystem, Monitor monitor, LogProvider logProvider) {
        this.fileSystem = fileSystem;
        this.upgradeConfiguration = upgradeConfiguration;
        this.monitor = monitor;
        this.log = logProvider.getLog(this.getClass());
    }

    public void addParticipant(StoreMigrationParticipant participant) {
        assert (participant != null);
        this.participants.add(participant);
    }

    public void migrateIfNeeded(File storeDirectory, UpgradableDatabase upgradableDatabase, SchemaIndexProvider schemaIndexProvider) {
        File migrationDirectory = new File(storeDirectory, MIGRATION_DIRECTORY);
        this.cleanupLegacyLeftOverDirsIn(storeDirectory);
        File migrationStateFile = new File(migrationDirectory, MIGRATION_STATUS_FILE);
        if (upgradableDatabase.hasCurrentVersion(storeDirectory) && !this.fileSystem.fileExists(migrationStateFile)) {
            return;
        }
        this.monitor.migrationNeeded();
        try {
            this.upgradeConfiguration.checkConfigurationAllowsAutomaticUpgrade();
        }
        catch (UpgradeNotAllowedException e) {
            this.monitor.migrationNotAllowed();
            throw e;
        }
        MigrationStatus migrationStatus = MigrationStatus.readMigrationStatus(this.fileSystem, migrationStateFile);
        String versionToMigrateFrom = null;
        if (MigrationStatus.migrating.isNeededFor(migrationStatus)) {
            versionToMigrateFrom = upgradableDatabase.checkUpgradeable(storeDirectory);
            this.cleanMigrationDirectory(migrationDirectory);
            MigrationStatus.migrating.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            this.migrateToIsolatedDirectory(storeDirectory, migrationDirectory, schemaIndexProvider, versionToMigrateFrom);
            MigrationStatus.moving.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
        }
        if (MigrationStatus.moving.isNeededFor(migrationStatus)) {
            versionToMigrateFrom = MigrationStatus.moving.maybeReadInfo(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            this.moveMigratedFilesToStoreDirectory(this.participants, migrationDirectory, storeDirectory, versionToMigrateFrom);
            MigrationStatus.countsRebuilding.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
        }
        if (MigrationStatus.countsRebuilding.isNeededFor(migrationStatus)) {
            versionToMigrateFrom = MigrationStatus.countsRebuilding.maybeReadInfo(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            this.rebuildCountsInStoreDirectory(this.participants, storeDirectory, versionToMigrateFrom);
            MigrationStatus.completed.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
        }
        this.cleanup(this.participants, migrationDirectory);
        this.monitor.migrationCompleted();
    }

    private void cleanupLegacyLeftOverDirsIn(File storeDir) {
        final Pattern leftOverDirsPattern = Pattern.compile("upgrade_backup(_\\d*)?");
        File[] leftOverDirs = storeDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File file, String name) {
                return file.isDirectory() && leftOverDirsPattern.matcher(name).matches();
            }
        });
        if (leftOverDirs != null) {
            for (File leftOverDir : leftOverDirs) {
                this.deleteSilently(leftOverDir);
            }
        }
    }

    private void cleanup(Iterable<StoreMigrationParticipant> participants, File migrationDirectory) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.cleanup(migrationDirectory);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Failure cleaning up after migration", e);
        }
    }

    private void moveMigratedFilesToStoreDirectory(Iterable<StoreMigrationParticipant> participants, File migrationDirectory, File storeDirectory, String versionToMigrateFrom) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.moveMigratedFiles(migrationDirectory, storeDirectory, versionToMigrateFrom);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Unable to move migrated files into place", e);
        }
    }

    private void rebuildCountsInStoreDirectory(List<StoreMigrationParticipant> participants, File storeDirectory, String versionToMigrateFrom) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.rebuildCounts(storeDirectory, versionToMigrateFrom);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Unable to move migrated files into place", e);
        }
    }

    private void migrateToIsolatedDirectory(File storeDir, File migrationDirectory, SchemaIndexProvider schemaIndexProvider, String versionToMigrateFrom) {
        try {
            for (StoreMigrationParticipant participant : this.participants) {
                participant.migrate(storeDir, migrationDirectory, schemaIndexProvider, versionToMigrateFrom);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Failure doing migration", e);
        }
        catch (Exception e) {
            throw Exceptions.launderedException(e);
        }
    }

    private void cleanMigrationDirectory(File migrationDirectory) {
        if (migrationDirectory.exists()) {
            try {
                this.fileSystem.deleteRecursively(migrationDirectory);
            }
            catch (IOException e) {
                throw new UnableToUpgradeException("Failure deleting upgrade directory " + migrationDirectory, e);
            }
        }
        this.fileSystem.mkdir(migrationDirectory);
    }

    private void deleteSilently(File dir) {
        try {
            this.fileSystem.deleteRecursively(dir);
        }
        catch (IOException e) {
            this.log.error("Unable to delete directory: " + dir, (Throwable)e);
        }
    }

    public static class UnexpectedUpgradingStoreVersionException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "'%s' has a store version number that we cannot upgrade from. Expected '%s' but file is version '%s'.";

        public UnexpectedUpgradingStoreVersionException(String filename, String expectedVersion, String actualVersion) {
            super(String.format(MESSAGE, filename, expectedVersion, actualVersion));
        }
    }

    public static class UpgradingStoreVersionNotFoundException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "'%s' does not contain a store version, please ensure that the original database was shut down in a clean state.";

        public UpgradingStoreVersionNotFoundException(String filenameWithoutStoreVersion) {
            super(String.format(MESSAGE, filenameWithoutStoreVersion));
        }
    }

    public static class UpgradeMissingStoreFilesException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "Missing required store file '%s'.";

        public UpgradeMissingStoreFilesException(String filenameExpectedToExist) {
            super(String.format(MESSAGE, filenameExpectedToExist));
        }
    }

    public static class UnableToUpgradeException
    extends RuntimeException {
        public UnableToUpgradeException(String message, Throwable cause) {
            super(message, cause);
        }

        public UnableToUpgradeException(String message) {
            super(message);
        }
    }

    public static abstract class MonitorAdapter
    implements Monitor {
        @Override
        public void migrationNeeded() {
        }

        @Override
        public void migrationNotAllowed() {
        }

        @Override
        public void migrationCompleted() {
        }
    }

    public static interface Monitor {
        public void migrationNeeded();

        public void migrationNotAllowed();

        public void migrationCompleted();
    }
}

