/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.storemigration;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.helpers.Exceptions;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.configuration.Config;
import org.neo4j.kernel.impl.storemigration.MigrationStatus;
import org.neo4j.kernel.impl.storemigration.StoreMigrationParticipant;
import org.neo4j.kernel.impl.storemigration.UpgradableDatabase;
import org.neo4j.kernel.impl.storemigration.UpgradeNotAllowedByConfigurationException;
import org.neo4j.kernel.impl.storemigration.monitoring.MigrationProgressMonitor;
import org.neo4j.kernel.impl.util.CustomIOConfigValidator;
import org.neo4j.logging.Log;
import org.neo4j.logging.LogProvider;

public class StoreUpgrader {
    public static final String MIGRATION_DIRECTORY = "upgrade";
    public static final String MIGRATION_LEFT_OVERS_DIRECTORY = "upgrade_backup";
    private static final String MIGRATION_STATUS_FILE = "_status";
    public static final String CUSTOM_IO_EXCEPTION_MESSAGE = "Store upgrade not allowed with custom IO integrations";
    private final UpgradableDatabase upgradableDatabase;
    private final MigrationProgressMonitor progressMonitor;
    private final List<StoreMigrationParticipant> participants = new ArrayList<StoreMigrationParticipant>();
    private final Config config;
    private final FileSystemAbstraction fileSystem;
    private final Log log;

    public StoreUpgrader(UpgradableDatabase upgradableDatabase, MigrationProgressMonitor progressMonitor, Config config, FileSystemAbstraction fileSystem, LogProvider logProvider) {
        this.upgradableDatabase = upgradableDatabase;
        this.progressMonitor = progressMonitor;
        this.fileSystem = fileSystem;
        this.config = config;
        this.log = logProvider.getLog(this.getClass());
    }

    public void addParticipant(StoreMigrationParticipant participant) {
        assert (participant != null);
        this.participants.add(participant);
    }

    public void migrateIfNeeded(File storeDirectory) {
        File migrationDirectory = new File(storeDirectory, MIGRATION_DIRECTORY);
        this.cleanupLegacyLeftOverDirsIn(storeDirectory);
        File migrationStateFile = new File(migrationDirectory, MIGRATION_STATUS_FILE);
        if (this.upgradableDatabase.hasCurrentVersion(storeDirectory) && !this.fileSystem.fileExists(migrationStateFile)) {
            return;
        }
        if (!this.isUpgradeAllowed()) {
            throw new UpgradeNotAllowedByConfigurationException();
        }
        CustomIOConfigValidator.assertCustomIOConfigNotUsed(this.config, CUSTOM_IO_EXCEPTION_MESSAGE);
        this.progressMonitor.started();
        MigrationStatus migrationStatus = MigrationStatus.readMigrationStatus(this.fileSystem, migrationStateFile);
        String versionToMigrateFrom = null;
        if (MigrationStatus.migrating.isNeededFor(migrationStatus)) {
            versionToMigrateFrom = this.upgradableDatabase.checkUpgradeable(storeDirectory).storeVersion();
            this.cleanMigrationDirectory(migrationDirectory);
            MigrationStatus.migrating.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            this.migrateToIsolatedDirectory(storeDirectory, migrationDirectory, versionToMigrateFrom);
            MigrationStatus.moving.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
        }
        if (MigrationStatus.moving.isNeededFor(migrationStatus)) {
            versionToMigrateFrom = MigrationStatus.moving.maybeReadInfo(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            this.moveMigratedFilesToStoreDirectory(this.participants, migrationDirectory, storeDirectory, versionToMigrateFrom, this.upgradableDatabase.currentVersion());
            MigrationStatus.countsRebuilding.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
        }
        if (MigrationStatus.countsRebuilding.isNeededFor(migrationStatus)) {
            versionToMigrateFrom = MigrationStatus.countsRebuilding.maybeReadInfo(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            this.rebuildCountsInStoreDirectory(this.participants, storeDirectory, versionToMigrateFrom);
            MigrationStatus.completed.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
        }
        this.cleanup(this.participants, migrationDirectory);
        this.progressMonitor.completed();
    }

    private boolean isUpgradeAllowed() {
        return this.config.get(GraphDatabaseSettings.allow_store_upgrade);
    }

    private void cleanupLegacyLeftOverDirsIn(File storeDir) {
        final Pattern leftOverDirsPattern = Pattern.compile("upgrade_backup(_\\d*)?");
        File[] leftOverDirs = storeDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File file, String name) {
                return file.isDirectory() && leftOverDirsPattern.matcher(name).matches();
            }
        });
        if (leftOverDirs != null) {
            for (File leftOverDir : leftOverDirs) {
                this.deleteSilently(leftOverDir);
            }
        }
    }

    private void cleanup(Iterable<StoreMigrationParticipant> participants, File migrationDirectory) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.cleanup(migrationDirectory);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Failure cleaning up after migration", e);
        }
    }

    private void moveMigratedFilesToStoreDirectory(Iterable<StoreMigrationParticipant> participants, File migrationDirectory, File storeDirectory, String versionToMigrateFrom, String versionToMigrateTo) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.moveMigratedFiles(migrationDirectory, storeDirectory, versionToMigrateFrom, versionToMigrateTo);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Unable to move migrated files into place", e);
        }
    }

    private void rebuildCountsInStoreDirectory(List<StoreMigrationParticipant> participants, File storeDirectory, String versionToMigrateFrom) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.rebuildCounts(storeDirectory, versionToMigrateFrom, this.upgradableDatabase.currentVersion());
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Unable to move migrated files into place", e);
        }
    }

    private void migrateToIsolatedDirectory(File storeDir, File migrationDirectory, String versionToMigrateFrom) {
        try {
            int index = 1;
            for (StoreMigrationParticipant participant : this.participants) {
                MigrationProgressMonitor.Section section = this.progressMonitor.startSection(String.format("%s (%d/%d)", participant.getName(), index, this.participants.size()));
                participant.migrate(storeDir, migrationDirectory, section, versionToMigrateFrom, this.upgradableDatabase.currentVersion());
                section.completed();
                ++index;
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Failure doing migration", e);
        }
        catch (Exception e) {
            throw Exceptions.launderedException(e);
        }
    }

    private void cleanMigrationDirectory(File migrationDirectory) {
        if (migrationDirectory.exists()) {
            try {
                this.fileSystem.deleteRecursively(migrationDirectory);
            }
            catch (IOException e) {
                throw new UnableToUpgradeException("Failure deleting upgrade directory " + migrationDirectory, e);
            }
        }
        this.fileSystem.mkdir(migrationDirectory);
    }

    private void deleteSilently(File dir) {
        try {
            this.fileSystem.deleteRecursively(dir);
        }
        catch (IOException e) {
            this.log.error("Unable to delete directory: " + dir, (Throwable)e);
        }
    }

    public static class DatabaseNotCleanlyShutDownException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "The database is not cleanly shutdown. The database needs recovery, in order to recover the database, please run the old version of the database on this store.";

        public DatabaseNotCleanlyShutDownException() {
            super(MESSAGE);
        }
    }

    public static class UnexpectedUpgradingStoreVersionException
    extends UnableToUpgradeException {
        protected static final String MESSAGE = "'%s' has a store version '%s' that we cannot upgrade from.";

        public UnexpectedUpgradingStoreVersionException(String filename, String actualVersion) {
            super(String.format(MESSAGE, filename, actualVersion));
        }
    }

    public static class UpgradingStoreVersionNotFoundException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "'%s' does not contain a store version, please ensure that the original database was shut down in a clean state.";

        public UpgradingStoreVersionNotFoundException(String filenameWithoutStoreVersion) {
            super(String.format(MESSAGE, filenameWithoutStoreVersion));
        }
    }

    public static class UpgradeMissingStoreFilesException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "Missing required store file '%s'.";

        public UpgradeMissingStoreFilesException(String filenameExpectedToExist) {
            super(String.format(MESSAGE, filenameExpectedToExist));
        }
    }

    public static class UnableToUpgradeException
    extends RuntimeException {
        public UnableToUpgradeException(String message, Throwable cause) {
            super(message, cause);
        }

        public UnableToUpgradeException(String message) {
            super(message);
        }
    }
}

