/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.core;

import java.util.Arrays;
import java.util.HashSet;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.neo4j.graphdb.ConstraintViolationException;
import org.neo4j.graphdb.Direction;
import org.neo4j.graphdb.DynamicRelationshipType;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.Transaction;
import org.neo4j.helpers.collection.PrefetchingIterator;
import org.neo4j.kernel.impl.AbstractNeo4jTestCase;
import org.neo4j.kernel.impl.MyRelTypes;

public class TestLoopRelationships
extends AbstractNeo4jTestCase {
    @Rule
    public ExpectedException exception = ExpectedException.none();

    @Test
    public void canCreateRelationshipBetweenTwoNodesWithLoopsThenDeleteOneOfTheNodesAndItsRelationships() throws Exception {
        Node source = this.getGraphDb().createNode();
        Node target = this.getGraphDb().createNode();
        source.createRelationshipTo(source, (RelationshipType)MyRelTypes.TEST);
        target.createRelationshipTo(target, (RelationshipType)MyRelTypes.TEST);
        source.createRelationshipTo(target, (RelationshipType)MyRelTypes.TEST);
        this.newTransaction();
        for (Relationship rel : target.getRelationships()) {
            rel.delete();
        }
        target.delete();
    }

    @Test
    public void canDeleteNodeAfterDeletingItsRelationshipsIfThoseRelationshipsIncludeLoops() throws Exception {
        Node node = this.getGraphDb().createNode();
        this.txCreateLoop(node);
        this.txCreateRel(node);
        this.txCreateLoop(node);
        for (Relationship rel : node.getRelationships()) {
            rel.delete();
        }
        node.delete();
        this.commit();
    }

    private void txCreateRel(Node node) {
        node.createRelationshipTo(this.getGraphDb().createNode(), (RelationshipType)MyRelTypes.TEST);
        this.newTransaction();
    }

    private void txCreateLoop(Node node) {
        node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
        this.newTransaction();
    }

    @Test
    public void canAddLoopRelationship() {
        Node node = this.getGraphDb().createNode();
        node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
        this.newTransaction();
        for (Direction dir : Direction.values()) {
            int count = 0;
            for (Relationship rel : node.getRelationships(dir)) {
                ++count;
                Assert.assertEquals((String)"start node", (Object)node, (Object)rel.getStartNode());
                Assert.assertEquals((String)"end node", (Object)node, (Object)rel.getEndNode());
                Assert.assertEquals((String)"other node", (Object)node, (Object)rel.getOtherNode(node));
            }
            Assert.assertEquals((String)(dir.name() + " relationship count"), (long)1L, (long)count);
        }
    }

    @Test
    public void canAddManyLoopRelationships() {
        this.testAddManyLoopRelationships(2);
        this.testAddManyLoopRelationships(3);
        this.testAddManyLoopRelationships(5);
    }

    private void testAddManyLoopRelationships(int count) {
        for (boolean[] loop : TestLoopRelationships.permutations(count)) {
            Node root = this.getGraphDb().createNode();
            Relationship[] relationships = new Relationship[count];
            for (int i = 0; i < count; ++i) {
                relationships[i] = loop[i] ? root.createRelationshipTo(root, (RelationshipType)MyRelTypes.TEST) : root.createRelationshipTo(this.getGraphDb().createNode(), (RelationshipType)MyRelTypes.TEST);
            }
            this.newTransaction();
            this.verifyRelationships(Arrays.toString(loop), root, loop, relationships);
        }
    }

    @Test
    public void canAddLoopRelationshipAndOtherRelationships() {
        this.testAddLoopRelationshipAndOtherRelationships(2);
        this.testAddLoopRelationshipAndOtherRelationships(3);
        this.testAddLoopRelationshipAndOtherRelationships(5);
    }

    private void testAddLoopRelationshipAndOtherRelationships(int size) {
        for (int i = 0; i < size; ++i) {
            Node root = this.getGraphDb().createNode();
            Relationship[] relationships = this.createRelationships(size, i, root);
            this.verifyRelationships(String.format("loop on %s of %s", i, size), root, i, relationships);
        }
    }

    @Test
    public void canAddAndRemoveLoopRelationshipAndOtherRelationships() {
        this.testAddAndRemoveLoopRelationshipAndOtherRelationships(2);
        this.testAddAndRemoveLoopRelationshipAndOtherRelationships(3);
        this.testAddAndRemoveLoopRelationshipAndOtherRelationships(5);
    }

    @Test
    public void getSingleRelationshipOnNodeWithOneLoopOnly() throws Exception {
        Node node = this.getGraphDb().createNode();
        Relationship singleRelationship = node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
        Assert.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.OUTGOING));
        Assert.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.INCOMING));
        Assert.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.BOTH));
        this.commit();
        this.newTransaction();
        Assert.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.OUTGOING));
        Assert.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.INCOMING));
        Assert.assertEquals((Object)singleRelationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.BOTH));
        this.finish();
    }

    @Test
    public void cannotDeleteNodeWithLoopStillAttached() throws Exception {
        Node node;
        GraphDatabaseService db = this.getGraphDb();
        try (Transaction tx = db.beginTx();){
            node = db.createNode();
            node.createRelationshipTo(node, (RelationshipType)DynamicRelationshipType.withName((String)"MAYOR_OF"));
            tx.success();
        }
        tx = this.newTransaction();
        node.delete();
        tx.success();
        this.exception.expect(ConstraintViolationException.class);
        this.exception.expectMessage("Cannot delete node<" + node.getId() + ">, because it still has relationships. To delete this node, you must first delete its relationships.");
        tx.close();
    }

    @Test
    public void getOtherNodeFunctionsCorrectly() throws Exception {
        Node node = this.getGraphDb().createNode();
        Relationship relationship = node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
        for (int i = 0; i < 2; ++i) {
            Assert.assertEquals((Object)node, (Object)relationship.getOtherNode(node));
            Assert.assertEquals(Arrays.asList(node, node), Arrays.asList(relationship.getNodes()));
            try {
                relationship.getOtherNode(this.getGraphDb().createNode());
                Assert.fail((String)"Should throw exception if another node is passed into loop.getOtherNode");
            }
            catch (NotFoundException notFoundException) {
                // empty catch block
            }
            this.newTransaction();
        }
    }

    @Test
    public void getNewlyCreatedLoopRelationshipFromCache() throws Exception {
        Node node = this.getGraphDb().createNode();
        node.createRelationshipTo(this.getGraphDb().createNode(), (RelationshipType)MyRelTypes.TEST);
        this.newTransaction();
        Relationship relationship = node.createRelationshipTo(node, (RelationshipType)MyRelTypes.TEST);
        this.newTransaction();
        Assert.assertEquals((Object)relationship, (Object)node.getSingleRelationship((RelationshipType)MyRelTypes.TEST, Direction.INCOMING));
    }

    private void testAddAndRemoveLoopRelationshipAndOtherRelationships(int size) {
        for (boolean[] delete : TestLoopRelationships.permutations(size)) {
            for (int i = 0; i < size; ++i) {
                Node root = this.getGraphDb().createNode();
                Relationship[] relationships = this.createRelationships(size, i, root);
                for (int j = 0; j < size; ++j) {
                    if (delete[j]) {
                        relationships[j].delete();
                        relationships[j] = null;
                    }
                    this.newTransaction();
                }
                this.verifyRelationships(String.format("loop on %s of %s, delete %s", i, size, Arrays.toString(delete)), root, i, relationships);
            }
        }
    }

    private static Iterable<boolean[]> permutations(final int size) {
        final int max = 1 << size;
        return () -> new PrefetchingIterator<boolean[]>(){
            int pos;

            protected boolean[] fetchNextOrNull() {
                if (this.pos < max) {
                    int cur = this.pos++;
                    boolean[] result = new boolean[size];
                    for (int i = 0; i < size; ++i) {
                        result[i] = (cur & 1) == 1;
                        cur >>= 1;
                    }
                    return result;
                }
                return null;
            }
        };
    }

    private Relationship[] createRelationships(int count, int loop, Node root) {
        Node[] nodes = new Node[count];
        for (int i = 0; i < count; ++i) {
            nodes[i] = loop == i ? root : this.getGraphDb().createNode();
        }
        this.newTransaction();
        Relationship[] relationships = new Relationship[count];
        for (int i = 0; i < count; ++i) {
            relationships[i] = root.createRelationshipTo(nodes[i], (RelationshipType)MyRelTypes.TEST);
            this.newTransaction();
        }
        return relationships;
    }

    private void verifyRelationships(String message, Node root, int loop, Relationship ... relationships) {
        boolean[] loops = new boolean[relationships.length];
        for (int i = 0; i < relationships.length; ++i) {
            loops[i] = i == loop;
        }
        this.verifyRelationships(message, root, loops, relationships);
    }

    private void verifyRelationships(String message, Node root, boolean[] loop, Relationship ... relationships) {
        for (Direction dir : Direction.values()) {
            HashSet<Relationship> expected = new HashSet<Relationship>();
            for (int i = 0; i < relationships.length; ++i) {
                if (relationships[i] == null || dir == Direction.INCOMING && !loop[i]) continue;
                expected.add(relationships[i]);
            }
            for (Relationship rel : root.getRelationships(dir)) {
                Assert.assertTrue((String)(message + ": unexpected relationship: " + rel), (boolean)expected.remove(rel));
            }
            Assert.assertTrue((String)(message + ": expected relationships not seen " + expected), (boolean)expected.isEmpty());
        }
    }
}

