/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.api.index;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.neo4j.graphdb.DependencyResolver;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.internal.kernel.api.IndexReference;
import org.neo4j.internal.kernel.api.exceptions.EntityNotFoundException;
import org.neo4j.internal.kernel.api.exceptions.KernelException;
import org.neo4j.internal.kernel.api.schema.SchemaDescriptor;
import org.neo4j.kernel.api.KernelTransaction;
import org.neo4j.kernel.api.Statement;
import org.neo4j.kernel.api.exceptions.index.IndexNotFoundKernelException;
import org.neo4j.kernel.api.schema.LabelSchemaDescriptor;
import org.neo4j.kernel.api.schema.SchemaDescriptorFactory;
import org.neo4j.kernel.api.schema.index.SchemaIndexDescriptor;
import org.neo4j.kernel.impl.api.index.IndexingService;
import org.neo4j.kernel.impl.api.index.UpdatesTracker;
import org.neo4j.kernel.impl.api.store.DefaultIndexReference;
import org.neo4j.kernel.impl.core.ThreadToStatementContextBridge;
import org.neo4j.kernel.impl.storageengine.impl.recordstorage.RecordStorageEngine;
import org.neo4j.kernel.impl.store.NeoStores;
import org.neo4j.kernel.impl.store.RecordStore;
import org.neo4j.kernel.impl.store.SchemaStorage;
import org.neo4j.kernel.impl.store.counts.CountsTracker;
import org.neo4j.kernel.internal.GraphDatabaseAPI;
import org.neo4j.kernel.monitoring.Monitors;
import org.neo4j.register.Register;
import org.neo4j.register.Registers;
import org.neo4j.test.rule.DatabaseRule;
import org.neo4j.test.rule.EmbeddedDatabaseRule;
import org.neo4j.values.storable.Values;

@RunWith(value=Parameterized.class)
public class IndexStatisticsTest {
    private static final double UNIQUE_NAMES = 10.0;
    private static final String[] NAMES = new String[]{"Andres", "Davide", "Jakub", "Chris", "Tobias", "Stefan", "Petra", "Rickard", "Mattias", "Emil", "Chris", "Chris"};
    private static final int CREATION_MULTIPLIER = Integer.getInteger(IndexStatisticsTest.class.getName() + ".creationMultiplier", 1000);
    private static final int MISSED_UPDATES_TOLERANCE = NAMES.length;
    private static final double DOUBLE_ERROR_TOLERANCE = 1.0E-5;
    @Parameterized.Parameter
    public boolean multiThreadedPopulationEnabled;
    @Rule
    public DatabaseRule dbRule;
    private GraphDatabaseService db;
    private ThreadToStatementContextBridge bridge;
    private final IndexOnlineMonitor indexOnlineMonitor;

    public IndexStatisticsTest() {
        this.dbRule = new EmbeddedDatabaseRule().withSetting(GraphDatabaseSettings.index_background_sampling_enabled, "false").withSetting(GraphDatabaseSettings.multi_threaded_schema_index_population_enabled, this.multiThreadedPopulationEnabled + "");
        this.indexOnlineMonitor = new IndexOnlineMonitor();
    }

    @Parameterized.Parameters(name="multiThreadedIndexPopulationEnabled = {0}")
    public static Object[] multiThreadedIndexPopulationEnabledValues() {
        return new Object[]{true, false};
    }

    @Before
    public void before() {
        GraphDatabaseAPI graphDatabaseAPI = this.dbRule.getGraphDatabaseAPI();
        this.db = graphDatabaseAPI;
        DependencyResolver dependencyResolver = graphDatabaseAPI.getDependencyResolver();
        this.bridge = (ThreadToStatementContextBridge)dependencyResolver.resolveDependency(ThreadToStatementContextBridge.class);
        ((Monitors)graphDatabaseAPI.getDependencyResolver().resolveDependency(Monitors.class)).addMonitorListener((Object)this.indexOnlineMonitor, new String[0]);
    }

    @Test
    public void shouldProvideIndexStatisticsForDataCreatedWhenPopulationBeforeTheIndexIsOnline() throws KernelException {
        this.createSomePersons();
        IndexReference index = this.createIndex("Person", "name");
        this.awaitIndexesOnline();
        Assert.assertEquals((double)0.75, (double)this.indexSelectivity(index), (double)1.0E-5);
        Assert.assertEquals((long)4L, (long)this.indexSize(index));
        Assert.assertEquals((long)0L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldNotSeeDataCreatedAfterPopulation() throws KernelException {
        IndexReference index = this.createIndex("Person", "name");
        this.awaitIndexesOnline();
        this.createSomePersons();
        Assert.assertEquals((double)1.0, (double)this.indexSelectivity(index), (double)1.0E-5);
        Assert.assertEquals((long)0L, (long)this.indexSize(index));
        Assert.assertEquals((long)4L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsForDataSeenDuringPopulationAndIgnoreDataCreatedAfterPopulation() throws KernelException {
        this.createSomePersons();
        IndexReference index = this.createIndex("Person", "name");
        this.awaitIndexesOnline();
        this.createSomePersons();
        Assert.assertEquals((double)0.75, (double)this.indexSelectivity(index), (double)1.0E-5);
        Assert.assertEquals((long)4L, (long)this.indexSize(index));
        Assert.assertEquals((long)4L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldRemoveIndexStatisticsAfterIndexIsDeleted() throws KernelException {
        this.createSomePersons();
        IndexReference index = this.createIndex("Person", "name");
        this.awaitIndexesOnline();
        SchemaStorage storage = new SchemaStorage((RecordStore)this.neoStores().getSchemaStore());
        long indexId = storage.indexGetForSchema(DefaultIndexReference.toDescriptor((IndexReference)index)).getId();
        this.dropIndex(index);
        try {
            this.indexSelectivity(index);
            Assert.fail((String)"Expected IndexNotFoundKernelException to be thrown");
        }
        catch (IndexNotFoundKernelException e) {
            Register.DoubleLongRegister actual = this.getTracker().indexSample(indexId, Registers.newDoubleLongRegister());
            this.assertDoubleLongEquals(0L, 0L, actual);
        }
        Register.DoubleLongRegister actual = this.getTracker().indexUpdatesAndSize(indexId, Registers.newDoubleLongRegister());
        this.assertDoubleLongEquals(0L, 0L, actual);
    }

    @Test
    public void shouldProvideIndexSelectivityWhenThereAreManyDuplicates() throws Exception {
        int created = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER).length;
        IndexReference index = this.createIndex("Person", "name");
        this.awaitIndexesOnline();
        double expectedSelectivity = 10.0 / (double)created;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(created, this.indexSize(index));
        Assert.assertEquals((long)0L, (long)this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditions() throws Exception {
        int initialNodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER).length;
        IndexReference index = this.createIndex("Person", "name");
        UpdatesTracker updatesTracker = this.executeCreations(index, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        IndexStatisticsTest.assertCorrectIndexUpdates(updatesTracker.createdAfterPopulation(), this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndDeletions() throws Exception {
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexReference index = this.createIndex("Person", "name");
        UpdatesTracker updatesTracker = this.executeCreationsAndDeletions(nodes, index, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation() - updatesTracker.deletedDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = updatesTracker.deletedAfterPopulation() + updatesTracker.createdAfterPopulation();
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndChanges() throws Exception {
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexReference index = this.createIndex("Person", "name");
        UpdatesTracker updatesTracker = this.executeCreationsAndUpdates(nodes, index, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        int expectedIndexUpdates = updatesTracker.createdAfterPopulation() + updatesTracker.updatedAfterPopulation();
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    public void shouldProvideIndexStatisticsWhenIndexIsBuiltViaPopulationAndConcurrentAdditionsAndChangesAndDeletions() throws Exception {
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        IndexReference index = this.createIndex("Person", "name");
        UpdatesTracker updatesTracker = this.executeCreationsDeletionsAndUpdates(nodes, index, CREATION_MULTIPLIER);
        this.awaitIndexesOnline();
        int seenWhilePopulating = initialNodes + updatesTracker.createdDuringPopulation() - updatesTracker.deletedDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        int expectedIndexUpdates = updatesTracker.deletedAfterPopulation() + updatesTracker.createdAfterPopulation() + updatesTracker.updatedAfterPopulation();
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index));
        IndexStatisticsTest.assertCorrectIndexSize(seenWhilePopulating, this.indexSize(index));
        IndexStatisticsTest.assertCorrectIndexUpdates(expectedIndexUpdates, this.indexUpdates(index));
    }

    @Test
    public void shouldWorkWhileHavingHeavyConcurrentUpdates() throws Exception {
        long[] nodes = this.repeatCreateNamedPeopleFor(NAMES.length * CREATION_MULTIPLIER);
        int initialNodes = nodes.length;
        int threads = 5;
        ExecutorService executorService = Executors.newFixedThreadPool(threads);
        IndexReference index = this.createIndex("Person", "name");
        ArrayList<Callable<UpdatesTracker>> jobs = new ArrayList<Callable<UpdatesTracker>>(threads);
        for (int i = 0; i < threads; ++i) {
            jobs.add(() -> this.executeCreationsDeletionsAndUpdates(nodes, index, CREATION_MULTIPLIER));
        }
        List futures = executorService.invokeAll(jobs);
        UpdatesTracker result = new UpdatesTracker();
        result.notifyPopulationCompleted();
        for (Future future : futures) {
            result.add((UpdatesTracker)future.get());
        }
        this.awaitIndexesOnline();
        executorService.awaitTermination(1L, TimeUnit.SECONDS);
        executorService.shutdown();
        int tolerance = MISSED_UPDATES_TOLERANCE * threads;
        double doubleTolerance = 1.0E-5 * (double)threads;
        int seenWhilePopulating = initialNodes + result.createdDuringPopulation() - result.deletedDuringPopulation();
        double expectedSelectivity = 10.0 / (double)seenWhilePopulating;
        IndexStatisticsTest.assertCorrectIndexSelectivity(expectedSelectivity, this.indexSelectivity(index), doubleTolerance);
        IndexStatisticsTest.assertCorrectIndexSize("Tracker had " + result, seenWhilePopulating, this.indexSize(index), tolerance);
        int expectedIndexUpdates = result.deletedAfterPopulation() + result.createdAfterPopulation() + result.updatedAfterPopulation();
        IndexStatisticsTest.assertCorrectIndexUpdates("Tracker had " + result, expectedIndexUpdates, this.indexUpdates(index), tolerance);
    }

    private void deleteNode(long nodeId) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            this.db.getNodeById(nodeId).delete();
            tx.success();
        }
    }

    private boolean changeName(long nodeId, String propertyKeyName, Object newValue) {
        boolean changeIndexedNode = false;
        try (Transaction tx = this.db.beginTx();){
            Node node = this.db.getNodeById(nodeId);
            Object oldValue = node.getProperty(propertyKeyName);
            if (!oldValue.equals(newValue)) {
                changeIndexedNode = true;
            }
            node.setProperty(propertyKeyName, newValue);
            tx.success();
        }
        return changeIndexedNode;
    }

    private int createNamedPeople(long[] nodes, int offset) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            for (String name : NAMES) {
                long nodeId = this.createNode(this.bridge.getKernelTransactionBoundToThisThread(true), "Person", "name", name);
                if (nodes == null) continue;
                nodes[offset++] = nodeId;
            }
            tx.success();
        }
        return NAMES.length;
    }

    private long[] repeatCreateNamedPeopleFor(int totalNumberOfPeople) throws Exception {
        long[] nodes = new long[totalNumberOfPeople];
        int threads = 100;
        int peoplePerThread = totalNumberOfPeople / 100;
        ExecutorService service = Executors.newFixedThreadPool(100);
        AtomicReference exception = new AtomicReference();
        ArrayList<Callable<Void>> jobs = new ArrayList<Callable<Void>>(100);
        int i = 0;
        while (i < 100) {
            int finalI = i++;
            jobs.add(() -> {
                for (int offset = finalI * peoplePerThread; offset < (finalI + 1) * peoplePerThread; offset += this.createNamedPeople(nodes, offset)) {
                    try {
                        continue;
                    }
                    catch (KernelException e) {
                        exception.compareAndSet(null, e);
                        throw new RuntimeException(e);
                    }
                }
                return null;
            });
        }
        for (Future job : service.invokeAll(jobs)) {
            job.get();
        }
        service.awaitTermination(1L, TimeUnit.SECONDS);
        service.shutdown();
        Exception ex = (Exception)exception.get();
        if (ex != null) {
            throw ex;
        }
        return nodes;
    }

    private void dropIndex(IndexReference index) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = this.bridge.getKernelTransactionBoundToThisThread(true);
            try (Statement ignore = ktx.acquireStatement();){
                ktx.schemaWrite().indexDrop(index);
            }
            tx.success();
        }
    }

    private long indexSize(IndexReference reference) throws KernelException {
        return ((IndexingService)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(IndexingService.class)).indexUpdatesAndSize((SchemaDescriptor)SchemaDescriptorFactory.forLabel((int)reference.label(), (int[])reference.properties())).readSecond();
    }

    private long indexUpdates(IndexReference reference) throws KernelException {
        return ((IndexingService)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(IndexingService.class)).indexUpdatesAndSize((SchemaDescriptor)SchemaDescriptorFactory.forLabel((int)reference.label(), (int[])reference.properties())).readFirst();
    }

    private double indexSelectivity(IndexReference reference) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            double selectivity = this.getSelectivity(reference);
            tx.success();
            double d = selectivity;
            return d;
        }
    }

    private double getSelectivity(IndexReference reference) throws IndexNotFoundKernelException {
        return this.bridge.getKernelTransactionBoundToThisThread(true).schemaRead().indexUniqueValuesSelectivity(reference);
    }

    private CountsTracker getTracker() {
        return ((RecordStorageEngine)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(RecordStorageEngine.class)).testAccessNeoStores().getCounts();
    }

    private void createSomePersons() throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            KernelTransaction ktx = this.bridge.getKernelTransactionBoundToThisThread(true);
            this.createNode(ktx, "Person", "name", "Davide");
            this.createNode(ktx, "Person", "name", "Stefan");
            this.createNode(ktx, "Person", "name", "John");
            this.createNode(ktx, "Person", "name", "John");
            tx.success();
        }
    }

    private long createNode(KernelTransaction ktx, String labelName, String propertyKeyName, Object value) throws KernelException {
        int labelId = ktx.tokenWrite().labelGetOrCreateForName(labelName);
        int propertyKeyId = ktx.tokenWrite().propertyKeyGetOrCreateForName(propertyKeyName);
        long nodeId = ktx.dataWrite().nodeCreate();
        ktx.dataWrite().nodeAddLabel(nodeId, labelId);
        ktx.dataWrite().nodeSetProperty(nodeId, propertyKeyId, Values.of((Object)value));
        return nodeId;
    }

    private IndexReference createIndex(String labelName, String propertyKeyName) throws KernelException {
        try (Transaction tx = this.db.beginTx();){
            IndexReference index;
            KernelTransaction ktx = this.bridge.getKernelTransactionBoundToThisThread(true);
            try (Statement ignore = ktx.acquireStatement();){
                int labelId = ktx.tokenWrite().labelGetOrCreateForName(labelName);
                int propertyKeyId = ktx.tokenWrite().propertyKeyGetOrCreateForName(propertyKeyName);
                LabelSchemaDescriptor descriptor = SchemaDescriptorFactory.forLabel((int)labelId, (int[])new int[]{propertyKeyId});
                index = ktx.schemaWrite().indexCreate((SchemaDescriptor)descriptor);
            }
            tx.success();
            IndexReference indexReference = index;
            return indexReference;
        }
    }

    private NeoStores neoStores() {
        return ((RecordStorageEngine)((GraphDatabaseAPI)this.db).getDependencyResolver().resolveDependency(RecordStorageEngine.class)).testAccessNeoStores();
    }

    private void awaitIndexesOnline() {
        try (Transaction ignored = this.db.beginTx();){
            this.db.schema().awaitIndexesOnline(3L, TimeUnit.MINUTES);
        }
    }

    private UpdatesTracker executeCreations(IndexReference index, int numberOfCreations) throws KernelException {
        return this.internalExecuteCreationsDeletionsAndUpdates(null, index, numberOfCreations, false, false);
    }

    private UpdatesTracker executeCreationsAndDeletions(long[] nodes, IndexReference index, int numberOfCreations) throws KernelException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, index, numberOfCreations, true, false);
    }

    private UpdatesTracker executeCreationsAndUpdates(long[] nodes, IndexReference index, int numberOfCreations) throws KernelException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, index, numberOfCreations, false, true);
    }

    private UpdatesTracker executeCreationsDeletionsAndUpdates(long[] nodes, IndexReference index, int numberOfCreations) throws KernelException {
        return this.internalExecuteCreationsDeletionsAndUpdates(nodes, index, numberOfCreations, true, true);
    }

    private UpdatesTracker internalExecuteCreationsDeletionsAndUpdates(long[] nodes, IndexReference index, int numberOfCreations, boolean allowDeletions, boolean allowUpdates) throws KernelException {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        UpdatesTracker updatesTracker = new UpdatesTracker();
        int offset = 0;
        while (updatesTracker.created() < numberOfCreations) {
            int created = this.createNamedPeople(nodes, offset);
            offset += created;
            updatesTracker.increaseCreated(created);
            this.notifyIfPopulationCompleted(index, updatesTracker);
            if (allowDeletions && updatesTracker.created() % 24 == 0) {
                long nodeId = nodes[((Random)random).nextInt(nodes.length)];
                try {
                    this.deleteNode(nodeId);
                    updatesTracker.increaseDeleted(1);
                }
                catch (NotFoundException | EntityNotFoundException throwable) {
                    // empty catch block
                }
                this.notifyIfPopulationCompleted(index, updatesTracker);
            }
            if (!allowUpdates || updatesTracker.created() % 24 != 0) continue;
            int randomIndex = ((Random)random).nextInt(nodes.length);
            try {
                if (this.changeName(nodes[randomIndex], "name", NAMES[((Random)random).nextInt(NAMES.length)])) {
                    updatesTracker.increaseUpdated(1);
                }
            }
            catch (NotFoundException notFoundException) {
                // empty catch block
            }
            this.notifyIfPopulationCompleted(index, updatesTracker);
        }
        updatesTracker.notifyPopulationCompleted();
        return updatesTracker;
    }

    private void notifyIfPopulationCompleted(IndexReference index, UpdatesTracker updatesTracker) {
        if (this.isCompletedPopulation(index, updatesTracker)) {
            updatesTracker.notifyPopulationCompleted();
        }
    }

    private boolean isCompletedPopulation(IndexReference index, UpdatesTracker updatesTracker) {
        return !updatesTracker.isPopulationCompleted() && this.indexOnlineMonitor.isIndexOnline(index);
    }

    private void assertDoubleLongEquals(long expectedUniqueValue, long expectedSampledSize, Register.DoubleLongRegister register) {
        Assert.assertEquals((long)expectedUniqueValue, (long)register.readFirst());
        Assert.assertEquals((long)expectedSampledSize, (long)register.readSecond());
    }

    private static void assertCorrectIndexSize(long expected, long actual) {
        IndexStatisticsTest.assertCorrectIndexSize("", expected, actual, MISSED_UPDATES_TOLERANCE);
    }

    private static void assertCorrectIndexSize(String info, long expected, long actual, int tolerance) {
        String message = String.format("Expected number of entries to not differ by more than %d (expected: %d actual: %d) %s", tolerance, expected, actual, info);
        Assert.assertTrue((String)message, (Math.abs(expected - actual) <= (long)tolerance ? 1 : 0) != 0);
    }

    private static void assertCorrectIndexUpdates(long expected, long actual) {
        IndexStatisticsTest.assertCorrectIndexUpdates("", expected, actual, MISSED_UPDATES_TOLERANCE);
    }

    private static void assertCorrectIndexUpdates(String info, long expected, long actual, int tolerance) {
        String message = String.format("Expected number of index updates to not differ by more than %d (expected: %d actual: %d). %s", tolerance, expected, actual, info);
        Assert.assertTrue((String)message, (Math.abs(expected - actual) <= (long)tolerance ? 1 : 0) != 0);
    }

    private static void assertCorrectIndexSelectivity(double expected, double actual) {
        IndexStatisticsTest.assertCorrectIndexSelectivity(expected, actual, 1.0E-5);
    }

    private static void assertCorrectIndexSelectivity(double expected, double actual, double tolerance) {
        String message = String.format("Expected number of entries to not differ by more than %f (expected: %f actual: %f)", tolerance, expected, actual);
        Assert.assertEquals((String)message, (double)expected, (double)actual, (double)tolerance);
    }

    private static class IndexOnlineMonitor
    extends IndexingService.MonitorAdapter {
        private final Set<IndexReference> onlineIndexes = Collections.newSetFromMap(new ConcurrentHashMap());

        private IndexOnlineMonitor() {
        }

        public void populationCompleteOn(SchemaIndexDescriptor descriptor) {
            this.onlineIndexes.add(DefaultIndexReference.fromDescriptor((SchemaIndexDescriptor)descriptor));
        }

        public boolean isIndexOnline(IndexReference descriptor) {
            return this.onlineIndexes.contains(descriptor);
        }
    }
}

