/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.index.schema;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.lang3.mutable.MutableLong;
import org.eclipse.collections.api.set.primitive.MutableLongSet;
import org.eclipse.collections.impl.factory.primitive.LongSets;
import org.eclipse.collections.impl.set.mutable.primitive.LongHashSet;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.neo4j.index.internal.gbptree.Layout;
import org.neo4j.index.internal.gbptree.SimpleLongLayout;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.impl.index.schema.BlockEntry;
import org.neo4j.kernel.impl.index.schema.BlockEntryReader;
import org.neo4j.kernel.impl.index.schema.BlockReader;
import org.neo4j.kernel.impl.index.schema.BlockStorage;
import org.neo4j.kernel.impl.index.schema.ByteBufferFactory;
import org.neo4j.test.Barrier;
import org.neo4j.test.OtherThreadExecutor;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.RandomExtension;
import org.neo4j.test.extension.TestDirectoryExtension;
import org.neo4j.test.rule.RandomRule;
import org.neo4j.test.rule.TestDirectory;

@ExtendWith(value={TestDirectoryExtension.class, RandomExtension.class})
class BlockStorageTest {
    @Inject
    TestDirectory directory;
    @Inject
    RandomRule random;
    private File file;
    private FileSystemAbstraction fileSystem;
    private SimpleLongLayout layout;

    BlockStorageTest() {
    }

    @BeforeEach
    void setup() {
        this.file = this.directory.file("block");
        this.fileSystem = this.directory.getFileSystem();
        this.layout = SimpleLongLayout.longLayout().withFixedSize(this.random.nextBoolean()).withKeyPadding(this.random.nextInt(10)).build();
    }

    @Test
    void shouldCreateAndCloseTheBlockFile() throws IOException {
        Assertions.assertFalse((boolean)this.fileSystem.fileExists(this.file));
        try (BlockStorage ignored = new BlockStorage((Layout)this.layout, ByteBufferFactory.HEAP_BUFFER_FACTORY, this.fileSystem, this.file, BlockStorage.Monitor.NO_MONITOR, 100);){
            Assertions.assertTrue((boolean)this.fileSystem.fileExists(this.file));
        }
    }

    @Test
    void shouldAddSingleEntryInLastBlock() throws IOException {
        TrackingMonitor monitor = new TrackingMonitor();
        int blockSize = 100;
        MutableLong key = new MutableLong(10L);
        MutableLong value = new MutableLong(20L);
        try (BlockStorage storage = new BlockStorage((Layout)this.layout, ByteBufferFactory.HEAP_BUFFER_FACTORY, this.fileSystem, this.file, (BlockStorage.Monitor)monitor, blockSize);){
            storage.add((Object)key, (Object)value);
            storage.doneAdding();
            Assertions.assertEquals((int)1, (int)monitor.blockFlushedCallCount);
            Assertions.assertEquals((long)1L, (long)monitor.lastKeyCount);
            Assertions.assertEquals((long)(16L + monitor.totalEntrySize), (long)monitor.lastNumberOfBytes);
            Assertions.assertEquals((long)blockSize, (long)monitor.lastPositionAfterFlush);
            MatcherAssert.assertThat((Object)monitor.lastNumberOfBytes, (Matcher)Matchers.lessThan((Comparable)Integer.valueOf(blockSize)));
            this.assertContents(this.layout, (BlockStorage<MutableLong, MutableLong>)storage, Collections.singletonList(Collections.singletonList(new BlockEntry((Object)key, (Object)value))));
        }
    }

    @Test
    void shouldSortAndAddMultipleEntriesInLastBlock() throws IOException {
        TrackingMonitor monitor = new TrackingMonitor();
        int blockSize = 1000;
        ArrayList<BlockEntry<MutableLong, MutableLong>> expected = new ArrayList<BlockEntry<MutableLong, MutableLong>>();
        try (BlockStorage storage = new BlockStorage((Layout)this.layout, ByteBufferFactory.HEAP_BUFFER_FACTORY, this.fileSystem, this.file, (BlockStorage.Monitor)monitor, blockSize);){
            for (int i = 0; i < 10; ++i) {
                long keyNumber = this.random.nextLong(10000000L);
                MutableLong key = new MutableLong(keyNumber);
                MutableLong value = new MutableLong((long)i);
                storage.add((Object)key, (Object)value);
                expected.add((BlockEntry<MutableLong, MutableLong>)new BlockEntry((Object)key, (Object)value));
            }
            storage.doneAdding();
            this.sort(expected);
            this.assertContents(this.layout, (BlockStorage<MutableLong, MutableLong>)storage, Collections.singletonList(expected));
        }
    }

    @Test
    void shouldSortAndAddMultipleEntriesInMultipleBlocks() throws IOException {
        TrackingMonitor monitor = new TrackingMonitor();
        int blockSize = 1000;
        try (BlockStorage storage = new BlockStorage((Layout)this.layout, ByteBufferFactory.HEAP_BUFFER_FACTORY, this.fileSystem, this.file, (BlockStorage.Monitor)monitor, blockSize);){
            List<List<BlockEntry<MutableLong, MutableLong>>> expectedBlocks = this.addACoupleOfBlocksOfEntries(monitor, (BlockStorage<MutableLong, MutableLong>)storage, 3);
            this.assertContents(this.layout, (BlockStorage<MutableLong, MutableLong>)storage, expectedBlocks);
        }
    }

    @Test
    void shouldMergeWhenEmpty() throws IOException {
        TrackingMonitor monitor = new TrackingMonitor();
        int blockSize = 1000;
        try (BlockStorage storage = new BlockStorage((Layout)this.layout, ByteBufferFactory.HEAP_BUFFER_FACTORY, this.fileSystem, this.file, (BlockStorage.Monitor)monitor, blockSize);){
            storage.merge(this.randomMergeFactor(), BlockStorage.NOT_CANCELLABLE);
            Assertions.assertEquals((int)0, (int)monitor.mergeIterationCallCount);
            this.assertContents(this.layout, (BlockStorage<MutableLong, MutableLong>)storage, Collections.emptyList());
        }
    }

    @Test
    void shouldMergeSingleBlock() throws IOException {
        TrackingMonitor monitor = new TrackingMonitor();
        int blockSize = 1000;
        try (BlockStorage storage = new BlockStorage((Layout)this.layout, ByteBufferFactory.HEAP_BUFFER_FACTORY, this.fileSystem, this.file, (BlockStorage.Monitor)monitor, blockSize);){
            List<List<BlockEntry<MutableLong, MutableLong>>> expectedBlocks = Collections.singletonList(this.addEntries((BlockStorage<MutableLong, MutableLong>)storage, 4));
            storage.doneAdding();
            storage.merge(this.randomMergeFactor(), BlockStorage.NOT_CANCELLABLE);
            Assertions.assertEquals((int)0, (int)monitor.mergeIterationCallCount);
            this.assertContents(this.layout, (BlockStorage<MutableLong, MutableLong>)storage, expectedBlocks);
        }
    }

    @Test
    void shouldMergeMultipleBlocks() throws IOException {
        TrackingMonitor monitor = new TrackingMonitor();
        int blockSize = 1000;
        try (BlockStorage storage = new BlockStorage((Layout)this.layout, ByteBufferFactory.HEAP_BUFFER_FACTORY, this.fileSystem, this.file, (BlockStorage.Monitor)monitor, blockSize);){
            int numberOfBlocks = this.random.nextInt(100) + 2;
            List<List<BlockEntry<MutableLong, MutableLong>>> expectedBlocks = this.addACoupleOfBlocksOfEntries(monitor, (BlockStorage<MutableLong, MutableLong>)storage, numberOfBlocks);
            storage.doneAdding();
            storage.merge(this.randomMergeFactor(), BlockStorage.NOT_CANCELLABLE);
            this.assertContents(this.layout, (BlockStorage<MutableLong, MutableLong>)storage, this.asOneBigBlock(expectedBlocks));
            MatcherAssert.assertThat((Object)monitor.totalEntriesToMerge, (Matcher)Matchers.greaterThanOrEqualTo((Comparable)Long.valueOf(monitor.entryAddedCallCount)));
            Assertions.assertEquals((long)monitor.totalEntriesToMerge, (long)monitor.entriesMerged);
        }
    }

    @Test
    void shouldOnlyLeaveSingleFileAfterMerge() throws IOException {
        TrackingMonitor monitor = new TrackingMonitor();
        int blockSize = 1000;
        try (BlockStorage storage = new BlockStorage((Layout)this.layout, ByteBufferFactory.HEAP_BUFFER_FACTORY, this.fileSystem, this.file, (BlockStorage.Monitor)monitor, blockSize);){
            int numberOfBlocks = this.random.nextInt(100) + 2;
            this.addACoupleOfBlocksOfEntries(monitor, (BlockStorage<MutableLong, MutableLong>)storage, numberOfBlocks);
            storage.doneAdding();
            storage.merge(2, BlockStorage.NOT_CANCELLABLE);
            File[] files = this.fileSystem.listFiles(this.directory.directory());
            Assertions.assertEquals((int)1, (int)files.length, (String)"Expected only a single file to exist after merge.");
        }
    }

    @Test
    void shouldNotAcceptAddedEntriesAfterDoneAdding() throws IOException {
        try (BlockStorage storage = new BlockStorage((Layout)this.layout, ByteBufferFactory.HEAP_BUFFER_FACTORY, this.fileSystem, this.file, BlockStorage.Monitor.NO_MONITOR, 100);){
            storage.doneAdding();
            Assertions.assertThrows(IllegalStateException.class, () -> storage.add((Object)new MutableLong(0L), (Object)new MutableLong(1L)));
        }
    }

    @Test
    void shouldNotFlushAnythingOnEmptyBufferInDoneAdding() throws IOException {
        TrackingMonitor monitor = new TrackingMonitor();
        try (BlockStorage storage = new BlockStorage((Layout)this.layout, ByteBufferFactory.HEAP_BUFFER_FACTORY, this.fileSystem, this.file, (BlockStorage.Monitor)monitor, 100);){
            storage.doneAdding();
            Assertions.assertEquals((int)0, (int)monitor.blockFlushedCallCount);
        }
    }

    @Test
    void shouldNoticeCancelRequest() throws IOException, ExecutionException, InterruptedException {
        final Barrier.Control barrier = new Barrier.Control();
        TrackingMonitor monitor = new TrackingMonitor(){

            @Override
            public void mergedBlocks(long resultingBlockSize, long resultingEntryCount, long numberOfBlocks) {
                super.mergedBlocks(resultingBlockSize, resultingEntryCount, numberOfBlocks);
                barrier.reached();
            }
        };
        int blocks = 10;
        int mergeFactor = 2;
        LongHashSet uniqueKeys = new LongHashSet();
        AtomicBoolean cancelled = new AtomicBoolean();
        try (BlockStorage storage = new BlockStorage((Layout)this.layout, ByteBufferFactory.HEAP_BUFFER_FACTORY, this.fileSystem, this.file, (BlockStorage.Monitor)monitor, 100);
             OtherThreadExecutor t2 = new OtherThreadExecutor("T2", null);){
            while (monitor.blockFlushedCallCount < blocks) {
                storage.add((Object)this.uniqueKey((MutableLongSet)uniqueKeys), (Object)new MutableLong());
            }
            storage.doneAdding();
            Future merge = t2.executeDontWait(OtherThreadExecutor.command(() -> storage.merge(mergeFactor, cancelled::get)));
            barrier.awaitUninterruptibly();
            cancelled.set(true);
            barrier.release();
            merge.get();
        }
        Assertions.assertEquals((int)1, (int)monitor.mergeIterationCallCount);
    }

    @Test
    void shouldCalculateCorrectNumberOfEntriesToWriteDuringMerge() {
        long entryCountForOneBlock = BlockStorage.calculateNumberOfEntriesWrittenDuringMerges((long)100L, (long)1L, (int)2);
        long entryCountForMergeFactorBlocks = BlockStorage.calculateNumberOfEntriesWrittenDuringMerges((long)100L, (long)4L, (int)4);
        long entryCountForMoreThanMergeFactorBlocks = BlockStorage.calculateNumberOfEntriesWrittenDuringMerges((long)100L, (long)5L, (int)4);
        long entryCountForThreeFactorsMergeFactorBlocks = BlockStorage.calculateNumberOfEntriesWrittenDuringMerges((long)100L, (long)61L, (int)4);
        Assertions.assertEquals((long)0L, (long)entryCountForOneBlock);
        Assertions.assertEquals((long)100L, (long)entryCountForMergeFactorBlocks);
        Assertions.assertEquals((long)200L, (long)entryCountForMoreThanMergeFactorBlocks);
        Assertions.assertEquals((long)300L, (long)entryCountForThreeFactorsMergeFactorBlocks);
    }

    private Iterable<List<BlockEntry<MutableLong, MutableLong>>> asOneBigBlock(List<List<BlockEntry<MutableLong, MutableLong>>> expectedBlocks) {
        ArrayList<BlockEntry<MutableLong, MutableLong>> all = new ArrayList<BlockEntry<MutableLong, MutableLong>>();
        for (List<BlockEntry<MutableLong, MutableLong>> expectedBlock : expectedBlocks) {
            all.addAll(expectedBlock);
        }
        this.sort(all);
        return Collections.singletonList(all);
    }

    private int randomMergeFactor() {
        return this.random.nextInt(2, 8);
    }

    private List<BlockEntry<MutableLong, MutableLong>> addEntries(BlockStorage<MutableLong, MutableLong> storage, int numberOfEntries) throws IOException {
        MutableLongSet uniqueKeys = LongSets.mutable.empty();
        ArrayList<BlockEntry<MutableLong, MutableLong>> entries = new ArrayList<BlockEntry<MutableLong, MutableLong>>();
        for (int i = 0; i < numberOfEntries; ++i) {
            MutableLong key = this.uniqueKey(uniqueKeys);
            MutableLong value = new MutableLong(this.random.nextLong(10000000L));
            storage.add((Object)key, (Object)value);
            entries.add((BlockEntry<MutableLong, MutableLong>)new BlockEntry((Object)key, (Object)value));
        }
        this.sort(entries);
        return entries;
    }

    private List<List<BlockEntry<MutableLong, MutableLong>>> addACoupleOfBlocksOfEntries(TrackingMonitor monitor, BlockStorage<MutableLong, MutableLong> storage, int numberOfBlocks) throws IOException {
        assert (numberOfBlocks != 1);
        MutableLongSet uniqueKeys = LongSets.mutable.empty();
        ArrayList<List<BlockEntry<MutableLong, MutableLong>>> expected = new ArrayList<List<BlockEntry<MutableLong, MutableLong>>>();
        ArrayList<Object> currentExpected = new ArrayList<BlockEntry<MutableLong, MutableLong>>();
        long currentBlock = 0L;
        while (monitor.blockFlushedCallCount < numberOfBlocks - 1) {
            MutableLong key = this.uniqueKey(uniqueKeys);
            MutableLong value = new MutableLong(this.random.nextLong(10000000L));
            storage.add((Object)key, (Object)value);
            if ((long)monitor.blockFlushedCallCount > currentBlock) {
                this.sort(currentExpected);
                expected.add(currentExpected);
                currentExpected = new ArrayList();
                currentBlock = monitor.blockFlushedCallCount;
            }
            currentExpected.add((BlockEntry<MutableLong, MutableLong>)new BlockEntry((Object)key, (Object)value));
        }
        storage.doneAdding();
        if (!currentExpected.isEmpty()) {
            expected.add(currentExpected);
        }
        return expected;
    }

    private MutableLong uniqueKey(MutableLongSet uniqueKeys) {
        MutableLong key;
        while (!uniqueKeys.add((key = new MutableLong(this.random.nextLong(10000000L))).longValue())) {
        }
        return key;
    }

    private void sort(List<BlockEntry<MutableLong, MutableLong>> entries) {
        entries.sort(Comparator.comparingLong(p -> ((MutableLong)p.key()).longValue()));
    }

    private void assertContents(SimpleLongLayout layout, BlockStorage<MutableLong, MutableLong> storage, Iterable<List<BlockEntry<MutableLong, MutableLong>>> expectedBlocks) throws IOException {
        try (BlockReader reader = storage.reader();){
            for (List<BlockEntry<MutableLong, MutableLong>> expectedBlock : expectedBlocks) {
                BlockEntryReader block = reader.nextBlock();
                Throwable throwable = null;
                try {
                    Assertions.assertNotNull((Object)block);
                    Assertions.assertEquals((long)expectedBlock.size(), (long)block.entryCount());
                    for (BlockEntry<MutableLong, MutableLong> expectedEntry : expectedBlock) {
                        Assertions.assertTrue((boolean)block.next());
                        Assertions.assertEquals((int)0, (int)layout.compare((MutableLong)expectedEntry.key(), (MutableLong)block.key()));
                        Assertions.assertEquals((Object)expectedEntry.value(), (Object)block.value());
                    }
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (block == null) continue;
                    if (throwable != null) {
                        try {
                            block.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    block.close();
                }
            }
        }
    }

    private static class TrackingMonitor
    implements BlockStorage.Monitor {
        long entryAddedCallCount;
        int lastEntrySize;
        long totalEntrySize;
        int blockFlushedCallCount;
        long lastKeyCount;
        int lastNumberOfBytes;
        long lastPositionAfterFlush;
        int mergeIterationCallCount;
        long lastNumberOfBlocksBefore;
        long lastNumberOfBlocksAfter;
        long totalEntriesToMerge;
        long entriesMerged;

        private TrackingMonitor() {
        }

        public void entryAdded(int entrySize) {
            ++this.entryAddedCallCount;
            this.lastEntrySize = entrySize;
            this.totalEntrySize += (long)entrySize;
        }

        public void blockFlushed(long keyCount, int numberOfBytes, long positionAfterFlush) {
            ++this.blockFlushedCallCount;
            this.lastKeyCount = keyCount;
            this.lastNumberOfBytes = numberOfBytes;
            this.lastPositionAfterFlush = positionAfterFlush;
        }

        public void mergeIterationFinished(long numberOfBlocksBefore, long numberOfBlocksAfter) {
            ++this.mergeIterationCallCount;
            this.lastNumberOfBlocksBefore = numberOfBlocksBefore;
            this.lastNumberOfBlocksAfter = numberOfBlocksAfter;
        }

        public void mergedBlocks(long resultingBlockSize, long resultingEntryCount, long numberOfBlocks) {
        }

        public void mergeStarted(long entryCount, long totalEntriesToWriteDuringMerge) {
            this.totalEntriesToMerge = totalEntriesToWriteDuringMerge;
        }

        public void entriesMerged(int entries) {
            this.entriesMerged += (long)entries;
        }
    }
}

