/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction.log;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.util.Random;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.neo4j.io.fs.DefaultFileSystemAbstraction;
import org.neo4j.io.fs.PhysicalFlushableChannel;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.fs.StoreFileChannel;
import org.neo4j.io.memory.ByteBuffers;
import org.neo4j.kernel.impl.transaction.log.LogPosition;
import org.neo4j.kernel.impl.transaction.log.LogPositionMarker;
import org.neo4j.kernel.impl.transaction.log.LogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.PhysicalFlushableLogChannel;
import org.neo4j.kernel.impl.transaction.log.PhysicalLogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.PositionAwarePhysicalFlushableChecksumChannel;
import org.neo4j.kernel.impl.transaction.log.files.ChannelNativeAccessor;
import org.neo4j.kernel.impl.transaction.log.files.LogFileChannelNativeAccessor;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.rule.TestDirectory;

@TestDirectoryExtension
class PhysicalFlushableChannelTest {
    @Inject
    private DefaultFileSystemAbstraction fileSystem;
    @Inject
    private TestDirectory directory;
    private final LogFileChannelNativeAccessor nativeChannelAccessor = (LogFileChannelNativeAccessor)Mockito.mock(LogFileChannelNativeAccessor.class);

    PhysicalFlushableChannelTest() {
    }

    @Test
    void shouldBeAbleToWriteSmallNumberOfBytes() throws IOException {
        byte[] bytes;
        File firstFile = new File(this.directory.homeDir(), "file1");
        StoreFileChannel storeChannel = this.fileSystem.write(firstFile);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, firstFile, (ChannelNativeAccessor)this.nativeChannelAccessor);
        int length = 26145;
        try (PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel);){
            bytes = PhysicalFlushableChannelTest.generateBytes(length);
            channel.put(bytes, length);
        }
        byte[] writtenBytes = new byte[length];
        try (FileInputStream in = new FileInputStream(firstFile);){
            ((InputStream)in).read(writtenBytes);
        }
        Assertions.assertArrayEquals((byte[])bytes, (byte[])writtenBytes);
    }

    @Test
    void shouldBeAbleToWriteValuesGreaterThanHalfTheBufferSize() throws IOException {
        byte[] bytes;
        File firstFile = new File(this.directory.homeDir(), "file1");
        StoreFileChannel storeChannel = this.fileSystem.write(firstFile);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, firstFile, (ChannelNativeAccessor)this.nativeChannelAccessor);
        int length = 262145;
        try (PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel);){
            bytes = PhysicalFlushableChannelTest.generateBytes(length);
            channel.put(bytes, length);
        }
        byte[] writtenBytes = new byte[length];
        try (FileInputStream in = new FileInputStream(firstFile);){
            ((InputStream)in).read(writtenBytes);
        }
        Assertions.assertArrayEquals((byte[])bytes, (byte[])writtenBytes);
    }

    @Test
    void shouldBeAbleToWriteValuesGreaterThanTheBufferSize() throws IOException {
        byte[] bytes;
        File firstFile = new File(this.directory.homeDir(), "file1");
        StoreFileChannel storeChannel = this.fileSystem.write(firstFile);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, firstFile, (ChannelNativeAccessor)this.nativeChannelAccessor);
        int length = 1000000;
        try (PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel);){
            bytes = PhysicalFlushableChannelTest.generateBytes(length);
            channel.put(bytes, length);
        }
        byte[] writtenBytes = new byte[length];
        try (FileInputStream in = new FileInputStream(firstFile);){
            ((InputStream)in).read(writtenBytes);
        }
        Assertions.assertArrayEquals((byte[])bytes, (byte[])writtenBytes);
    }

    private static byte[] generateBytes(int length) {
        Random random = new Random();
        char[] validCharacters = new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o'};
        byte[] bytes = new byte[length];
        for (int i = 0; i < length; ++i) {
            bytes[i] = (byte)validCharacters[random.nextInt(validCharacters.length)];
        }
        return bytes;
    }

    @Test
    void shouldWriteThroughRotation() throws Exception {
        File firstFile = new File(this.directory.homeDir(), "file1");
        File secondFile = new File(this.directory.homeDir(), "file2");
        StoreFileChannel storeChannel = this.fileSystem.write(firstFile);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, firstFile, (ChannelNativeAccessor)this.nativeChannelAccessor);
        ByteBuffer buf = ByteBuffer.allocate(100);
        PhysicalFlushableLogChannel channel = new PhysicalFlushableLogChannel((StoreChannel)versionedStoreChannel, buf);
        byte byteValue = 4;
        short shortValue = 10;
        int intValue = 3545;
        long longValue = 45849589L;
        float floatValue = 45849.332f;
        double doubleValue = 4.58493343E8;
        byte[] byteArrayValue = new byte[]{1, 4, 2, 5, 3, 6};
        channel.put(byteValue);
        channel.putShort(shortValue);
        channel.putInt(intValue);
        channel.putLong(longValue);
        channel.prepareForFlush().flush();
        channel.close();
        storeChannel = this.fileSystem.write(secondFile);
        channel.setChannel((StoreChannel)new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 2L, -1, secondFile, (ChannelNativeAccessor)this.nativeChannelAccessor));
        channel.putFloat(floatValue);
        channel.putDouble(doubleValue);
        channel.put(byteArrayValue, byteArrayValue.length);
        channel.close();
        ByteBuffer firstFileContents = this.readFile(firstFile);
        Assertions.assertEquals((byte)byteValue, (byte)firstFileContents.get());
        Assertions.assertEquals((short)shortValue, (short)firstFileContents.getShort());
        Assertions.assertEquals((int)intValue, (int)firstFileContents.getInt());
        Assertions.assertEquals((long)longValue, (long)firstFileContents.getLong());
        ByteBuffer secondFileContents = this.readFile(secondFile);
        Assertions.assertEquals((float)floatValue, (float)secondFileContents.getFloat(), (float)0.001f);
        Assertions.assertEquals((double)doubleValue, (double)secondFileContents.getDouble(), (double)0.001);
        byte[] readByteArray = new byte[byteArrayValue.length];
        secondFileContents.get(readByteArray);
        Assertions.assertArrayEquals((byte[])byteArrayValue, (byte[])readByteArray);
    }

    @Test
    void shouldSeeCorrectPositionEvenBeforeEmptyingDataIntoChannel() throws Exception {
        File file = new File(this.directory.homeDir(), "file");
        StoreFileChannel storeChannel = this.fileSystem.write(file);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, file, (ChannelNativeAccessor)this.nativeChannelAccessor);
        PositionAwarePhysicalFlushableChecksumChannel channel = new PositionAwarePhysicalFlushableChecksumChannel((LogVersionedStoreChannel)versionedStoreChannel, ByteBuffers.allocateDirect((int)1024));
        LogPositionMarker positionMarker = new LogPositionMarker();
        LogPosition initialPosition = channel.getCurrentPosition(positionMarker).newPosition();
        channel.putLong(67L);
        channel.putInt(1234);
        LogPosition positionAfterSomeData = channel.getCurrentPosition(positionMarker).newPosition();
        Assertions.assertEquals((long)12L, (long)(positionAfterSomeData.getByteOffset() - initialPosition.getByteOffset()));
        channel.close();
    }

    @Test
    void shouldThrowIllegalStateExceptionAfterClosed() throws Exception {
        File file = new File(this.directory.homeDir(), "file");
        StoreFileChannel storeChannel = this.fileSystem.write(file);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, file, (ChannelNativeAccessor)this.nativeChannelAccessor);
        PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel);
        channel.close();
        storeChannel.close();
        channel.put((byte)0);
        Assertions.assertThrows(IllegalStateException.class, () -> ((PhysicalFlushableChannel)channel).prepareForFlush());
    }

    @Test
    void shouldThrowClosedChannelExceptionWhenChannelUnexpectedlyClosed() throws Exception {
        File file = new File(this.directory.homeDir(), "file");
        StoreFileChannel storeChannel = this.fileSystem.write(file);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, file, (ChannelNativeAccessor)this.nativeChannelAccessor);
        PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel);
        storeChannel.close();
        channel.put((byte)0);
        Assertions.assertThrows(ClosedChannelException.class, () -> ((PhysicalFlushableChannel)channel).prepareForFlush());
    }

    private ByteBuffer readFile(File file) throws IOException {
        try (StoreFileChannel channel = this.fileSystem.read(file);){
            ByteBuffer buffer = ByteBuffers.allocate((int)((int)channel.size()));
            channel.readAll(buffer);
            buffer.flip();
            ByteBuffer byteBuffer = buffer;
            return byteBuffer;
        }
    }
}

