/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.dbms.database;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import org.assertj.core.api.Assertions;
import org.eclipse.collections.api.set.ImmutableSet;
import org.eclipse.collections.impl.factory.Sets;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;
import org.mockito.verification.VerificationMode;
import org.neo4j.dbms.database.DatabasePageCache;
import org.neo4j.io.pagecache.IOLimiter;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.io.pagecache.PagedFile;
import org.neo4j.io.pagecache.tracing.cursor.context.EmptyVersionContextSupplier;
import org.neo4j.io.pagecache.tracing.cursor.context.VersionContextSupplier;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.rule.TestDirectory;

@TestDirectoryExtension
class DatabasePageCacheTest {
    @Inject
    private TestDirectory testDirectory;
    private DatabasePageCache databasePageCache;
    private PageCache globalPageCache;
    private PagedFileAnswer pagedFileMapper;

    DatabasePageCacheTest() {
    }

    @BeforeEach
    void setUp() throws IOException {
        this.globalPageCache = (PageCache)Mockito.mock(PageCache.class);
        this.pagedFileMapper = new PagedFileAnswer();
        Mockito.when((Object)this.globalPageCache.map((Path)ArgumentMatchers.any(Path.class), (VersionContextSupplier)ArgumentMatchers.any(), ArgumentMatchers.eq((int)8192), (ImmutableSet)ArgumentMatchers.any(), (String)ArgumentMatchers.any())).then((Answer)this.pagedFileMapper);
        this.databasePageCache = new DatabasePageCache(this.globalPageCache, EmptyVersionContextSupplier.EMPTY, "test database");
    }

    @AfterEach
    void tearDown() {
        if (this.databasePageCache != null) {
            this.databasePageCache.close();
        }
    }

    @Test
    void mapDatabaseFile() throws IOException {
        Path mapFile = this.testDirectory.createFile("mapFile");
        PagedFile pagedFile = this.databasePageCache.map(mapFile, EmptyVersionContextSupplier.EMPTY, 8192, Sets.immutable.empty());
        org.junit.jupiter.api.Assertions.assertNotNull((Object)pagedFile);
        ((PageCache)Mockito.verify((Object)this.globalPageCache)).map(mapFile, EmptyVersionContextSupplier.EMPTY, 8192, Sets.immutable.empty(), "test database");
    }

    @Test
    void listExistingDatabaseMappings() throws IOException {
        Path mapFile1 = this.testDirectory.createFile("mapFile1");
        Path mapFile2 = this.testDirectory.createFile("mapFile2");
        PagedFile pagedFile = this.databasePageCache.map(mapFile1, 8192);
        PagedFile pagedFile2 = this.databasePageCache.map(mapFile2, 8192);
        List pagedFiles = this.databasePageCache.listExistingMappings();
        Assertions.assertThat((List)pagedFiles).hasSize(2);
        Assertions.assertThat((List)pagedFiles).contains((Object[])new PagedFile[]{pagedFile});
        Assertions.assertThat((List)pagedFiles).contains((Object[])new PagedFile[]{pagedFile2});
    }

    @Test
    void doNotIncludeNotDatabaseFilesInMappingsList() throws IOException {
        try (DatabasePageCache anotherDatabaseCache = new DatabasePageCache(this.globalPageCache, EmptyVersionContextSupplier.EMPTY, null);){
            Path mapFile1 = this.testDirectory.createFile("mapFile1");
            Path mapFile2 = this.testDirectory.createFile("mapFile2");
            Path mapFile3 = this.testDirectory.createFile("mapFile3");
            Path mapFile4 = this.testDirectory.createFile("mapFile4");
            PagedFile pagedFile = this.databasePageCache.map(mapFile1, 8192);
            PagedFile pagedFile2 = this.databasePageCache.map(mapFile2, 8192);
            PagedFile pagedFile3 = anotherDatabaseCache.map(mapFile3, 8192);
            PagedFile pagedFile4 = anotherDatabaseCache.map(mapFile4, 8192);
            List pagedFiles = this.databasePageCache.listExistingMappings();
            Assertions.assertThat((List)pagedFiles).hasSize(2);
            Assertions.assertThat((List)pagedFiles).contains((Object[])new PagedFile[]{pagedFile, pagedFile2});
            List anotherPagedFiles = anotherDatabaseCache.listExistingMappings();
            Assertions.assertThat((List)anotherPagedFiles).hasSize(2);
            Assertions.assertThat((List)anotherPagedFiles).contains((Object[])new PagedFile[]{pagedFile3, pagedFile4});
        }
    }

    @Test
    void existingMappingRestrictedToDatabaseMappedFiles() throws IOException {
        try (DatabasePageCache anotherDatabaseCache = new DatabasePageCache(this.globalPageCache, EmptyVersionContextSupplier.EMPTY, null);){
            Path mapFile1 = this.testDirectory.createFile("mapFile1");
            Path mapFile2 = this.testDirectory.createFile("mapFile2");
            Path mapFile3 = this.testDirectory.createFile("mapFile3");
            Path mapFile4 = this.testDirectory.createFile("mapFile4");
            this.databasePageCache.map(mapFile1, 8192);
            this.databasePageCache.map(mapFile2, 8192);
            anotherDatabaseCache.map(mapFile3, 8192);
            anotherDatabaseCache.map(mapFile4, 8192);
            org.junit.jupiter.api.Assertions.assertTrue((boolean)this.databasePageCache.getExistingMapping(mapFile1).isPresent());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)this.databasePageCache.getExistingMapping(mapFile2).isPresent());
            org.junit.jupiter.api.Assertions.assertFalse((boolean)this.databasePageCache.getExistingMapping(mapFile3).isPresent());
            org.junit.jupiter.api.Assertions.assertFalse((boolean)this.databasePageCache.getExistingMapping(mapFile4).isPresent());
            org.junit.jupiter.api.Assertions.assertFalse((boolean)anotherDatabaseCache.getExistingMapping(mapFile1).isPresent());
            org.junit.jupiter.api.Assertions.assertFalse((boolean)anotherDatabaseCache.getExistingMapping(mapFile2).isPresent());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)anotherDatabaseCache.getExistingMapping(mapFile3).isPresent());
            org.junit.jupiter.api.Assertions.assertTrue((boolean)anotherDatabaseCache.getExistingMapping(mapFile4).isPresent());
        }
    }

    @Test
    void throwOnMultipleCloseAttempts() {
        this.databasePageCache.close();
        org.junit.jupiter.api.Assertions.assertThrows(IllegalStateException.class, () -> this.databasePageCache.close());
        this.databasePageCache = null;
    }

    @Test
    void flushOnlyAffectsDatabaseRelatedFiles() throws IOException {
        try (DatabasePageCache anotherDatabaseCache = new DatabasePageCache(this.globalPageCache, EmptyVersionContextSupplier.EMPTY, null);){
            Path mapFile1 = this.testDirectory.createFile("mapFile1");
            Path mapFile2 = this.testDirectory.createFile("mapFile2");
            Path mapFile3 = this.testDirectory.createFile("mapFile3");
            Path mapFile4 = this.testDirectory.createFile("mapFile4");
            this.databasePageCache.map(mapFile1, 8192);
            this.databasePageCache.map(mapFile2, 8192);
            anotherDatabaseCache.map(mapFile3, 8192);
            anotherDatabaseCache.map(mapFile4, 8192);
            this.databasePageCache.flushAndForce();
            List<PagedFile> pagedFiles = this.pagedFileMapper.getPagedFiles();
            PagedFile originalPagedFile1 = DatabasePageCacheTest.findPagedFile(pagedFiles, mapFile1);
            PagedFile originalPagedFile2 = DatabasePageCacheTest.findPagedFile(pagedFiles, mapFile2);
            PagedFile originalPagedFile3 = DatabasePageCacheTest.findPagedFile(pagedFiles, mapFile3);
            PagedFile originalPagedFile4 = DatabasePageCacheTest.findPagedFile(pagedFiles, mapFile4);
            ((PagedFile)Mockito.verify((Object)originalPagedFile1)).flushAndForce();
            ((PagedFile)Mockito.verify((Object)originalPagedFile2)).flushAndForce();
            ((PagedFile)Mockito.verify((Object)originalPagedFile3, (VerificationMode)Mockito.never())).flushAndForce();
            ((PagedFile)Mockito.verify((Object)originalPagedFile4, (VerificationMode)Mockito.never())).flushAndForce();
        }
    }

    @Test
    void flushWithLimiterOnlyAffectsDatabaseRelatedFiles() throws IOException {
        try (DatabasePageCache anotherDatabaseCache = new DatabasePageCache(this.globalPageCache, EmptyVersionContextSupplier.EMPTY, null);){
            Path mapFile1 = this.testDirectory.createFile("mapFile1");
            Path mapFile2 = this.testDirectory.createFile("mapFile2");
            Path mapFile3 = this.testDirectory.createFile("mapFile3");
            Path mapFile4 = this.testDirectory.createFile("mapFile4");
            this.databasePageCache.map(mapFile1, 8192);
            this.databasePageCache.map(mapFile2, 8192);
            anotherDatabaseCache.map(mapFile3, 8192);
            anotherDatabaseCache.map(mapFile4, 8192);
            this.databasePageCache.flushAndForce(IOLimiter.UNLIMITED);
            List<PagedFile> pagedFiles = this.pagedFileMapper.getPagedFiles();
            PagedFile originalPagedFile1 = DatabasePageCacheTest.findPagedFile(pagedFiles, mapFile1);
            PagedFile originalPagedFile2 = DatabasePageCacheTest.findPagedFile(pagedFiles, mapFile2);
            PagedFile originalPagedFile3 = DatabasePageCacheTest.findPagedFile(pagedFiles, mapFile3);
            PagedFile originalPagedFile4 = DatabasePageCacheTest.findPagedFile(pagedFiles, mapFile4);
            ((PagedFile)Mockito.verify((Object)originalPagedFile1)).flushAndForce(IOLimiter.UNLIMITED);
            ((PagedFile)Mockito.verify((Object)originalPagedFile2)).flushAndForce(IOLimiter.UNLIMITED);
            ((PagedFile)Mockito.verify((Object)originalPagedFile3, (VerificationMode)Mockito.never())).flushAndForce(IOLimiter.UNLIMITED);
            ((PagedFile)Mockito.verify((Object)originalPagedFile4, (VerificationMode)Mockito.never())).flushAndForce(IOLimiter.UNLIMITED);
        }
    }

    @Test
    void closingFileCloseCacheMapping() throws IOException {
        Path mapFile1 = this.testDirectory.createFile("mapFile1");
        Path mapFile2 = this.testDirectory.createFile("mapFile2");
        PagedFile pagedFile1 = this.databasePageCache.map(mapFile1, 8192);
        PagedFile pagedFile2 = this.databasePageCache.map(mapFile2, 8192);
        org.junit.jupiter.api.Assertions.assertEquals((int)2, (int)this.databasePageCache.listExistingMappings().size());
        pagedFile1.close();
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)this.databasePageCache.listExistingMappings().size());
        pagedFile2.close();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)this.databasePageCache.listExistingMappings().isEmpty());
    }

    private static PagedFile findPagedFile(List<PagedFile> pagedFiles, Path mapFile) {
        return pagedFiles.stream().filter(pagedFile -> pagedFile.path().equals(mapFile)).findFirst().orElseThrow(() -> new IllegalStateException(String.format("Mapped paged file '%s' not found", mapFile.getFileName())));
    }

    private static class PagedFileAnswer
    implements Answer<PagedFile> {
        private final List<PagedFile> pagedFiles = new ArrayList<PagedFile>();

        private PagedFileAnswer() {
        }

        public PagedFile answer(InvocationOnMock invocation) {
            PagedFile pagedFile = (PagedFile)Mockito.mock(PagedFile.class);
            Mockito.when((Object)pagedFile.path()).thenReturn((Object)((Path)invocation.getArgument(0)));
            this.pagedFiles.add(pagedFile);
            return pagedFile;
        }

        List<PagedFile> getPagedFiles() {
            return this.pagedFiles;
        }
    }
}

