/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.locking.community;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;
import org.neo4j.kernel.DeadlockDetectedException;
import org.neo4j.kernel.impl.locking.community.LockResource;
import org.neo4j.kernel.impl.locking.community.LockTransaction;
import org.neo4j.kernel.impl.locking.community.RWLock;
import org.neo4j.kernel.impl.locking.community.RagManager;
import org.neo4j.lock.LockTracer;
import org.neo4j.lock.ResourceType;
import org.neo4j.lock.ResourceTypes;
import org.neo4j.time.Clocks;

class RWLockTest {
    private static ExecutorService executor;

    RWLockTest() {
    }

    @BeforeAll
    static void initExecutor() {
        executor = Executors.newCachedThreadPool();
    }

    @AfterAll
    static void stopExecutor() throws InterruptedException {
        executor.shutdown();
        executor.awaitTermination(2L, TimeUnit.SECONDS);
    }

    @Test
    void assertWriteLockDoesNotLeakMemory() {
        RagManager ragManager = new RagManager();
        LockResource resource = new LockResource((ResourceType)ResourceTypes.NODE, 10L);
        RWLock lock = RWLockTest.createRWLock(ragManager, resource);
        LockTransaction tx = (LockTransaction)Mockito.mock(LockTransaction.class);
        lock.mark();
        lock.acquireWriteLock(LockTracer.NONE, tx);
        lock.mark();
        Assertions.assertEquals((int)1, (int)lock.getTxLockElementCount());
        lock.releaseWriteLock(tx);
        Assertions.assertEquals((int)0, (int)lock.getTxLockElementCount());
    }

    @Test
    void assertReadLockDoesNotLeakMemory() {
        RagManager ragManager = new RagManager();
        LockResource resource = new LockResource((ResourceType)ResourceTypes.NODE, 10L);
        RWLock lock = RWLockTest.createRWLock(ragManager, resource);
        LockTransaction tx = (LockTransaction)Mockito.mock(LockTransaction.class);
        lock.mark();
        lock.acquireReadLock(LockTracer.NONE, tx);
        lock.mark();
        Assertions.assertEquals((int)1, (int)lock.getTxLockElementCount());
        lock.releaseReadLock(tx);
        Assertions.assertEquals((int)0, (int)lock.getTxLockElementCount());
    }

    @Test
    void testWaitingWriterLock() throws Exception {
        RagManager ragManager = new RagManager();
        LockResource resource = new LockResource((ResourceType)ResourceTypes.NODE, 10L);
        RWLock lock = RWLockTest.createRWLock(ragManager, resource);
        LockTransaction lockTransaction = new LockTransaction();
        LockTransaction anotherTransaction = new LockTransaction();
        lock.mark();
        lock.acquireReadLock(LockTracer.NONE, lockTransaction);
        lock.mark();
        lock.acquireReadLock(LockTracer.NONE, anotherTransaction);
        CountDownLatch writerCompletedLatch = new CountDownLatch(1);
        Runnable writer = RWLockTest.createWriter(lock, lockTransaction, writerCompletedLatch);
        executor.execute(writer);
        RWLockTest.waitWaitingThreads(lock, 1);
        Assertions.assertEquals((int)0, (int)lock.getWriteCount(), (String)"No writers for now.");
        Assertions.assertEquals((int)2, (int)lock.getReadCount());
        lock.releaseReadLock(lockTransaction);
        lock.releaseReadLock(anotherTransaction);
        writerCompletedLatch.await();
        Assertions.assertEquals((int)1, (int)lock.getWriteCount());
        Assertions.assertEquals((int)0, (int)lock.getReadCount());
        lock.releaseWriteLock(lockTransaction);
        Assertions.assertEquals((int)0, (int)lock.getWriteCount(), (String)"Lock should not have any writers left.");
        Assertions.assertEquals((int)0, (int)lock.getWaitingThreadsCount(), (String)"No waiting threads left.");
        Assertions.assertEquals((int)0, (int)lock.getTxLockElementCount(), (String)"No lock elements left.");
    }

    @Test
    void testWaitingReaderLock() throws Exception {
        RagManager ragManager = new RagManager();
        LockResource resource = new LockResource((ResourceType)ResourceTypes.NODE, 10L);
        RWLock lock = RWLockTest.createRWLock(ragManager, resource);
        LockTransaction transaction = new LockTransaction();
        LockTransaction readerTransaction = new LockTransaction();
        CountDownLatch readerCompletedLatch = new CountDownLatch(1);
        lock.mark();
        lock.acquireWriteLock(LockTracer.NONE, transaction);
        Runnable reader = RWLockTest.createReader(lock, readerTransaction, readerCompletedLatch);
        executor.execute(reader);
        RWLockTest.waitWaitingThreads(lock, 1);
        Assertions.assertEquals((int)1, (int)lock.getWriteCount());
        Assertions.assertEquals((int)0, (int)lock.getReadCount(), (String)"No readers for now");
        lock.releaseWriteLock(transaction);
        readerCompletedLatch.await();
        Assertions.assertEquals((int)0, (int)lock.getWriteCount());
        Assertions.assertEquals((int)1, (int)lock.getReadCount());
        lock.releaseReadLock(readerTransaction);
        Assertions.assertEquals((int)0, (int)lock.getReadCount(), (String)"Lock should not have any readers left.");
        Assertions.assertEquals((int)0, (int)lock.getWaitingThreadsCount(), (String)"No waiting threads left.");
        Assertions.assertEquals((int)0, (int)lock.getTxLockElementCount(), (String)"No lock elements left.");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testThreadRemovedFromWaitingListOnDeadlock() throws Exception {
        RagManager ragManager = (RagManager)Mockito.mock(RagManager.class);
        LockResource resource = new LockResource((ResourceType)ResourceTypes.NODE, 10L);
        RWLock lock = RWLockTest.createRWLock(ragManager, resource);
        LockTransaction lockTransaction = new LockTransaction();
        LockTransaction anotherTransaction = new LockTransaction();
        CountDownLatch exceptionLatch = new CountDownLatch(1);
        CountDownLatch completionLatch = new CountDownLatch(1);
        ((RagManager)Mockito.doNothing().doAnswer(invocation -> {
            exceptionLatch.countDown();
            throw new DeadlockDetectedException("Deadlock");
        }).when((Object)ragManager)).checkWaitOn((Object)lock, (Object)lockTransaction);
        lock.mark();
        lock.mark();
        lock.acquireReadLock(LockTracer.NONE, lockTransaction);
        lock.acquireReadLock(LockTracer.NONE, anotherTransaction);
        Runnable writer = () -> {
            try {
                lock.mark();
                lock.acquireWriteLock(LockTracer.NONE, lockTransaction);
            }
            catch (DeadlockDetectedException deadlockDetectedException) {
                // empty catch block
            }
            completionLatch.countDown();
        };
        executor.execute(writer);
        RWLockTest.waitWaitingThreads(lock, 1);
        do {
            RWLock rWLock = lock;
            synchronized (rWLock) {
                lock.notifyAll();
            }
        } while (exceptionLatch.getCount() == 1L);
        completionLatch.await();
        Assertions.assertEquals((int)0, (int)lock.getWaitingThreadsCount(), (String)"In case of deadlock caused by spurious wake up thread should be removed from waiting list");
    }

    @Test
    void testLockCounters() throws InterruptedException {
        RagManager ragManager = new RagManager();
        LockResource resource = new LockResource((ResourceType)ResourceTypes.NODE, 10L);
        RWLock lock = RWLockTest.createRWLock(ragManager, resource);
        LockTransaction lockTransaction = new LockTransaction();
        LockTransaction anotherTransaction = new LockTransaction();
        LockTransaction writeTransaction = new LockTransaction();
        CountDownLatch writerCompletedLatch = new CountDownLatch(1);
        lock.mark();
        lock.acquireReadLock(LockTracer.NONE, lockTransaction);
        lock.mark();
        lock.acquireReadLock(LockTracer.NONE, anotherTransaction);
        Assertions.assertEquals((int)2, (int)lock.getReadCount());
        Assertions.assertEquals((int)0, (int)lock.getWriteCount());
        Assertions.assertEquals((int)2, (int)lock.getTxLockElementCount());
        Runnable writer = RWLockTest.createWriter(lock, writeTransaction, writerCompletedLatch);
        executor.submit(writer);
        RWLockTest.waitWaitingThreads(lock, 1);
        Assertions.assertEquals((int)2, (int)lock.getReadCount());
        Assertions.assertEquals((int)0, (int)lock.getWriteCount());
        Assertions.assertEquals((int)3, (int)lock.getTxLockElementCount());
        Assertions.assertEquals((int)1, (int)lock.getWaitingThreadsCount());
        lock.releaseReadLock(lockTransaction);
        lock.releaseReadLock(anotherTransaction);
        writerCompletedLatch.await();
        Assertions.assertEquals((int)0, (int)lock.getReadCount());
        Assertions.assertEquals((int)1, (int)lock.getWriteCount());
        Assertions.assertEquals((int)1, (int)lock.getTxLockElementCount());
        Assertions.assertEquals((int)0, (int)lock.getWaitingThreadsCount());
        lock.releaseWriteLock(writeTransaction);
        Assertions.assertEquals((int)0, (int)lock.getTxLockElementCount());
        Assertions.assertEquals((int)0, (int)lock.getWaitingThreadsCount());
        Assertions.assertEquals((int)0, (int)lock.getReadCount());
        Assertions.assertEquals((int)0, (int)lock.getWriteCount());
    }

    @Test
    void testDeadlockDetection() throws Exception {
        RagManager ragManager = new RagManager();
        LockResource node1 = new LockResource((ResourceType)ResourceTypes.NODE, 10L);
        LockResource node2 = new LockResource((ResourceType)ResourceTypes.NODE, 10L);
        LockResource node3 = new LockResource((ResourceType)ResourceTypes.NODE, 10L);
        RWLock lockNode1 = RWLockTest.createRWLock(ragManager, node1);
        RWLock lockNode2 = RWLockTest.createRWLock(ragManager, node2);
        RWLock lockNode3 = RWLockTest.createRWLock(ragManager, node3);
        LockTransaction client1Transaction = new LockTransaction();
        LockTransaction client2Transaction = new LockTransaction();
        LockTransaction client3Transaction = new LockTransaction();
        CountDownLatch deadLockDetector = new CountDownLatch(1);
        lockNode1.mark();
        lockNode1.acquireWriteLock(LockTracer.NONE, client1Transaction);
        lockNode2.mark();
        lockNode2.acquireWriteLock(LockTracer.NONE, client2Transaction);
        lockNode3.mark();
        lockNode3.acquireWriteLock(LockTracer.NONE, client3Transaction);
        Runnable readerLockNode2 = RWLockTest.createReaderForDeadlock(lockNode3, client1Transaction, deadLockDetector);
        Runnable readerLockNode3 = RWLockTest.createReaderForDeadlock(lockNode1, client2Transaction, deadLockDetector);
        Runnable readerLockNode1 = RWLockTest.createReaderForDeadlock(lockNode2, client3Transaction, deadLockDetector);
        executor.execute(readerLockNode2);
        executor.execute(readerLockNode3);
        executor.execute(readerLockNode1);
        Assertions.assertTrue((boolean)deadLockDetector.await(100L, TimeUnit.SECONDS), (String)"Deadlock was detected as expected.");
        lockNode3.releaseWriteLock(client3Transaction);
        lockNode2.releaseWriteLock(client2Transaction);
        lockNode1.releaseWriteLock(client1Transaction);
    }

    @Test
    void testLockRequestsTermination() throws Exception {
        RagManager ragManager = new RagManager();
        LockResource node1 = new LockResource((ResourceType)ResourceTypes.NODE, 10L);
        RWLock lock = RWLockTest.createRWLock(ragManager, node1);
        LockTransaction mainTransaction = new LockTransaction();
        LockTransaction writerTransaction = new LockTransaction();
        CountDownLatch writerCompletedLatch = new CountDownLatch(1);
        Runnable conflictingWriter = RWLockTest.createFailedWriter(lock, writerTransaction, writerCompletedLatch);
        LockTransaction readerTransaction = new LockTransaction();
        CountDownLatch readerCompletedLatch = new CountDownLatch(1);
        Runnable reader = RWLockTest.createFailedReader(lock, readerTransaction, readerCompletedLatch);
        lock.mark();
        Assertions.assertTrue((boolean)lock.acquireWriteLock(LockTracer.NONE, mainTransaction));
        executor.submit(reader);
        executor.submit(conflictingWriter);
        RWLockTest.waitWaitingThreads(lock, 2);
        Assertions.assertEquals((int)3, (int)lock.getTxLockElementCount());
        lock.terminateLockRequestsForLockTransaction(readerTransaction);
        lock.terminateLockRequestsForLockTransaction(writerTransaction);
        readerCompletedLatch.await();
        writerCompletedLatch.await();
        Assertions.assertEquals((int)0, (int)lock.getWaitingThreadsCount());
        Assertions.assertEquals((int)0, (int)lock.getReadCount());
        Assertions.assertEquals((int)1, (int)lock.getWriteCount());
        Assertions.assertEquals((int)1, (int)lock.getTxLockElementCount());
    }

    private static Runnable createReader(RWLock lock, LockTransaction transaction, CountDownLatch latch) {
        return () -> {
            lock.mark();
            lock.acquireReadLock(LockTracer.NONE, transaction);
            latch.countDown();
        };
    }

    private static Runnable createFailedReader(RWLock lock, LockTransaction transaction, CountDownLatch latch) {
        return () -> {
            lock.mark();
            Assertions.assertFalse((boolean)lock.acquireReadLock(LockTracer.NONE, transaction));
            latch.countDown();
        };
    }

    private static Runnable createWriter(RWLock lock, LockTransaction transaction, CountDownLatch latch) {
        return () -> {
            lock.mark();
            lock.acquireWriteLock(LockTracer.NONE, transaction);
            latch.countDown();
        };
    }

    private static Runnable createFailedWriter(RWLock lock, LockTransaction transaction, CountDownLatch latch) {
        return () -> {
            lock.mark();
            Assertions.assertFalse((boolean)lock.acquireWriteLock(LockTracer.NONE, transaction));
            latch.countDown();
        };
    }

    private static Runnable createReaderForDeadlock(RWLock node, LockTransaction transaction, CountDownLatch latch) {
        return () -> {
            try {
                node.mark();
                node.acquireReadLock(LockTracer.NONE, transaction);
            }
            catch (DeadlockDetectedException e) {
                latch.countDown();
            }
        };
    }

    private static RWLock createRWLock(RagManager ragManager, LockResource resource) {
        return new RWLock(resource, ragManager, Clocks.nanoClock(), 0L);
    }

    private static void waitWaitingThreads(RWLock lock, int expectedThreads) throws InterruptedException {
        while (lock.getWaitingThreadsCount() != expectedThreads) {
            Thread.sleep(20L);
        }
    }
}

