/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.util;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.TreeSet;
import java.util.stream.Stream;
import org.apache.commons.lang3.SystemUtils;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.neo4j.io.fs.FileUtils;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.rule.TestDirectory;

@TestDirectoryExtension
class FileUtilsTest {
    @Inject
    private TestDirectory testDirectory;
    private Path path;

    FileUtilsTest() {
    }

    @BeforeEach
    void beforeEach() {
        this.path = this.testDirectory.directory("path");
    }

    @Test
    void moveFileToDirectory() throws Exception {
        Path file = this.touchFile("source");
        Path targetDir = this.directory("dir");
        Path newLocationOfFile = FileUtils.moveFileToDirectory((Path)file, (Path)targetDir);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Files.exists(newLocationOfFile, new LinkOption[0]));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)Files.exists(file, new LinkOption[0]));
        Path[] files = FileUtils.listPaths((Path)targetDir);
        org.junit.jupiter.api.Assertions.assertNotNull((Object)files);
        org.junit.jupiter.api.Assertions.assertEquals((Object)newLocationOfFile, (Object)files[0]);
    }

    @Test
    void moveFileToDirectoryCreatesNonExistingDirectory() throws Exception {
        Path file = this.touchFile("source");
        Path targetDir = this.path.resolve("nonexisting");
        Path newLocationOfFile = FileUtils.moveFileToDirectory((Path)file, (Path)targetDir);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Files.exists(newLocationOfFile, new LinkOption[0]));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)Files.exists(file, new LinkOption[0]));
        Path[] files = FileUtils.listPaths((Path)targetDir);
        org.junit.jupiter.api.Assertions.assertNotNull((Object)files);
        org.junit.jupiter.api.Assertions.assertEquals((Object)newLocationOfFile, (Object)files[0]);
    }

    @Test
    void moveFile() throws Exception {
        Path file = this.touchFile("source");
        Path targetDir = this.directory("dir");
        Path newLocationOfFile = targetDir.resolve("new-name");
        FileUtils.moveFile((Path)file, (Path)newLocationOfFile);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Files.exists(newLocationOfFile, new LinkOption[0]));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)Files.exists(file, new LinkOption[0]));
        Path[] files = FileUtils.listPaths((Path)targetDir);
        org.junit.jupiter.api.Assertions.assertNotNull((Object)files);
        org.junit.jupiter.api.Assertions.assertEquals((Object)newLocationOfFile, (Object)files[0]);
    }

    @Test
    void deletePathRecursively() throws IOException {
        Path root = this.testDirectory.directory("a");
        Path child = root.resolve("b");
        Path file = child.resolve("c");
        Files.createDirectories(child, new FileAttribute[0]);
        Files.createFile(file, new FileAttribute[0]);
        FileUtils.deleteDirectory((Path)root);
        org.junit.jupiter.api.Assertions.assertFalse((boolean)Files.exists(file, new LinkOption[0]));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)Files.exists(child, new LinkOption[0]));
    }

    @Test
    void deletePathRecursivelyWithFilter() throws IOException {
        Path root = this.testDirectory.directory("a");
        Path child = root.resolve("b");
        Path file = child.resolve("c");
        Path toKeepDir = root.resolve("d");
        Path toKeepFile = toKeepDir.resolve("e");
        Files.createDirectories(child, new FileAttribute[0]);
        Files.createFile(file, new FileAttribute[0]);
        Files.createDirectories(toKeepDir, new FileAttribute[0]);
        Files.createFile(toKeepFile, new FileAttribute[0]);
        FileUtils.deleteDirectory((Path)root, path -> !path.equals(toKeepFile));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)Files.exists(file, new LinkOption[0]));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)Files.exists(child, new LinkOption[0]));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Files.exists(toKeepFile, new LinkOption[0]));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Files.exists(toKeepDir, new LinkOption[0]));
    }

    @Test
    void deleteNestedPathRecursivelyWithFilter() throws IOException {
        Path root = this.testDirectory.directory("a");
        Path child = root.resolve("a");
        Path file = child.resolve("aaFile");
        Path toKeepDelete = root.resolve("b");
        Files.createDirectories(child, new FileAttribute[0]);
        Files.createFile(file, new FileAttribute[0]);
        Files.createDirectories(toKeepDelete, new FileAttribute[0]);
        FileUtils.deleteDirectory((Path)root, path -> !path.equals(file));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Files.exists(file, new LinkOption[0]));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Files.exists(child, new LinkOption[0]));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)Files.exists(toKeepDelete, new LinkOption[0]));
    }

    @Test
    void pathToFileAfterMoveMustThrowIfFileNotSubPathToFromShorter() {
        Path file = Path.of("/a", new String[0]);
        Path from = Path.of("/a/b", new String[0]);
        Path to = Path.of("/a/c", new String[0]);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file));
    }

    @Test
    void pathToFileAfterMoveMustThrowIfFileNotSubPathToFromSameLength() {
        Path file = Path.of("/a/f", new String[0]);
        Path from = Path.of("/a/b", new String[0]);
        Path to = Path.of("/a/c", new String[0]);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file));
    }

    @Test
    void pathToFileAfterMoveMustThrowIfFileNotSubPathToFromLonger() {
        Path file = Path.of("/a/c/f", new String[0]);
        Path from = Path.of("/a/b", new String[0]);
        Path to = Path.of("/a/c", new String[0]);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file));
    }

    @Test
    void pathToFileAfterMoveMustThrowIfFromDirIsCompletePathToFile() {
        Path file = Path.of("/a/b/f", new String[0]);
        Path from = Path.of("/a/b/f", new String[0]);
        Path to = Path.of("/a/c", new String[0]);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalArgumentException.class, () -> FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file));
    }

    @Test
    void pathToFileAfterMoveMustWorkIfMovingToSibling() {
        Path file = Path.of("/a/b/f", new String[0]);
        Path from = Path.of("/a/b", new String[0]);
        Path to = Path.of("/a/c", new String[0]);
        Assertions.assertThat((Path)FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file)).isEqualTo((Object)this.path("/a/c/f"));
    }

    @Test
    void pathToFileAfterMoveMustWorkIfMovingToSiblingAndFileHasSubDir() {
        Path file = Path.of("/a/b/d/f", new String[0]);
        Path from = Path.of("/a/b", new String[0]);
        Path to = Path.of("/a/c", new String[0]);
        Assertions.assertThat((Path)FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file)).isEqualTo((Object)this.path("/a/c/d/f"));
    }

    @Test
    void pathToFileAfterMoveMustWorkIfMovingToSubDir() {
        Path file = Path.of("/a/b/f", new String[0]);
        Path from = Path.of("/a/b", new String[0]);
        Path to = Path.of("/a/b/c", new String[0]);
        Assertions.assertThat((Path)FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file)).isEqualTo((Object)this.path("/a/b/c/f"));
    }

    @Test
    void pathToFileAfterMoveMustWorkIfMovingToSubDirAndFileHasSubDir() {
        Path file = Path.of("/a/b/d/f", new String[0]);
        Path from = Path.of("/a/b", new String[0]);
        Path to = Path.of("/a/b/c", new String[0]);
        Assertions.assertThat((Path)FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file)).isEqualTo((Object)this.path("/a/b/c/d/f"));
    }

    @Test
    void pathToFileAfterMoveMustWorkIfMovingOutOfDir() {
        Path file = Path.of("/a/b/f", new String[0]);
        Path from = Path.of("/a/b", new String[0]);
        Path to = Path.of("/c", new String[0]);
        Assertions.assertThat((Path)FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file)).isEqualTo((Object)this.path("/c/f"));
    }

    @Test
    void pathToFileAfterMoveMustWorkIfMovingOutOfDirAndFileHasSubDir() {
        Path file = Path.of("/a/b/d/f", new String[0]);
        Path from = Path.of("/a/b", new String[0]);
        Path to = Path.of("/c", new String[0]);
        Assertions.assertThat((Path)FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file)).isEqualTo((Object)this.path("/c/d/f"));
    }

    @Test
    void pathToFileAfterMoveMustWorkIfNotMovingAtAll() {
        Path file = Path.of("/a/b/f", new String[0]);
        Path from = Path.of("/a/b", new String[0]);
        Path to = Path.of("/a/b", new String[0]);
        Assertions.assertThat((Path)FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file)).isEqualTo((Object)this.path("/a/b/f"));
    }

    @Test
    void pathToFileAfterMoveMustWorkIfNotMovingAtAllAndFileHasSubDir() {
        Path file = Path.of("/a/b/d/f", new String[0]);
        Path from = Path.of("/a/b", new String[0]);
        Path to = Path.of("/a/b", new String[0]);
        Assertions.assertThat((Path)FileUtils.pathToFileAfterMove((Path)from, (Path)to, (Path)file)).isEqualTo((Object)this.path("/a/b/d/f"));
    }

    @Test
    void allMacsHaveHighIO() {
        Assumptions.assumeTrue((boolean)SystemUtils.IS_OS_MAC);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)FileUtils.highIODevice((Path)Paths.get(".", new String[0])));
    }

    @Test
    void allWindowsHaveHighIO() {
        Assumptions.assumeTrue((boolean)SystemUtils.IS_OS_WINDOWS);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)FileUtils.highIODevice((Path)Paths.get(".", new String[0])));
    }

    @Test
    void onLinuxDevShmHasHighIO() {
        Assumptions.assumeTrue((boolean)SystemUtils.IS_OS_LINUX);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)FileUtils.highIODevice((Path)Paths.get("/dev/shm", new String[0])));
    }

    @Test
    void mustCountDirectoryContents() throws Exception {
        Path dir = this.directory("dir");
        Path file = dir.resolve("file");
        Path subdir = dir.resolve("subdir");
        Files.createFile(file, new FileAttribute[0]);
        Files.createDirectories(subdir, new FileAttribute[0]);
        Assertions.assertThat((long)FileUtils.countFilesInDirectoryPath((Path)dir)).isEqualTo(2L);
    }

    @Test
    void nonExistingDirectoryCanBeDeleted() throws IOException {
        Path dir = this.path.resolve("dir");
        FileUtils.deleteFile((Path)dir);
    }

    @Test
    void emptyDirectoryCanBeDeleted() throws Exception {
        Path dir = this.directory("dir");
        FileUtils.deleteFile((Path)dir);
    }

    @Test
    void nonEmptyDirectoryCannotBeDeleted() throws Exception {
        Path dir = this.directory("dir");
        Path file = dir.resolve("file");
        Files.createFile(file, new FileAttribute[0]);
        org.junit.jupiter.api.Assertions.assertThrows(DirectoryNotEmptyException.class, () -> FileUtils.deleteFile((Path)dir));
    }

    @Test
    void copySubTree() throws IOException {
        Path dir = Files.createTempDirectory("dir", new FileAttribute[0]);
        Files.writeString(dir.resolve("file1"), (CharSequence)"file1", StandardCharsets.UTF_8, new OpenOption[0]);
        Files.createDirectory(dir.resolve("sub1"), new FileAttribute[0]);
        Path sub2 = dir.resolve("sub2");
        Files.createDirectory(sub2, new FileAttribute[0]);
        Files.writeString(sub2.resolve("file2"), (CharSequence)"file2", StandardCharsets.UTF_8, new OpenOption[0]);
        FileUtils.copyDirectory((Path)dir, (Path)dir.resolve("sub2"));
        TreeSet structure = new TreeSet();
        try (Stream<Path> walk = Files.walk(dir, new FileVisitOption[0]);){
            walk.forEach(path -> structure.add(dir.relativize((Path)path)));
        }
        Assertions.assertThat(structure).containsExactly((Object[])new Path[]{Path.of("", new String[0]), Path.of("file1", new String[0]), Path.of("sub1", new String[0]), Path.of("sub2", new String[0]), Path.of("sub2/file1", new String[0]), Path.of("sub2/file2", new String[0]), Path.of("sub2/sub1", new String[0]), Path.of("sub2/sub2", new String[0]), Path.of("sub2/sub2/file2", new String[0])});
    }

    @Test
    void copyWithFilter() throws IOException {
        Path source = Files.createTempDirectory("source", new FileAttribute[0]);
        Files.writeString(source.resolve("file1"), (CharSequence)"file1", StandardCharsets.UTF_8, new OpenOption[0]);
        Files.writeString(source.resolve("file2"), (CharSequence)"file2", StandardCharsets.UTF_8, new OpenOption[0]);
        Files.writeString(source.resolve("file3"), (CharSequence)"file3", StandardCharsets.UTF_8, new OpenOption[0]);
        Files.writeString(source.resolve("file14"), (CharSequence)"file14", StandardCharsets.UTF_8, new OpenOption[0]);
        Path target = Files.createTempDirectory("target", new FileAttribute[0]);
        FileUtils.copyDirectory((Path)source, (Path)target, path -> path.getFileName().toString().startsWith("file1"));
        TreeSet structure = new TreeSet();
        try (Stream<Path> walk = Files.walk(target, new FileVisitOption[0]);){
            walk.forEach(path -> structure.add(target.relativize((Path)path)));
        }
        Assertions.assertThat(structure).containsExactly((Object[])new Path[]{Path.of("", new String[0]), Path.of("file1", new String[0]), Path.of("file14", new String[0])});
    }

    @Test
    void copyWithFilterInSubTree() throws IOException {
        Path dir = Files.createTempDirectory("dir", new FileAttribute[0]);
        Files.writeString(dir.resolve("file1"), (CharSequence)"file1", StandardCharsets.UTF_8, new OpenOption[0]);
        Files.createDirectory(dir.resolve("sub1"), new FileAttribute[0]);
        Path sub2 = dir.resolve("sub2");
        Files.createDirectory(sub2, new FileAttribute[0]);
        Files.writeString(sub2.resolve("file2"), (CharSequence)"file2", StandardCharsets.UTF_8, new OpenOption[0]);
        FileUtils.copyDirectory((Path)dir, (Path)dir.resolve("sub2"), path -> Files.isDirectory(path, new LinkOption[0]) || path.getFileName().toString().startsWith("file1"));
        TreeSet structure = new TreeSet();
        try (Stream<Path> walk = Files.walk(dir, new FileVisitOption[0]);){
            walk.forEach(path -> structure.add(dir.relativize((Path)path)));
        }
        Assertions.assertThat(structure).containsExactly((Object[])new Path[]{Path.of("", new String[0]), Path.of("file1", new String[0]), Path.of("sub1", new String[0]), Path.of("sub2", new String[0]), Path.of("sub2/file1", new String[0]), Path.of("sub2/file2", new String[0]), Path.of("sub2/sub1", new String[0]), Path.of("sub2/sub2", new String[0])});
    }

    private Path directory(String name) throws IOException {
        Path dir = this.path.resolve(name);
        Files.createDirectories(dir, new FileAttribute[0]);
        return dir;
    }

    private Path touchFile(String name) throws IOException {
        Path file = this.path.resolve(name);
        Files.createFile(file, new FileAttribute[0]);
        return file;
    }

    private Path path(String path) {
        return Path.of(path, new String[0]);
    }
}

