/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.recovery;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.Instant;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.FileSystemUtils;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.kernel.impl.transaction.SimpleLogVersionRepository;
import org.neo4j.kernel.impl.transaction.SimpleTransactionIdStore;
import org.neo4j.kernel.impl.transaction.log.FlushablePositionAwareChecksumChannel;
import org.neo4j.kernel.impl.transaction.log.LogPosition;
import org.neo4j.kernel.impl.transaction.log.TestLogEntryReader;
import org.neo4j.kernel.impl.transaction.log.files.LogFile;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.LogFilesBuilder;
import org.neo4j.kernel.impl.transaction.log.files.checkpoint.CheckpointFile;
import org.neo4j.kernel.impl.transaction.tracing.LogCheckPointEvent;
import org.neo4j.kernel.lifecycle.LifeSupport;
import org.neo4j.kernel.lifecycle.Lifecycle;
import org.neo4j.kernel.recovery.CorruptedLogsTruncator;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.storageengine.api.LogVersionRepository;
import org.neo4j.storageengine.api.StoreId;
import org.neo4j.storageengine.api.TransactionIdStore;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.rule.TestDirectory;

@TestDirectoryExtension
class CorruptedLogsTruncatorTest {
    private static final long SINGLE_LOG_FILE_SIZE = 73L;
    private static final int TOTAL_NUMBER_OF_TRANSACTION_LOG_FILES = 12;
    private static final int TOTAL_NUMBER_OF_LOG_FILES = 13;
    @Inject
    private FileSystemAbstraction fs;
    @Inject
    private TestDirectory testDirectory;
    private final LifeSupport life = new LifeSupport();
    private Path databaseDirectory;
    private LogFiles logFiles;
    private CorruptedLogsTruncator logPruner;
    private SimpleLogVersionRepository logVersionRepository;

    CorruptedLogsTruncatorTest() {
    }

    @BeforeEach
    void setUp() throws Exception {
        this.databaseDirectory = this.testDirectory.homePath();
        this.logVersionRepository = new SimpleLogVersionRepository();
        SimpleTransactionIdStore transactionIdStore = new SimpleTransactionIdStore();
        this.logFiles = LogFilesBuilder.logFilesBasedOnlyBuilder((Path)this.databaseDirectory, (FileSystemAbstraction)this.fs).withRotationThreshold(73L).withLogVersionRepository((LogVersionRepository)this.logVersionRepository).withTransactionIdStore((TransactionIdStore)transactionIdStore).withLogEntryReader(TestLogEntryReader.logEntryReader()).withStoreId(StoreId.UNKNOWN).withConfig(Config.newBuilder().set(GraphDatabaseInternalSettings.checkpoint_logical_log_rotation_threshold, (Object)1024L).build()).build();
        this.life.add((Lifecycle)this.logFiles);
        this.logPruner = new CorruptedLogsTruncator(this.databaseDirectory, this.logFiles, this.fs, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
    }

    @AfterEach
    void tearDown() {
        this.life.shutdown();
    }

    @Test
    void doNotPruneEmptyLogs() throws IOException {
        this.logPruner.truncate(new LogPosition(0L, 64L));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)FileSystemUtils.isEmptyOrNonExistingDirectory((FileSystemAbstraction)this.fs, (Path)this.databaseDirectory));
    }

    @Test
    void doNotPruneNonCorruptedLogs() throws IOException {
        this.life.start();
        CorruptedLogsTruncatorTest.generateTransactionLogFiles(this.logFiles);
        LogFile logFile = this.logFiles.getLogFile();
        long highestLogVersion = logFile.getHighestLogVersion();
        long fileSizeBeforePrune = Files.size(logFile.getHighestLogFile());
        LogPosition endOfLogsPosition = new LogPosition(highestLogVersion, fileSizeBeforePrune);
        org.junit.jupiter.api.Assertions.assertEquals((long)11L, (long)highestLogVersion);
        this.logPruner.truncate(endOfLogsPosition);
        org.junit.jupiter.api.Assertions.assertEquals((int)13, (int)this.logFiles.logFiles().length);
        org.junit.jupiter.api.Assertions.assertEquals((long)fileSizeBeforePrune, (long)Files.size(logFile.getHighestLogFile()));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)ArrayUtils.isEmpty((Object[])this.databaseDirectory.toFile().listFiles(File::isDirectory)));
    }

    @Test
    void pruneAndArchiveLastLog() throws IOException {
        this.life.start();
        CorruptedLogsTruncatorTest.generateTransactionLogFiles(this.logFiles);
        LogFile logFile = this.logFiles.getLogFile();
        long highestLogVersion = logFile.getHighestLogVersion();
        Path highestLogFile = logFile.getHighestLogFile();
        long fileSizeBeforePrune = Files.size(highestLogFile);
        int bytesToPrune = 5;
        long byteOffset = fileSizeBeforePrune - (long)bytesToPrune;
        LogPosition prunePosition = new LogPosition(highestLogVersion, byteOffset);
        this.logPruner.truncate(prunePosition);
        org.junit.jupiter.api.Assertions.assertEquals((int)13, (int)this.logFiles.logFiles().length);
        org.junit.jupiter.api.Assertions.assertEquals((long)byteOffset, (long)Files.size(highestLogFile));
        Path corruptedLogsDirectory = this.databaseDirectory.resolve("corrupted-neostore.transaction.db");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Files.exists(corruptedLogsDirectory, new LinkOption[0]));
        File[] files = corruptedLogsDirectory.toFile().listFiles();
        org.junit.jupiter.api.Assertions.assertNotNull((Object)files);
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)files.length);
        File corruptedLogsArchive = files[0];
        CorruptedLogsTruncatorTest.checkArchiveName(highestLogVersion, byteOffset, corruptedLogsArchive);
        try (ZipFile zipFile = new ZipFile(corruptedLogsArchive);){
            org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)zipFile.size());
            CorruptedLogsTruncatorTest.checkEntryNameAndSize(zipFile, highestLogFile.getFileName().toString(), bytesToPrune);
        }
    }

    @Test
    void pruneAndArchiveMultipleLogs() throws IOException {
        this.life.start();
        CorruptedLogsTruncatorTest.generateTransactionLogFiles(this.logFiles);
        long highestCorrectLogFileIndex = 5L;
        LogFile logFile = this.logFiles.getLogFile();
        Path highestCorrectLogFile = logFile.getLogFileForVersion(highestCorrectLogFileIndex);
        long fileSizeBeforePrune = Files.size(highestCorrectLogFile);
        long highestLogFileLength = Files.size(logFile.getHighestLogFile());
        int bytesToPrune = 7;
        long byteOffset = fileSizeBeforePrune - (long)bytesToPrune;
        LogPosition prunePosition = new LogPosition(highestCorrectLogFileIndex, byteOffset);
        CheckpointFile checkpointFile = this.logFiles.getCheckpointFile();
        checkpointFile.getCheckpointAppender().checkPoint(LogCheckPointEvent.NULL, new LogPosition(highestCorrectLogFileIndex, byteOffset - 1L), Instant.now(), "within okay transactions");
        for (int i = 0; i < 5; ++i) {
            checkpointFile.getCheckpointAppender().checkPoint(LogCheckPointEvent.NULL, new LogPosition(highestCorrectLogFileIndex, byteOffset + 1L), Instant.now(), "in the part being truncated");
        }
        this.life.shutdown();
        this.logPruner.truncate(prunePosition);
        this.life.start();
        this.logVersionRepository.setCheckpointLogVersion(0L, CursorContext.NULL);
        org.junit.jupiter.api.Assertions.assertEquals((int)7, (int)this.logFiles.logFiles().length);
        org.junit.jupiter.api.Assertions.assertEquals((long)byteOffset, (long)Files.size(highestCorrectLogFile));
        Assertions.assertThat((Object[])checkpointFile.getDetachedCheckpointFiles()).hasSize(1);
        org.junit.jupiter.api.Assertions.assertEquals((long)256L, (long)Files.size(checkpointFile.getDetachedCheckpointFiles()[0]));
        Path corruptedLogsDirectory = this.databaseDirectory.resolve("corrupted-neostore.transaction.db");
        org.junit.jupiter.api.Assertions.assertTrue((boolean)Files.exists(corruptedLogsDirectory, new LinkOption[0]));
        File[] files = corruptedLogsDirectory.toFile().listFiles();
        org.junit.jupiter.api.Assertions.assertNotNull((Object)files);
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)files.length);
        File corruptedLogsArchive = files[0];
        CorruptedLogsTruncatorTest.checkArchiveName(highestCorrectLogFileIndex, byteOffset, corruptedLogsArchive);
        try (ZipFile zipFile = new ZipFile(corruptedLogsArchive);){
            org.junit.jupiter.api.Assertions.assertEquals((int)9, (int)zipFile.size());
            CorruptedLogsTruncatorTest.checkEntryNameAndSize(zipFile, highestCorrectLogFile.getFileName().toString(), bytesToPrune);
            long nextLogFileIndex = highestCorrectLogFileIndex + 1L;
            int lastFileIndex = 11;
            for (long index = nextLogFileIndex; index < (long)lastFileIndex; ++index) {
                CorruptedLogsTruncatorTest.checkEntryNameAndSize(zipFile, "neostore.transaction.db." + index, 73L);
            }
            CorruptedLogsTruncatorTest.checkEntryNameAndSize(zipFile, "neostore.transaction.db." + lastFileIndex, highestLogFileLength);
            CorruptedLogsTruncatorTest.checkEntryNameAndSize(zipFile, "checkpoint.0", 768L);
            CorruptedLogsTruncatorTest.checkEntryNameAndSize(zipFile, "checkpoint.1", 256L);
        }
    }

    private static void checkEntryNameAndSize(ZipFile zipFile, String entryName, long expectedSize) throws IOException {
        ZipEntry entry = zipFile.getEntry(entryName);
        InputStream inputStream = zipFile.getInputStream(entry);
        int entryBytes = 0;
        while (inputStream.read() >= 0) {
            ++entryBytes;
        }
        org.junit.jupiter.api.Assertions.assertEquals((long)expectedSize, (long)entryBytes);
    }

    private static void checkArchiveName(long highestLogVersion, long byteOffset, File corruptedLogsArchive) {
        String name = corruptedLogsArchive.getName();
        org.junit.jupiter.api.Assertions.assertTrue((boolean)name.startsWith("corrupted-neostore.transaction.db-" + highestLogVersion + "-" + byteOffset));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)FilenameUtils.isExtension((String)name, (String)"zip"));
    }

    private static void generateTransactionLogFiles(LogFiles logFiles) throws IOException {
        LogFile logFile = logFiles.getLogFile();
        FlushablePositionAwareChecksumChannel writer = logFile.getTransactionLogWriter().getChannel();
        for (byte i = 0; i < 107; i = (byte)(i + 1)) {
            writer.put(i);
            writer.prepareForFlush();
            if (!logFile.rotationNeeded()) continue;
            logFile.rotate();
        }
    }
}

