/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.index.schema;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetTime;
import java.time.Period;
import java.time.ZonedDateTime;
import java.time.temporal.TemporalAmount;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.function.Supplier;
import java.util.stream.Stream;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.MethodSource;
import org.neo4j.io.pagecache.ByteArrayPageCursor;
import org.neo4j.io.pagecache.PageCursor;
import org.neo4j.kernel.impl.index.schema.GenericKey;
import org.neo4j.kernel.impl.index.schema.NativeIndexKey;
import org.neo4j.string.UTF8;
import org.neo4j.test.RandomSupport;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.RandomExtension;
import org.neo4j.values.storable.ArrayValue;
import org.neo4j.values.storable.ByteArray;
import org.neo4j.values.storable.ByteValue;
import org.neo4j.values.storable.DateTimeValue;
import org.neo4j.values.storable.DateValue;
import org.neo4j.values.storable.DoubleArray;
import org.neo4j.values.storable.DoubleValue;
import org.neo4j.values.storable.DurationValue;
import org.neo4j.values.storable.FloatArray;
import org.neo4j.values.storable.FloatValue;
import org.neo4j.values.storable.IntArray;
import org.neo4j.values.storable.IntValue;
import org.neo4j.values.storable.LocalDateTimeValue;
import org.neo4j.values.storable.LocalTimeValue;
import org.neo4j.values.storable.LongArray;
import org.neo4j.values.storable.LongValue;
import org.neo4j.values.storable.PointArray;
import org.neo4j.values.storable.PointValue;
import org.neo4j.values.storable.RandomValues;
import org.neo4j.values.storable.ShortArray;
import org.neo4j.values.storable.ShortValue;
import org.neo4j.values.storable.TextArray;
import org.neo4j.values.storable.TextValue;
import org.neo4j.values.storable.TimeValue;
import org.neo4j.values.storable.Value;
import org.neo4j.values.storable.ValueGroup;
import org.neo4j.values.storable.Values;

@ExtendWith(value={RandomExtension.class})
@TestInstance(value=TestInstance.Lifecycle.PER_CLASS)
abstract class IndexKeyStateTest<KEY extends GenericKey<KEY>> {
    @Inject
    RandomSupport random;

    IndexKeyStateTest() {
    }

    @BeforeEach
    void setupRandomConfig() {
        this.random = this.random.withConfiguration(new RandomValues.Configuration(){

            public int stringMinLength() {
                return 0;
            }

            public int stringMaxLength() {
                return 50;
            }

            public int arrayMinLength() {
                return 0;
            }

            public int arrayMaxLength() {
                return 10;
            }

            public int maxCodePoint() {
                return 65535;
            }

            public int minCodePoint() {
                return 0;
            }
        });
        this.random.reset();
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void readWhatIsWritten(ValueGenerator valueGenerator) {
        PageCursor cursor = IndexKeyStateTest.newPageCursor();
        KEY writeState = this.newKeyState();
        Value value = valueGenerator.next();
        int offset = cursor.getOffset();
        writeState.writeValue(value, NativeIndexKey.Inclusion.NEUTRAL);
        writeState.put(cursor);
        KEY readState = this.newKeyState();
        int size = writeState.size();
        cursor.setOffset(offset);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)readState.get(cursor, size), (String)"failed to read");
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)readState.compareValueTo(writeState), (String)"key states are not equal");
        Value readValue = readState.asValue();
        org.junit.jupiter.api.Assertions.assertEquals((Object)value, (Object)readValue, (String)"deserialized values are not equal");
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void readWhatIsWrittenCompositeKey(ValueGenerator valueGenerator) {
        int nbrOfSlots = this.random.nextInt(2, 5);
        PageCursor cursor = IndexKeyStateTest.newPageCursor();
        Layout<KEY> layout = this.newLayout(nbrOfSlots);
        KEY writeState = layout.newKey();
        int offset = cursor.getOffset();
        Value[] writtenValues = this.generateValuesForCompositeKey(nbrOfSlots, valueGenerator);
        for (int slot = 0; slot < nbrOfSlots; ++slot) {
            writeState.writeValue(slot, writtenValues[slot], NativeIndexKey.Inclusion.NEUTRAL);
        }
        writeState.put(cursor);
        KEY readState = layout.newKey();
        int size = writeState.size();
        cursor.setOffset(offset);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)readState.get(cursor, size), (String)"failed to read");
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)readState.compareValueTo(writeState), (String)"key states are not equal");
        Object[] readValues = readState.asValues();
        Assertions.assertThat((Object[])readValues).isEqualTo((Object)writtenValues);
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void copyShouldCopy(ValueGenerator valueGenerator) {
        KEY from = this.newKeyState();
        Value value = valueGenerator.next();
        from.writeValue(value, NativeIndexKey.Inclusion.NEUTRAL);
        KEY to = this.genericKeyStateWithSomePreviousState(valueGenerator);
        to.copyFrom(from);
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)from.compareValueTo(to), (String)"states not equals after copy");
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void copyShouldCopyCompositeKey(ValueGenerator valueGenerator) {
        int nbrOfSlots = this.random.nextInt(2, 5);
        Layout<KEY> layout = this.newLayout(nbrOfSlots);
        KEY from = layout.newKey();
        Value[] values = this.generateValuesForCompositeKey(nbrOfSlots, valueGenerator);
        for (int slot = 0; slot < nbrOfSlots; ++slot) {
            from.writeValue(slot, values[slot], NativeIndexKey.Inclusion.NEUTRAL);
        }
        KEY to = this.compositeKeyStateWithSomePreviousState(layout, nbrOfSlots, valueGenerator);
        to.copyFrom(from);
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)from.compareValueTo(to), (String)"states not equals after copy");
    }

    @Test
    void copyShouldCopyExtremeValues() {
        KEY extreme = this.newKeyState();
        KEY copy = this.newKeyState();
        for (ValueGroup valueGroup : ValueGroup.values()) {
            if (valueGroup == ValueGroup.NO_VALUE) continue;
            extreme.initValueAsLowest(valueGroup);
            copy.copyFrom(extreme);
            org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)extreme.compareValueTo(copy), (String)("states not equals after copy, valueGroup=" + valueGroup));
            extreme.initValueAsHighest(valueGroup);
            copy.copyFrom(extreme);
            org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)extreme.compareValueTo(copy), (String)("states not equals after copy, valueGroup=" + valueGroup));
        }
    }

    @ParameterizedTest
    @MethodSource(value={"validComparableValueGenerators"})
    void compareToMustAlignWithValuesCompareTo(ValueGenerator valueGenerator) {
        int i;
        ArrayList<Value> values = new ArrayList<Value>();
        ArrayList<KEY> states = new ArrayList<KEY>();
        for (i = 0; i < 10; ++i) {
            Value value = valueGenerator.next();
            values.add(value);
            KEY state = this.newKeyState();
            state.writeValue(value, NativeIndexKey.Inclusion.NEUTRAL);
            states.add(state);
        }
        values.sort((Comparator<Value>)Values.COMPARATOR);
        states.sort(GenericKey::compareValueTo);
        for (i = 0; i < values.size(); ++i) {
            org.junit.jupiter.api.Assertions.assertEquals(values.get(i), (Object)((GenericKey)states.get(i)).asValue(), (String)"sort order was different");
        }
    }

    @ParameterizedTest
    @MethodSource(value={"validComparableValueGenerators"})
    void compositeKeyCompareToMustAlignWithValuesCompareTo(ValueGenerator valueGenerator) {
        int i;
        int nbrOfSlots = this.random.nextInt(2, 5);
        ArrayList<KEY> states = new ArrayList<KEY>();
        Layout<KEY> layout = this.newLayout(nbrOfSlots);
        for (i = 0; i < 10; ++i) {
            KEY key = layout.newKey();
            states.add(key);
            for (int slot = 0; slot < nbrOfSlots; ++slot) {
                key.writeValue(slot, valueGenerator.next(), NativeIndexKey.Inclusion.NEUTRAL);
            }
        }
        states.sort(GenericKey::compareValueTo);
        for (i = 0; i < 9; ++i) {
            int result;
            GenericKey key1 = (GenericKey)states.get(i);
            GenericKey key2 = (GenericKey)states.get(i + 1);
            for (int slot = 0; slot < nbrOfSlots && (result = Values.COMPARATOR.compare(key1.asValues()[slot], key2.asValues()[slot])) >= 0; ++slot) {
                if (result <= 0) continue;
                org.junit.jupiter.api.Assertions.fail((String)("Keys incorrectly ordered: " + key1 + " , " + key2));
            }
        }
    }

    @ParameterizedTest
    @MethodSource(value={"validComparableValueGenerators"})
    void mustProduceValidMinimalSplitters(ValueGenerator valueGenerator) {
        Value value2;
        Value value1 = valueGenerator.next();
        while (Values.COMPARATOR.compare(value1, value2 = valueGenerator.next()) == 0) {
        }
        Value left = IndexKeyStateTest.pickSmaller(value1, value2);
        Value right = left == value1 ? value2 : value1;
        KEY leftState = this.newKeyState();
        leftState.writeValue(left, NativeIndexKey.Inclusion.NEUTRAL);
        KEY rightState = this.newKeyState();
        rightState.writeValue(right, NativeIndexKey.Inclusion.NEUTRAL);
        this.assertValidMinimalSplitter(leftState, rightState, this::newKeyState);
    }

    @ParameterizedTest
    @MethodSource(value={"validComparableValueGenerators"})
    void mustProduceValidMinimalSplittersCompositeKey(ValueGenerator valueGenerator) {
        Object[] values2;
        int nbrOfSlots = this.random.nextInt(2, 5);
        Layout<KEY> layout = this.newLayout(nbrOfSlots);
        KEY key1 = layout.newKey();
        Object[] values1 = this.generateValuesForCompositeKey(nbrOfSlots, valueGenerator);
        KEY key2 = layout.newKey();
        while (Arrays.equals(values1, values2 = this.generateValuesForCompositeKey(nbrOfSlots, valueGenerator))) {
        }
        for (int slot = 0; slot < nbrOfSlots; ++slot) {
            key1.writeValue(slot, (Value)values1[slot], NativeIndexKey.Inclusion.NEUTRAL);
            key2.writeValue(slot, (Value)values2[slot], NativeIndexKey.Inclusion.NEUTRAL);
        }
        KEY leftState = key1.compareValueTo(key2) < 0 ? key1 : key2;
        KEY rightState = leftState == key1 ? key2 : key1;
        this.assertValidMinimalSplitter(leftState, rightState, layout::newKey);
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void mustProduceValidMinimalSplittersWhenValuesAreEqual(ValueGenerator valueGenerator) {
        Value value = valueGenerator.next();
        KEY leftState = this.newKeyState();
        leftState.writeValue(value, NativeIndexKey.Inclusion.NEUTRAL);
        KEY rightState = this.newKeyState();
        rightState.writeValue(value, NativeIndexKey.Inclusion.NEUTRAL);
        this.assertValidMinimalSplitterForEqualValues(leftState, rightState, this::newKeyState);
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void mustProduceValidMinimalSplittersWhenValuesAreEqualCompositeKey(ValueGenerator valueGenerator) {
        int nbrOfSlots = this.random.nextInt(2, 5);
        Layout<KEY> layout = this.newLayout(nbrOfSlots);
        KEY leftState = layout.newKey();
        KEY rightState = layout.newKey();
        Value[] values = this.generateValuesForCompositeKey(nbrOfSlots, valueGenerator);
        for (int slot = 0; slot < nbrOfSlots; ++slot) {
            leftState.writeValue(slot, values[slot], NativeIndexKey.Inclusion.NEUTRAL);
            rightState.writeValue(slot, values[slot], NativeIndexKey.Inclusion.NEUTRAL);
        }
        this.assertValidMinimalSplitterForEqualValues(leftState, rightState, layout::newKey);
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void mustReportCorrectSize(ValueGenerator valueGenerator) {
        PageCursor cursor = IndexKeyStateTest.newPageCursor();
        Value value = valueGenerator.next();
        KEY state = this.newKeyState();
        state.writeValue(value, NativeIndexKey.Inclusion.NEUTRAL);
        int offsetBefore = cursor.getOffset();
        int reportedSize = state.size();
        state.put(cursor);
        int offsetAfter = cursor.getOffset();
        int actualSize = offsetAfter - offsetBefore;
        org.junit.jupiter.api.Assertions.assertEquals((int)reportedSize, (int)actualSize, (String)String.format("did not report correct size, value=%s, actualSize=%d, reportedSize=%d", value, actualSize, reportedSize));
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void mustReportCorrectSizeCompositeKey(ValueGenerator valueGenerator) {
        int nbrOfSlots = this.random.nextInt(2, 5);
        PageCursor cursor = IndexKeyStateTest.newPageCursor();
        Layout<KEY> layout = this.newLayout(nbrOfSlots);
        KEY state = layout.newKey();
        Object[] writtenValues = this.generateValuesForCompositeKey(nbrOfSlots, valueGenerator);
        for (int slot = 0; slot < nbrOfSlots; ++slot) {
            state.writeValue(slot, writtenValues[slot], NativeIndexKey.Inclusion.NEUTRAL);
        }
        int offsetBefore = cursor.getOffset();
        int reportedSize = state.size();
        state.put(cursor);
        int offsetAfter = cursor.getOffset();
        int actualSize = offsetAfter - offsetBefore;
        org.junit.jupiter.api.Assertions.assertEquals((int)reportedSize, (int)actualSize, (String)String.format("did not report correct size, value=%s, actualSize=%d, reportedSize=%d", Arrays.toString(writtenValues), actualSize, reportedSize));
    }

    @Test
    void lowestMustBeLowest() {
        this.assertLowest((Value)PointValue.MIN_VALUE);
        this.assertLowest((Value)DateTimeValue.MIN_VALUE);
        this.assertLowest((Value)LocalDateTimeValue.MIN_VALUE);
        this.assertLowest((Value)DateValue.MIN_VALUE);
        this.assertLowest((Value)TimeValue.MIN_VALUE);
        this.assertLowest((Value)LocalTimeValue.MIN_VALUE);
        this.assertLowest((Value)DurationValue.duration((Duration)Duration.ofSeconds(Long.MIN_VALUE, 0L)));
        this.assertLowest((Value)DurationValue.duration((Period)Period.of(Integer.MIN_VALUE, Integer.MIN_VALUE, Integer.MIN_VALUE)));
        this.assertLowest(Values.of((Object)UTF8.decode((byte[])new byte[0])));
        this.assertLowest(Values.of((Object)false));
        this.assertLowest(Values.of((Object)-128));
        this.assertLowest(Values.of((Object)Short.MIN_VALUE));
        this.assertLowest(Values.of((Object)Integer.MIN_VALUE));
        this.assertLowest(Values.of((Object)Long.MIN_VALUE));
        this.assertLowest(Values.of((Object)Float.valueOf(Float.NEGATIVE_INFINITY)));
        this.assertLowest(Values.of((Object)Double.NEGATIVE_INFINITY));
        this.assertLowest((Value)Values.pointArray((PointValue[])new PointValue[0]));
        this.assertLowest((Value)Values.dateTimeArray((ZonedDateTime[])new ZonedDateTime[0]));
        this.assertLowest((Value)Values.localDateTimeArray((LocalDateTime[])new LocalDateTime[0]));
        this.assertLowest((Value)Values.dateArray((LocalDate[])new LocalDate[0]));
        this.assertLowest((Value)Values.timeArray((OffsetTime[])new OffsetTime[0]));
        this.assertLowest((Value)Values.localTimeArray((LocalTime[])new LocalTime[0]));
        this.assertLowest((Value)Values.durationArray((DurationValue[])new DurationValue[0]));
        this.assertLowest((Value)Values.durationArray((TemporalAmount[])new TemporalAmount[0]));
        this.assertLowest(Values.of((Object)new String[0]));
        this.assertLowest(Values.of((Object)new boolean[0]));
        this.assertLowest(Values.of((Object)new byte[0]));
        this.assertLowest(Values.of((Object)new short[0]));
        this.assertLowest(Values.of((Object)new int[0]));
        this.assertLowest(Values.of((Object)new long[0]));
        this.assertLowest(Values.of((Object)new float[0]));
        this.assertLowest(Values.of((Object)new double[0]));
    }

    @Test
    void highestMustBeHighest() {
        this.assertHighest((Value)PointValue.MAX_VALUE);
        this.assertHighest((Value)DateTimeValue.MAX_VALUE);
        this.assertHighest((Value)LocalDateTimeValue.MAX_VALUE);
        this.assertHighest((Value)DateValue.MAX_VALUE);
        this.assertHighest((Value)TimeValue.MAX_VALUE);
        this.assertHighest((Value)LocalTimeValue.MAX_VALUE);
        this.assertHighest((Value)DurationValue.duration((Duration)Duration.ofSeconds(Long.MAX_VALUE, 999999999L)));
        this.assertHighest((Value)DurationValue.duration((Period)Period.of(Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE)));
        this.assertHighestString();
        this.assertHighest(Values.of((Object)true));
        this.assertHighest(Values.of((Object)127));
        this.assertHighest(Values.of((Object)Short.MAX_VALUE));
        this.assertHighest(Values.of((Object)Integer.MAX_VALUE));
        this.assertHighest(Values.of((Object)Long.MAX_VALUE));
        this.assertHighest(Values.of((Object)Float.valueOf(Float.POSITIVE_INFINITY)));
        this.assertHighest(Values.of((Object)Double.POSITIVE_INFINITY));
        this.assertHighest((Value)Values.pointArray((PointValue[])new PointValue[]{PointValue.MAX_VALUE}));
        this.assertHighest((Value)Values.dateTimeArray((ZonedDateTime[])new ZonedDateTime[]{(ZonedDateTime)DateTimeValue.MAX_VALUE.asObjectCopy()}));
        this.assertHighest((Value)Values.localDateTimeArray((LocalDateTime[])new LocalDateTime[]{(LocalDateTime)LocalDateTimeValue.MAX_VALUE.asObjectCopy()}));
        this.assertHighest((Value)Values.dateArray((LocalDate[])new LocalDate[]{(LocalDate)DateValue.MAX_VALUE.asObjectCopy()}));
        this.assertHighest((Value)Values.timeArray((OffsetTime[])new OffsetTime[]{(OffsetTime)TimeValue.MAX_VALUE.asObjectCopy()}));
        this.assertHighest((Value)Values.localTimeArray((LocalTime[])new LocalTime[]{(LocalTime)LocalTimeValue.MAX_VALUE.asObjectCopy()}));
        this.assertHighest((Value)Values.durationArray((DurationValue[])new DurationValue[]{DurationValue.duration((Duration)Duration.ofSeconds(Long.MAX_VALUE, 999999999L))}));
        this.assertHighest((Value)Values.durationArray((DurationValue[])new DurationValue[]{DurationValue.duration((Period)Period.of(Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE))}));
        this.assertHighest((Value)Values.durationArray((TemporalAmount[])new TemporalAmount[]{Duration.ofSeconds(Long.MAX_VALUE, 999999999L)}));
        this.assertHighest((Value)Values.durationArray((TemporalAmount[])new TemporalAmount[]{Period.of(Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE)}));
        this.assertHighestStringArray();
        this.assertHighest((Value)Values.booleanArray((boolean[])new boolean[]{true}));
        this.assertHighest((Value)Values.byteArray((byte[])new byte[]{127}));
        this.assertHighest((Value)Values.shortArray((short[])new short[]{Short.MAX_VALUE}));
        this.assertHighest((Value)Values.intArray((int[])new int[]{Integer.MAX_VALUE}));
        this.assertHighest((Value)Values.longArray((long[])new long[]{Long.MAX_VALUE}));
        this.assertHighest((Value)Values.floatArray((float[])new float[]{Float.POSITIVE_INFINITY}));
        this.assertHighest((Value)Values.doubleArray((double[])new double[]{Double.POSITIVE_INFINITY}));
    }

    @Test
    void shouldNeverOverwriteDereferencedTextValues() {
        TextValue srcValue = Values.utf8Value((byte[])"First string".getBytes(StandardCharsets.UTF_8));
        KEY genericKeyState = this.newKeyState();
        genericKeyState.writeValue((Value)srcValue, NativeIndexKey.Inclusion.NEUTRAL);
        Value dereferencedValue = genericKeyState.asValue();
        org.junit.jupiter.api.Assertions.assertEquals((Object)srcValue, (Object)dereferencedValue);
        PageCursor cursor = IndexKeyStateTest.newPageCursor();
        int offset = cursor.getOffset();
        genericKeyState.put(cursor);
        int keySize = cursor.getOffset() - offset;
        cursor.setOffset(offset);
        genericKeyState.clear();
        TextValue srcValue2 = Values.utf8Value((byte[])"Secondstring".getBytes(StandardCharsets.UTF_8));
        genericKeyState.writeValue((Value)srcValue2, NativeIndexKey.Inclusion.NEUTRAL);
        Value dereferencedValue2 = genericKeyState.asValue();
        org.junit.jupiter.api.Assertions.assertEquals((Object)srcValue2, (Object)dereferencedValue2);
        org.junit.jupiter.api.Assertions.assertEquals((Object)srcValue, (Object)dereferencedValue);
        genericKeyState.clear();
        genericKeyState.get(cursor, keySize);
        Value dereferencedValue3 = genericKeyState.asValue();
        org.junit.jupiter.api.Assertions.assertEquals((Object)srcValue, (Object)dereferencedValue3);
        org.junit.jupiter.api.Assertions.assertEquals((Object)srcValue2, (Object)dereferencedValue2);
        org.junit.jupiter.api.Assertions.assertEquals((Object)srcValue, (Object)dereferencedValue);
    }

    @Test
    void indexedCharShouldComeBackAsCharValue() {
        this.shouldReadBackToExactOriginalValue((Value)this.random.randomValues().nextCharValue());
    }

    @Test
    void indexedCharArrayShouldComeBackAsCharArrayValue() {
        this.shouldReadBackToExactOriginalValue((Value)this.random.randomValues().nextCharArray());
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void minimalSplitterForSameValueShouldDivideLeftAndRight(ValueGenerator valueGenerator) {
        Value value = valueGenerator.next();
        Layout<KEY> layout = this.newLayout(1);
        KEY left = layout.newKey();
        KEY right = layout.newKey();
        KEY minimalSplitter = layout.newKey();
        left.initialize(1L);
        left.initFromValue(0, value, NativeIndexKey.Inclusion.NEUTRAL);
        right.initialize(2L);
        right.initFromValue(0, value, NativeIndexKey.Inclusion.NEUTRAL);
        layout.minimalSplitter(left, right, minimalSplitter);
        org.junit.jupiter.api.Assertions.assertTrue((layout.compare(left, minimalSplitter) < 0 ? 1 : 0) != 0, (String)("Expected minimal splitter to be strictly greater than left but wasn't for value " + value));
        org.junit.jupiter.api.Assertions.assertTrue((layout.compare(minimalSplitter, right) <= 0 ? 1 : 0) != 0, (String)("Expected right to be greater than or equal to minimal splitter but wasn't for value " + value));
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void minimalSplitterShouldRemoveEntityIdIfPossible(ValueGenerator valueGenerator) {
        Value firstValue = valueGenerator.next();
        Value secondValue = IndexKeyStateTest.uniqueSecondValue(valueGenerator, firstValue);
        Value leftValue = IndexKeyStateTest.pickSmaller(firstValue, secondValue);
        Value rightValue = IndexKeyStateTest.pickOther(firstValue, secondValue, leftValue);
        Layout<KEY> layout = this.newLayout(1);
        KEY left = layout.newKey();
        KEY right = layout.newKey();
        KEY minimalSplitter = layout.newKey();
        left.initialize(1L);
        left.initFromValue(0, leftValue, NativeIndexKey.Inclusion.NEUTRAL);
        right.initialize(2L);
        right.initFromValue(0, rightValue, NativeIndexKey.Inclusion.NEUTRAL);
        layout.minimalSplitter(left, right, minimalSplitter);
        org.junit.jupiter.api.Assertions.assertEquals((long)-1L, (long)minimalSplitter.getEntityId(), (String)("Expected minimal splitter to have entityId removed when constructed from keys with unique values: left=" + leftValue + ", right=" + rightValue));
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void minimalSplitterForSameValueShouldDivideLeftAndRightCompositeKey(ValueGenerator valueGenerator) {
        int nbrOfSlots = this.random.nextInt(2, 5);
        Layout<KEY> layout = this.newLayout(nbrOfSlots);
        KEY left = layout.newKey();
        KEY right = layout.newKey();
        KEY minimalSplitter = layout.newKey();
        left.initialize(1L);
        right.initialize(2L);
        Object[] values = new Value[nbrOfSlots];
        for (int slot = 0; slot < nbrOfSlots; ++slot) {
            Value value;
            values[slot] = value = valueGenerator.next();
            left.initFromValue(slot, value, NativeIndexKey.Inclusion.NEUTRAL);
            right.initFromValue(slot, value, NativeIndexKey.Inclusion.NEUTRAL);
        }
        layout.minimalSplitter(left, right, minimalSplitter);
        org.junit.jupiter.api.Assertions.assertTrue((layout.compare(left, minimalSplitter) < 0 ? 1 : 0) != 0, (String)("Expected minimal splitter to be strictly greater than left but wasn't for value " + Arrays.toString(values)));
        org.junit.jupiter.api.Assertions.assertTrue((layout.compare(minimalSplitter, right) <= 0 ? 1 : 0) != 0, (String)("Expected right to be greater than or equal to minimal splitter but wasn't for value " + Arrays.toString(values)));
    }

    @ParameterizedTest
    @MethodSource(value={"validValueGenerators"})
    void minimalSplitterShouldRemoveEntityIdIfPossibleCompositeKey(ValueGenerator valueGenerator) {
        int nbrOfSlots = this.random.nextInt(2, 5);
        int differingSlot = this.random.nextInt(nbrOfSlots);
        Layout<KEY> layout = this.newLayout(nbrOfSlots);
        KEY left = layout.newKey();
        KEY right = layout.newKey();
        KEY minimalSplitter = layout.newKey();
        left.initialize(1L);
        right.initialize(2L);
        for (int slot = 0; slot < nbrOfSlots; ++slot) {
            if (slot == differingSlot) continue;
            Value value = valueGenerator.next();
            left.initFromValue(slot, value, NativeIndexKey.Inclusion.NEUTRAL);
            right.initFromValue(slot, value, NativeIndexKey.Inclusion.NEUTRAL);
        }
        Value firstValue = valueGenerator.next();
        Value secondValue = IndexKeyStateTest.uniqueSecondValue(valueGenerator, firstValue);
        Value leftValue = IndexKeyStateTest.pickSmaller(firstValue, secondValue);
        Value rightValue = IndexKeyStateTest.pickOther(firstValue, secondValue, leftValue);
        left.initFromValue(differingSlot, leftValue, NativeIndexKey.Inclusion.NEUTRAL);
        right.initFromValue(differingSlot, rightValue, NativeIndexKey.Inclusion.NEUTRAL);
        layout.minimalSplitter(left, right, minimalSplitter);
        org.junit.jupiter.api.Assertions.assertEquals((long)-1L, (long)minimalSplitter.getEntityId(), (String)("Expected minimal splitter to have entityId removed when constructed from keys with unique values: left=" + leftValue + ", right=" + rightValue));
    }

    @ParameterizedTest
    @MethodSource(value={"singleValueGeneratorsStream"})
    void testDocumentedKeySizesNonArrays(ValueGenerator generator) {
        int expectedSizeOfData;
        Value value = generator.next();
        KEY key = this.newKeyState();
        key.initFromValue(0, value, NativeIndexKey.Inclusion.NEUTRAL);
        int keySize = key.size();
        int keyOverhead = 8;
        int actualSizeOfData = keySize - keyOverhead;
        String typeName = value.getTypeName();
        switch (value.valueGroup()) {
            case NUMBER: {
                expectedSizeOfData = IndexKeyStateTest.getNumberSize(value);
                break;
            }
            case BOOLEAN: {
                expectedSizeOfData = 2;
                break;
            }
            case DATE: {
                expectedSizeOfData = 9;
                break;
            }
            case ZONED_TIME: {
                expectedSizeOfData = 13;
                break;
            }
            case LOCAL_TIME: {
                expectedSizeOfData = 9;
                break;
            }
            case ZONED_DATE_TIME: {
                expectedSizeOfData = 17;
                break;
            }
            case LOCAL_DATE_TIME: {
                expectedSizeOfData = 13;
                break;
            }
            case DURATION: {
                expectedSizeOfData = 29;
                break;
            }
            case GEOMETRY: {
                expectedSizeOfData = this.getGeometrySize(value);
                break;
            }
            case TEXT: {
                expectedSizeOfData = IndexKeyStateTest.getStringSize(value);
                break;
            }
            default: {
                throw new RuntimeException("Did not expect this type to be tested in this test. Value was " + value);
            }
        }
        IndexKeyStateTest.assertKeySize(expectedSizeOfData, actualSizeOfData, typeName);
    }

    @ParameterizedTest
    @MethodSource(value={"arrayValueGeneratorsStream"})
    void testDocumentedKeySizesArrays(ValueGenerator generator) {
        int arrayElementSize;
        int arrayOverhead;
        Value value = generator.next();
        KEY key = this.newKeyState();
        key.initFromValue(0, value, NativeIndexKey.Inclusion.NEUTRAL);
        int keySize = key.size();
        int keyOverhead = 8;
        int actualSizeOfData = keySize - keyOverhead;
        int arrayLength = 0;
        if (value instanceof ArrayValue) {
            arrayLength = ((ArrayValue)value).length();
        }
        int normalArrayOverhead = 3;
        int numberArrayOverhead = 4;
        int geometryArrayOverhead = 6;
        String typeName = value.getTypeName();
        switch (value.valueGroup()) {
            case NUMBER_ARRAY: {
                arrayOverhead = numberArrayOverhead;
                arrayElementSize = IndexKeyStateTest.getNumberArrayElementSize(value);
                break;
            }
            case BOOLEAN_ARRAY: {
                arrayOverhead = normalArrayOverhead;
                arrayElementSize = 1;
                break;
            }
            case DATE_ARRAY: {
                arrayOverhead = normalArrayOverhead;
                arrayElementSize = 8;
                break;
            }
            case ZONED_TIME_ARRAY: {
                arrayOverhead = normalArrayOverhead;
                arrayElementSize = 12;
                break;
            }
            case LOCAL_TIME_ARRAY: {
                arrayOverhead = normalArrayOverhead;
                arrayElementSize = 8;
                break;
            }
            case ZONED_DATE_TIME_ARRAY: {
                arrayOverhead = normalArrayOverhead;
                arrayElementSize = 16;
                break;
            }
            case LOCAL_DATE_TIME_ARRAY: {
                arrayOverhead = normalArrayOverhead;
                arrayElementSize = 12;
                break;
            }
            case DURATION_ARRAY: {
                arrayOverhead = normalArrayOverhead;
                arrayElementSize = 28;
                break;
            }
            case GEOMETRY_ARRAY: {
                arrayOverhead = geometryArrayOverhead;
                arrayElementSize = this.getGeometryArrayElementSize(value, arrayLength);
                break;
            }
            case TEXT_ARRAY: {
                IndexKeyStateTest.assertTextArraySize(value, actualSizeOfData, normalArrayOverhead, typeName);
                return;
            }
            default: {
                throw new RuntimeException("Did not expect this type to be tested in this test. Value was " + value + " is value group " + value.valueGroup());
            }
        }
        int expectedSizeOfData = arrayOverhead + arrayLength * arrayElementSize;
        IndexKeyStateTest.assertKeySize(expectedSizeOfData, actualSizeOfData, typeName);
    }

    private static void assertKeySize(int expectedKeySize, int actualKeySize, String type) {
        org.junit.jupiter.api.Assertions.assertEquals((int)expectedKeySize, (int)actualKeySize, (String)("Expected keySize for type " + type + " to be " + expectedKeySize + " but was " + actualKeySize));
    }

    private void shouldReadBackToExactOriginalValue(Value srcValue) {
        KEY state = this.newKeyState();
        state.clear();
        state.writeValue(srcValue, NativeIndexKey.Inclusion.NEUTRAL);
        Value retrievedValueAfterWrittenToState = state.asValue();
        org.junit.jupiter.api.Assertions.assertEquals((Object)srcValue, (Object)retrievedValueAfterWrittenToState);
        org.junit.jupiter.api.Assertions.assertEquals(srcValue.getClass(), retrievedValueAfterWrittenToState.getClass());
        PageCursor cursor = IndexKeyStateTest.newPageCursor();
        int offset = cursor.getOffset();
        state.put(cursor);
        int keySize = cursor.getOffset() - offset;
        cursor.setOffset(offset);
        state.clear();
        state.get(cursor, keySize);
        Value retrievedValueAfterReadFromCursor = state.asValue();
        org.junit.jupiter.api.Assertions.assertEquals((Object)srcValue, (Object)retrievedValueAfterReadFromCursor);
        org.junit.jupiter.api.Assertions.assertEquals(srcValue.getClass(), retrievedValueAfterReadFromCursor.getClass());
    }

    private void assertHighestStringArray() {
        for (int i = 0; i < 1000; ++i) {
            this.assertHighest((Value)this.random.randomValues().nextTextArray());
        }
    }

    private void assertHighestString() {
        for (int i = 0; i < 1000; ++i) {
            this.assertHighest((Value)this.random.randomValues().nextTextValue());
        }
    }

    private void assertHighest(Value value) {
        KEY highestOfAll = this.newKeyState();
        KEY highestInValueGroup = this.newKeyState();
        KEY other = this.newKeyState();
        highestOfAll.initValueAsHighest(ValueGroup.UNKNOWN);
        highestInValueGroup.initValueAsHighest(value.valueGroup());
        other.writeValue(value, NativeIndexKey.Inclusion.NEUTRAL);
        org.junit.jupiter.api.Assertions.assertTrue((highestInValueGroup.compareValueTo(other) > 0 ? 1 : 0) != 0, (String)("highestInValueGroup not higher than " + value));
        org.junit.jupiter.api.Assertions.assertTrue((highestOfAll.compareValueTo(other) > 0 ? 1 : 0) != 0, (String)("highestOfAll not higher than " + value));
        org.junit.jupiter.api.Assertions.assertTrue((highestOfAll.compareValueTo(highestInValueGroup) > 0 || ((GenericKey)highestOfAll).type == ((GenericKey)highestInValueGroup).type ? 1 : 0) != 0, (String)"highestOfAll not higher than highestInValueGroup");
    }

    private void assertLowest(Value value) {
        KEY lowestOfAll = this.newKeyState();
        KEY lowestInValueGroup = this.newKeyState();
        KEY other = this.newKeyState();
        lowestOfAll.initValueAsLowest(ValueGroup.UNKNOWN);
        lowestInValueGroup.initValueAsLowest(value.valueGroup());
        other.writeValue(value, NativeIndexKey.Inclusion.NEUTRAL);
        org.junit.jupiter.api.Assertions.assertTrue((lowestInValueGroup.compareValueTo(other) <= 0 ? 1 : 0) != 0);
        org.junit.jupiter.api.Assertions.assertTrue((lowestOfAll.compareValueTo(other) <= 0 ? 1 : 0) != 0);
        org.junit.jupiter.api.Assertions.assertTrue((lowestOfAll.compareValueTo(lowestInValueGroup) <= 0 ? 1 : 0) != 0);
    }

    private static Value pickSmaller(Value value1, Value value2) {
        return Values.COMPARATOR.compare(value1, value2) < 0 ? value1 : value2;
    }

    private void assertValidMinimalSplitter(KEY leftState, KEY rightState, Supplier<KEY> keyFactory) {
        GenericKey minimalSplitter = (GenericKey)keyFactory.get();
        rightState.minimalSplitter(leftState, rightState, minimalSplitter);
        org.junit.jupiter.api.Assertions.assertTrue((leftState.compareValueTo(minimalSplitter) < 0 ? 1 : 0) != 0, (String)("left state not less than minimal splitter, leftState=" + leftState + ", rightState=" + rightState + ", minimalSplitter=" + minimalSplitter));
        org.junit.jupiter.api.Assertions.assertTrue((rightState.compareValueTo(minimalSplitter) >= 0 ? 1 : 0) != 0, (String)("right state not greater than or equal to minimal splitter, leftState=" + leftState + ", rightState=" + rightState + ", minimalSplitter=" + minimalSplitter));
    }

    private void assertValidMinimalSplitterForEqualValues(KEY leftState, KEY rightState, Supplier<KEY> keyFactory) {
        GenericKey minimalSplitter = (GenericKey)keyFactory.get();
        rightState.minimalSplitter(leftState, rightState, minimalSplitter);
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)leftState.compareValueTo(minimalSplitter), (String)("left state not equal to minimal splitter, leftState=" + leftState + ", rightState=" + rightState + ", minimalSplitter=" + minimalSplitter));
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)rightState.compareValueTo(minimalSplitter), (String)("right state not equal to minimal splitter, leftState=" + leftState + ", rightState=" + rightState + ", minimalSplitter=" + minimalSplitter));
    }

    private Value nextValidValue(boolean includeIncomparable) {
        Value value;
        do {
            value = this.random.randomValues().nextValue();
        } while (!includeIncomparable && IndexKeyStateTest.isIncomparable(value));
        return value;
    }

    private static boolean isIncomparable(Value value) {
        return Values.isGeometryValue((Value)value) || Values.isGeometryArray((Value)value);
    }

    private ValueGenerator[] listValueGenerators(boolean includeIncomparable) {
        ArrayList<ValueGenerator> generators = new ArrayList<ValueGenerator>();
        generators.addAll(this.singleValueGenerators(includeIncomparable));
        generators.addAll(this.arrayValueGenerators(includeIncomparable));
        generators.add(() -> this.nextValidValue(includeIncomparable));
        return generators.toArray(new ValueGenerator[0]);
    }

    private List<ValueGenerator> singleValueGenerators(boolean includeIncomparable) {
        ArrayList<ValueGenerator> generators = new ArrayList<ValueGenerator>(Arrays.asList(() -> this.random.randomValues().nextDateTimeValue(), () -> this.random.randomValues().nextLocalDateTimeValue(), () -> this.random.randomValues().nextDateValue(), () -> this.random.randomValues().nextTimeValue(), () -> this.random.randomValues().nextLocalTimeValue(), () -> this.random.randomValues().nextPeriod(), () -> this.random.randomValues().nextDuration(), () -> this.random.randomValues().nextCharValue(), () -> this.random.randomValues().nextTextValue(), () -> this.random.randomValues().nextAlphaNumericTextValue(), () -> this.random.randomValues().nextBooleanValue(), () -> this.random.randomValues().nextNumberValue()));
        if (includeIncomparable) {
            generators.addAll(Arrays.asList(() -> this.random.randomValues().nextPointValue(), () -> this.random.randomValues().nextGeographicPoint(), () -> this.random.randomValues().nextGeographic3DPoint(), () -> this.random.randomValues().nextCartesianPoint(), () -> this.random.randomValues().nextCartesian3DPoint()));
        }
        return generators;
    }

    private List<ValueGenerator> arrayValueGenerators(boolean includeIncomparable) {
        ArrayList<ValueGenerator> generators = new ArrayList<ValueGenerator>(Arrays.asList(() -> this.random.randomValues().nextDateTimeArray(), () -> this.random.randomValues().nextLocalDateTimeArray(), () -> this.random.randomValues().nextDateArray(), () -> this.random.randomValues().nextTimeArray(), () -> this.random.randomValues().nextLocalTimeArray(), () -> this.random.randomValues().nextDurationArray(), () -> this.random.randomValues().nextDurationArray(), () -> this.random.randomValues().nextCharArray(), () -> this.random.randomValues().nextTextArray(), () -> this.random.randomValues().nextAlphaNumericTextArray(), () -> this.random.randomValues().nextBooleanArray(), () -> this.random.randomValues().nextByteArray(), () -> this.random.randomValues().nextShortArray(), () -> this.random.randomValues().nextIntArray(), () -> this.random.randomValues().nextLongArray(), () -> this.random.randomValues().nextFloatArray(), () -> this.random.randomValues().nextDoubleArray()));
        if (includeIncomparable) {
            generators.addAll(Arrays.asList(() -> this.random.randomValues().nextPointArray(), () -> this.random.randomValues().nextGeographicPointArray(), () -> this.random.randomValues().nextGeographic3DPointArray(), () -> this.random.randomValues().nextCartesianPointArray(), () -> this.random.randomValues().nextCartesian3DPointArray()));
        }
        return generators;
    }

    private Stream<ValueGenerator> validValueGenerators() {
        return Stream.of(this.listValueGenerators(true));
    }

    private Stream<ValueGenerator> singleValueGeneratorsStream() {
        return this.singleValueGenerators(true).stream();
    }

    private Stream<ValueGenerator> arrayValueGeneratorsStream() {
        return this.arrayValueGenerators(true).stream();
    }

    private Stream<ValueGenerator> validComparableValueGenerators() {
        return Stream.of(this.listValueGenerators(this.includePointTypesForComparisons()));
    }

    private ValueGenerator randomValueGenerator() {
        ValueGenerator[] generators = this.listValueGenerators(true);
        return generators[this.random.nextInt(generators.length)];
    }

    private Value[] generateValuesForCompositeKey(int nbrOfSlots, ValueGenerator firstSlotValueGenerator) {
        Value[] values = new Value[nbrOfSlots];
        values[0] = firstSlotValueGenerator.next();
        for (int slot = 1; slot < nbrOfSlots; ++slot) {
            values[slot] = this.randomValueGenerator().next();
        }
        return values;
    }

    private static int getStringSize(Value value) {
        if (!(value instanceof TextValue)) {
            throw new RuntimeException("Unexpected class for value in value group " + ValueGroup.TEXT + ", was " + value.getClass());
        }
        int expectedSizeOfData = 3 + ((TextValue)value).stringValue().getBytes(StandardCharsets.UTF_8).length;
        return expectedSizeOfData;
    }

    private int getGeometrySize(Value value) {
        if (!(value instanceof PointValue)) {
            throw new RuntimeException("Unexpected class for value in value group " + ValueGroup.GEOMETRY + ", was " + value.getClass());
        }
        int dimensions = ((PointValue)value).coordinate().length;
        return this.getPointSerialisedSize(dimensions);
    }

    private static int getNumberSize(Value value) {
        int expectedSizeOfData;
        if (value instanceof ByteValue) {
            expectedSizeOfData = 3;
        } else if (value instanceof ShortValue) {
            expectedSizeOfData = 4;
        } else if (value instanceof IntValue) {
            expectedSizeOfData = 6;
        } else if (value instanceof LongValue) {
            expectedSizeOfData = 10;
        } else if (value instanceof FloatValue) {
            expectedSizeOfData = 6;
        } else if (value instanceof DoubleValue) {
            expectedSizeOfData = 10;
        } else {
            throw new RuntimeException("Unexpected class for value in value group " + ValueGroup.NUMBER + ", was " + value.getClass());
        }
        return expectedSizeOfData;
    }

    private static int getNumberArrayElementSize(Value value) {
        int arrayElementSize;
        if (value instanceof ByteArray) {
            arrayElementSize = 1;
        } else if (value instanceof ShortArray) {
            arrayElementSize = 2;
        } else if (value instanceof IntArray) {
            arrayElementSize = 4;
        } else if (value instanceof LongArray) {
            arrayElementSize = 8;
        } else if (value instanceof FloatArray) {
            arrayElementSize = 4;
        } else if (value instanceof DoubleArray) {
            arrayElementSize = 8;
        } else {
            throw new RuntimeException("Unexpected class for value in value group " + ValueGroup.NUMBER_ARRAY + ", was " + value.getClass());
        }
        return arrayElementSize;
    }

    private static void assertTextArraySize(Value value, int actualSizeOfData, int normalArrayOverhead, String typeName) {
        int sumOfStrings;
        if (value instanceof TextArray) {
            sumOfStrings = 0;
            TextArray stringArray = (TextArray)value;
            for (int i = 0; i < stringArray.length(); ++i) {
                String string = stringArray.stringValue(i);
                sumOfStrings += 2 + string.getBytes(StandardCharsets.UTF_8).length;
            }
        } else {
            throw new RuntimeException("Unexpected class for value in value group " + ValueGroup.TEXT_ARRAY + ", was " + value.getClass());
        }
        int totalTextArraySize = normalArrayOverhead + sumOfStrings;
        IndexKeyStateTest.assertKeySize(totalTextArraySize, actualSizeOfData, typeName);
    }

    private int getGeometryArrayElementSize(Value value, int arrayLength) {
        if (arrayLength < 1) {
            return 0;
        }
        if (!(value instanceof PointArray)) {
            throw new RuntimeException("Unexpected class for value in value group " + ValueGroup.GEOMETRY_ARRAY + ", was " + value.getClass());
        }
        int dimensions = ((PointArray)value).pointValue(0).coordinate().length;
        return this.getArrayPointSerialisedSize(dimensions);
    }

    private KEY genericKeyStateWithSomePreviousState(ValueGenerator valueGenerator) {
        KEY to = this.newKeyState();
        if (this.random.nextBoolean()) {
            NativeIndexKey.Inclusion inclusion = (NativeIndexKey.Inclusion)this.random.among((Object[])NativeIndexKey.Inclusion.values());
            Value value = valueGenerator.next();
            to.writeValue(value, inclusion);
        }
        return to;
    }

    private KEY compositeKeyStateWithSomePreviousState(Layout<KEY> layout, int nbrOfSlots, ValueGenerator valueGenerator) {
        KEY to = layout.newKey();
        if (this.random.nextBoolean()) {
            Value[] previousValues = this.generateValuesForCompositeKey(nbrOfSlots, valueGenerator);
            for (int slot = 0; slot < nbrOfSlots; ++slot) {
                NativeIndexKey.Inclusion inclusion = (NativeIndexKey.Inclusion)this.random.among((Object[])NativeIndexKey.Inclusion.values());
                to.writeValue(slot, previousValues[slot], inclusion);
            }
        }
        return to;
    }

    private static PageCursor newPageCursor() {
        return ByteArrayPageCursor.wrap((int)8192);
    }

    private static Value pickOther(Value value1, Value value2, Value currentValue) {
        return currentValue == value1 ? value2 : value1;
    }

    private static Value uniqueSecondValue(ValueGenerator valueGenerator, Value firstValue) {
        Value secondValue;
        while (Values.COMPARATOR.compare(firstValue, secondValue = valueGenerator.next()) == 0) {
        }
        return secondValue;
    }

    KEY newKeyState() {
        return this.newLayout(1).newKey();
    }

    abstract Layout<KEY> newLayout(int var1);

    abstract boolean includePointTypesForComparisons();

    abstract int getPointSerialisedSize(int var1);

    abstract int getArrayPointSerialisedSize(int var1);

    static interface Layout<KEY extends GenericKey<KEY>> {
        public KEY newKey();

        public void minimalSplitter(KEY var1, KEY var2, KEY var3);

        public int compare(KEY var1, KEY var2);
    }

    @FunctionalInterface
    private static interface ValueGenerator {
        public Value next();
    }
}

