/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction.log.files;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.LockSupport;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.assertj.core.api.AbstractLongAssert;
import org.assertj.core.api.Assertions;
import org.eclipse.collections.api.map.primitive.LongObjectMap;
import org.eclipse.collections.api.map.primitive.MutableLongObjectMap;
import org.eclipse.collections.impl.factory.primitive.LongObjectMaps;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.condition.EnabledOnOs;
import org.junit.jupiter.api.condition.OS;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;
import org.neo4j.internal.nativeimpl.ErrorTranslator;
import org.neo4j.internal.nativeimpl.NativeAccess;
import org.neo4j.internal.nativeimpl.NativeCallResult;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.DelegatingFileSystemAbstraction;
import org.neo4j.io.fs.DelegatingStoreChannel;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.ReadableChannel;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.kernel.impl.transaction.SimpleLogVersionRepository;
import org.neo4j.kernel.impl.transaction.SimpleTransactionIdStore;
import org.neo4j.kernel.impl.transaction.log.FlushablePositionAwareChecksumChannel;
import org.neo4j.kernel.impl.transaction.log.LogPosition;
import org.neo4j.kernel.impl.transaction.log.PhysicalLogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.PhysicalLogicalTransactionStore;
import org.neo4j.kernel.impl.transaction.log.ReadableLogChannel;
import org.neo4j.kernel.impl.transaction.log.TestLogEntryReader;
import org.neo4j.kernel.impl.transaction.log.TransactionLogWriter;
import org.neo4j.kernel.impl.transaction.log.entry.IncompleteLogHeaderException;
import org.neo4j.kernel.impl.transaction.log.entry.LogHeader;
import org.neo4j.kernel.impl.transaction.log.entry.LogHeaderReader;
import org.neo4j.kernel.impl.transaction.log.entry.LogHeaderWriter;
import org.neo4j.kernel.impl.transaction.log.files.LogFile;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.LogFilesBuilder;
import org.neo4j.kernel.impl.transaction.log.files.LogHeaderVisitor;
import org.neo4j.kernel.impl.transaction.log.files.TransactionLogFile;
import org.neo4j.kernel.impl.transaction.log.files.TransactionLogFilesHelper;
import org.neo4j.kernel.impl.transaction.tracing.LogAppendEvent;
import org.neo4j.kernel.impl.transaction.tracing.LogForceEvents;
import org.neo4j.kernel.lifecycle.LifeSupport;
import org.neo4j.kernel.lifecycle.Lifecycle;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.storageengine.api.LogVersionRepository;
import org.neo4j.storageengine.api.StoreId;
import org.neo4j.storageengine.api.TransactionIdStore;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.LifeExtension;
import org.neo4j.test.extension.Neo4jLayoutExtension;
import org.neo4j.test.utils.TestDirectory;
import org.neo4j.util.concurrent.Futures;

@Neo4jLayoutExtension
@ExtendWith(value={LifeExtension.class})
class TransactionLogFileTest {
    @Inject
    private TestDirectory testDirectory;
    @Inject
    private DatabaseLayout databaseLayout;
    @Inject
    private FileSystemAbstraction fileSystem;
    @Inject
    private LifeSupport life;
    private CapturingChannelFileSystem wrappingFileSystem;
    private final long rotationThreshold = ByteUnit.mebiBytes((long)1L);
    private final LogVersionRepository logVersionRepository = new SimpleLogVersionRepository(1L);
    private final TransactionIdStore transactionIdStore = new SimpleTransactionIdStore(2L, 0, 0L, 0L, 0L);

    TransactionLogFileTest() {
    }

    @BeforeEach
    void setUp() {
        this.wrappingFileSystem = new CapturingChannelFileSystem(this.fileSystem);
    }

    @Test
    @EnabledOnOs(value={OS.LINUX})
    void truncateCurrentLogFile() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        this.life.add((Lifecycle)logFiles);
        this.life.start();
        LogFile logFile = logFiles.getLogFile();
        long sizeBefore = this.fileSystem.getFileSize(logFile.getLogFileForVersion(logFile.getCurrentLogVersion()));
        logFile.truncate();
        long sizeAfter = this.fileSystem.getFileSize(logFile.getLogFileForVersion(logFile.getCurrentLogVersion()));
        ((AbstractLongAssert)Assertions.assertThat((long)sizeBefore).describedAs("Truncation should truncate any preallocated space.", new Object[0])).isGreaterThan(sizeAfter);
    }

    @Test
    void skipLogFileWithoutHeader() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        this.life.add((Lifecycle)logFiles);
        this.life.start();
        this.logVersionRepository.incrementAndGetVersion(CursorContext.NULL);
        this.fileSystem.write(logFiles.getLogFile().getLogFileForVersion(this.logVersionRepository.getCurrentLogVersion())).close();
        this.transactionIdStore.transactionCommitted(5L, 5, 5L, CursorContext.NULL);
        PhysicalLogicalTransactionStore.LogVersionLocator versionLocator = new PhysicalLogicalTransactionStore.LogVersionLocator(4L);
        logFiles.getLogFile().accept((LogHeaderVisitor)versionLocator);
        LogPosition logPosition = versionLocator.getLogPosition();
        org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)logPosition.getLogVersion());
    }

    @Test
    void preAllocateOnStartAndEvictOnShutdownNewLogFile() throws IOException {
        CapturingNativeAccess capturingNativeAccess = new CapturingNativeAccess();
        LogFilesBuilder.builder((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem).withTransactionIdStore(this.transactionIdStore).withLogVersionRepository(this.logVersionRepository).withLogEntryReader(TestLogEntryReader.logEntryReader()).withStoreId(StoreId.UNKNOWN).withNativeAccess((NativeAccess)capturingNativeAccess).build();
        this.startStop(capturingNativeAccess, this.life);
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)capturingNativeAccess.getPreallocateCounter());
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)capturingNativeAccess.getEvictionCounter());
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)capturingNativeAccess.getAdviseCounter());
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)capturingNativeAccess.getKeepCounter());
    }

    @Test
    void adviseOnStartAndEvictOnShutdownExistingLogFile() throws IOException {
        CapturingNativeAccess capturingNativeAccess = new CapturingNativeAccess();
        this.startStop(capturingNativeAccess, this.life);
        capturingNativeAccess.reset();
        this.startStop(capturingNativeAccess, new LifeSupport());
        org.junit.jupiter.api.Assertions.assertEquals((int)0, (int)capturingNativeAccess.getPreallocateCounter());
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)capturingNativeAccess.getEvictionCounter());
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)capturingNativeAccess.getAdviseCounter());
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)capturingNativeAccess.getKeepCounter());
    }

    @Test
    void shouldOpenInFreshDirectoryAndFinallyAddHeader() throws Exception {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        this.life.shutdown();
        Path file = LogFilesBuilder.logFilesBasedOnlyBuilder((Path)this.databaseLayout.getTransactionLogsDirectory(), (FileSystemAbstraction)this.fileSystem).withLogEntryReader(TestLogEntryReader.logEntryReader()).build().getLogFile().getLogFileForVersion(1L);
        LogHeader header = LogHeaderReader.readLogHeader((FileSystemAbstraction)this.fileSystem, (Path)file, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
        org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)header.getLogVersion());
        org.junit.jupiter.api.Assertions.assertEquals((long)2L, (long)header.getLastCommittedTxId());
    }

    @Test
    void shouldWriteSomeDataIntoTheLog() throws Exception {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        LogFile logFile = logFiles.getLogFile();
        TransactionLogWriter transactionLogWriter = logFile.getTransactionLogWriter();
        FlushablePositionAwareChecksumChannel channel = transactionLogWriter.getChannel();
        LogPosition currentPosition = transactionLogWriter.getCurrentPosition();
        int intValue = 45;
        long longValue = 4854587L;
        channel.putInt(intValue);
        channel.putLong(longValue);
        logFile.flush();
        try (ReadableLogChannel reader = logFile.getReader(currentPosition);){
            org.junit.jupiter.api.Assertions.assertEquals((int)intValue, (int)reader.getInt());
            org.junit.jupiter.api.Assertions.assertEquals((long)longValue, (long)reader.getLong());
        }
    }

    @Test
    void shouldReadOlderLogs() throws Exception {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        LogFile logFile = logFiles.getLogFile();
        TransactionLogWriter logWriter = logFile.getTransactionLogWriter();
        FlushablePositionAwareChecksumChannel writer = logWriter.getChannel();
        LogPosition position1 = logWriter.getCurrentPosition();
        int intValue = 45;
        long longValue = 4854587L;
        byte[] someBytes = TransactionLogFileTest.someBytes(40);
        writer.putInt(intValue);
        writer.putLong(longValue);
        writer.put(someBytes, someBytes.length);
        logFile.flush();
        LogPosition position2 = logWriter.getCurrentPosition();
        long longValue2 = 123456789L;
        writer.putLong(longValue2);
        writer.put(someBytes, someBytes.length);
        logFile.flush();
        try (ReadableLogChannel reader = logFile.getReader(position1);){
            org.junit.jupiter.api.Assertions.assertEquals((int)intValue, (int)reader.getInt());
            org.junit.jupiter.api.Assertions.assertEquals((long)longValue, (long)reader.getLong());
            org.junit.jupiter.api.Assertions.assertArrayEquals((byte[])someBytes, (byte[])TransactionLogFileTest.readBytes((ReadableChannel)reader, 40));
        }
        reader = logFile.getReader(position2);
        try {
            org.junit.jupiter.api.Assertions.assertEquals((long)longValue2, (long)reader.getLong());
            org.junit.jupiter.api.Assertions.assertArrayEquals((byte[])someBytes, (byte[])TransactionLogFileTest.readBytes((ReadableChannel)reader, 40));
        }
        finally {
            if (reader != null) {
                reader.close();
            }
        }
    }

    @Test
    void shouldVisitLogFile() throws Exception {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        LogFile logFile = logFiles.getLogFile();
        TransactionLogWriter transactionLogWriter = logFile.getTransactionLogWriter();
        FlushablePositionAwareChecksumChannel writer = transactionLogWriter.getChannel();
        LogPosition position = transactionLogWriter.getCurrentPosition();
        for (int i = 0; i < 5; ++i) {
            writer.put((byte)i);
        }
        logFile.flush();
        AtomicBoolean called = new AtomicBoolean();
        logFile.accept(channel -> {
            for (int i = 0; i < 5; ++i) {
                org.junit.jupiter.api.Assertions.assertEquals((byte)((byte)i), (byte)channel.get());
            }
            called.set(true);
            return true;
        }, position);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)called.get());
    }

    @Test
    void shouldCloseChannelInFailedAttemptToReadHeaderAfterOpen() throws Exception {
        FileSystemAbstraction fs = (FileSystemAbstraction)Mockito.mock(FileSystemAbstraction.class);
        LogFiles logFiles = LogFilesBuilder.builder((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)fs).withTransactionIdStore(this.transactionIdStore).withLogVersionRepository(this.logVersionRepository).withLogEntryReader(TestLogEntryReader.logEntryReader()).build();
        int logVersion = 0;
        Path logFile = logFiles.getLogFile().getLogFileForVersion((long)logVersion);
        StoreChannel channel = (StoreChannel)Mockito.mock(StoreChannel.class);
        Mockito.when((Object)channel.read((ByteBuffer)ArgumentMatchers.any(ByteBuffer.class))).thenReturn((Object)32);
        Mockito.when((Object)fs.fileExists(logFile)).thenReturn((Object)true);
        Mockito.when((Object)fs.read((Path)ArgumentMatchers.eq((Object)logFile))).thenReturn((Object)channel);
        org.junit.jupiter.api.Assertions.assertThrows(IncompleteLogHeaderException.class, () -> logFiles.getLogFile().openForVersion((long)logVersion));
        ((StoreChannel)Mockito.verify((Object)channel)).close();
    }

    @Test
    void shouldSuppressFailureToCloseChannelInFailedAttemptToReadHeaderAfterOpen() throws Exception {
        FileSystemAbstraction fs = (FileSystemAbstraction)Mockito.mock(FileSystemAbstraction.class);
        LogFiles logFiles = LogFilesBuilder.builder((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)fs).withTransactionIdStore(this.transactionIdStore).withLogVersionRepository(this.logVersionRepository).withLogEntryReader(TestLogEntryReader.logEntryReader()).build();
        int logVersion = 0;
        Path logFile = logFiles.getLogFile().getLogFileForVersion((long)logVersion);
        StoreChannel channel = (StoreChannel)Mockito.mock(StoreChannel.class);
        Mockito.when((Object)channel.read((ByteBuffer)ArgumentMatchers.any(ByteBuffer.class))).thenReturn((Object)32);
        Mockito.when((Object)fs.fileExists(logFile)).thenReturn((Object)true);
        Mockito.when((Object)fs.read((Path)ArgumentMatchers.eq((Object)logFile))).thenReturn((Object)channel);
        ((StoreChannel)Mockito.doThrow(IOException.class).when((Object)channel)).close();
        IncompleteLogHeaderException exception = (IncompleteLogHeaderException)org.junit.jupiter.api.Assertions.assertThrows(IncompleteLogHeaderException.class, () -> logFiles.getLogFile().openForVersion((long)logVersion));
        ((StoreChannel)Mockito.verify((Object)channel)).close();
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)exception.getSuppressed().length);
        org.junit.jupiter.api.Assertions.assertTrue((boolean)(exception.getSuppressed()[0] instanceof IOException));
    }

    @Test
    void closeChannelThrowExceptionOnAttemptToAppendTransactionLogRecords() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        LogFile logFile = logFiles.getLogFile();
        FlushablePositionAwareChecksumChannel channel = logFile.getTransactionLogWriter().getChannel();
        this.life.shutdown();
        org.junit.jupiter.api.Assertions.assertThrows(Throwable.class, () -> channel.put((byte)7));
        org.junit.jupiter.api.Assertions.assertThrows(Throwable.class, () -> channel.putInt(7));
        org.junit.jupiter.api.Assertions.assertThrows(Throwable.class, () -> channel.putLong(7L));
        org.junit.jupiter.api.Assertions.assertThrows(Throwable.class, () -> channel.putDouble(7.0));
        org.junit.jupiter.api.Assertions.assertThrows(Throwable.class, () -> channel.putFloat(7.0f));
        org.junit.jupiter.api.Assertions.assertThrows(Throwable.class, () -> channel.putShort((short)7));
        org.junit.jupiter.api.Assertions.assertThrows(Throwable.class, () -> channel.put(new byte[]{1, 2, 3}, 3));
        org.junit.jupiter.api.Assertions.assertThrows(IllegalStateException.class, () -> ((LogFile)logFile).flush());
    }

    @Test
    void shouldForceLogChannel() throws Throwable {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        LogFile logFile = logFiles.getLogFile();
        CapturingStoreChannel capturingChannel = this.wrappingFileSystem.getCapturingChannel();
        int flushesBefore = capturingChannel.getFlushCounter().get();
        int writesBefore = capturingChannel.getWriteAllCounter().get();
        logFile.forceAfterAppend((LogForceEvents)LogAppendEvent.NULL);
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)(capturingChannel.getFlushCounter().get() - flushesBefore));
        org.junit.jupiter.api.Assertions.assertEquals((int)1, (int)(capturingChannel.getWriteAllCounter().get() - writesBefore));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void shouldBatchUpMultipleWaitingForceRequests() throws Throwable {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        LogFile logFile = logFiles.getLogFile();
        CapturingStoreChannel capturingChannel = this.wrappingFileSystem.getCapturingChannel();
        int flushesBefore = capturingChannel.getFlushCounter().get();
        int writesBefore = capturingChannel.getWriteAllCounter().get();
        ReentrantLock writeAllLock = capturingChannel.getWriteAllLock();
        writeAllLock.lock();
        int executors = 10;
        ExecutorService executorService = Executors.newFixedThreadPool(executors);
        try {
            List futures = Stream.iterate(0, i -> i + 1).limit(executors).map(v -> executorService.submit(() -> logFile.forceAfterAppend((LogForceEvents)LogAppendEvent.NULL))).collect(Collectors.toList());
            while (!writeAllLock.hasQueuedThreads()) {
                LockSupport.parkNanos(100L);
            }
            writeAllLock.unlock();
            Assertions.assertThat(futures).hasSize(executors);
            Futures.getAll(futures);
        }
        finally {
            if (writeAllLock.isLocked()) {
                writeAllLock.unlock();
            }
            executorService.shutdownNow();
        }
        Assertions.assertThat((int)(capturingChannel.getFlushCounter().get() - flushesBefore)).isLessThanOrEqualTo(executors);
        Assertions.assertThat((int)(capturingChannel.getWriteAllCounter().get() - writesBefore)).isLessThanOrEqualTo(executors);
    }

    @Test
    void combineLogFilesFromMultipleLocationsNonOverlappingFiles() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        Path additionalSource = this.testDirectory.directory("another");
        this.createFile(additionalSource, 2L);
        this.createFile(additionalSource, 3L);
        this.createFile(additionalSource, 4L);
        LogFile logFile = logFiles.getLogFile();
        org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)logFile.getHighestLogVersion());
        logFile.combine(additionalSource);
        org.junit.jupiter.api.Assertions.assertEquals((long)4L, (long)logFile.getHighestLogVersion());
        Assertions.assertThat(Arrays.stream(logFile.getMatchedFiles()).map(path -> path.getFileName().toString())).contains((Object[])new String[]{"neostore.transaction.db.1", "neostore.transaction.db.2", "neostore.transaction.db.3", "neostore.transaction.db.4"});
    }

    @Test
    void combineLogFilesFromMultipleLocationsOverlappingFiles() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        Path additionalSource = this.testDirectory.directory("another");
        this.createFile(additionalSource, 0L);
        this.createFile(additionalSource, 1L);
        this.createFile(additionalSource, 2L);
        LogFile logFile = logFiles.getLogFile();
        org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)logFile.getHighestLogVersion());
        logFile.combine(additionalSource);
        org.junit.jupiter.api.Assertions.assertEquals((long)4L, (long)logFile.getHighestLogVersion());
        Assertions.assertThat(Arrays.stream(logFile.getMatchedFiles()).map(path -> path.getFileName().toString())).contains((Object[])new String[]{"neostore.transaction.db.1", "neostore.transaction.db.2", "neostore.transaction.db.3", "neostore.transaction.db.4"});
    }

    @Test
    void combineShouldPreserveOrder() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        Path additionalSource = this.testDirectory.directory("another");
        int numberOfAdditionalFile = 20;
        for (int i = 0; i < numberOfAdditionalFile; ++i) {
            this.createFile(additionalSource, i, i);
        }
        LogFile logFile = logFiles.getLogFile();
        org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)logFile.getHighestLogVersion());
        logFile.combine(additionalSource);
        org.junit.jupiter.api.Assertions.assertEquals((long)(numberOfAdditionalFile + 1), (long)logFile.getHighestLogVersion());
        for (int i = 2; i < numberOfAdditionalFile + 2; ++i) {
            int expectedCommitIdx = i - 2;
            LogHeader header = LogHeaderReader.readLogHeader((FileSystemAbstraction)this.fileSystem, (Path)logFile.getLogFileForVersion((long)i), (MemoryTracker)EmptyMemoryTracker.INSTANCE);
            Long lastCommittedTxId = header.getLastCommittedTxId();
            ((AbstractLongAssert)Assertions.assertThat((Long)lastCommittedTxId).withFailMessage("File %s should have commit idx %s instead of %s", new Object[]{logFile.getLogFileForVersion((long)i), expectedCommitIdx, lastCommittedTxId})).isEqualTo((long)expectedCommitIdx);
        }
    }

    @Test
    void combineLogFilesFromMultipleLocationsNonSequentialFiles() throws IOException {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        Path additionalSource1 = this.testDirectory.directory("another");
        this.createFile(additionalSource1, 0L);
        this.createFile(additionalSource1, 6L);
        this.createFile(additionalSource1, 8L);
        Path additionalSource2 = this.testDirectory.directory("another2");
        this.createFile(additionalSource2, 10L);
        this.createFile(additionalSource2, 26L);
        this.createFile(additionalSource2, 38L);
        LogFile logFile = logFiles.getLogFile();
        org.junit.jupiter.api.Assertions.assertEquals((long)1L, (long)logFile.getHighestLogVersion());
        logFile.combine(additionalSource1);
        logFile.combine(additionalSource2);
        org.junit.jupiter.api.Assertions.assertEquals((long)7L, (long)logFile.getHighestLogVersion());
        Assertions.assertThat(Arrays.stream(logFile.getMatchedFiles()).map(path -> path.getFileName().toString())).contains((Object[])new String[]{"neostore.transaction.db.1", "neostore.transaction.db.2", "neostore.transaction.db.3", "neostore.transaction.db.4", "neostore.transaction.db.5", "neostore.transaction.db.6", "neostore.transaction.db.7"});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void shouldWaitForOngoingForceToCompleteBeforeForcingAgain() throws Throwable {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        LogFile logFile = logFiles.getLogFile();
        CapturingStoreChannel capturingChannel = this.wrappingFileSystem.getCapturingChannel();
        ReentrantLock writeAllLock = capturingChannel.getWriteAllLock();
        int flushesBefore = capturingChannel.getFlushCounter().get();
        int writesBefore = capturingChannel.getWriteAllCounter().get();
        writeAllLock.lock();
        int executors = 10;
        ExecutorService executorService = Executors.newFixedThreadPool(executors);
        try {
            Future<Boolean> future = executorService.submit(() -> logFile.forceAfterAppend((LogForceEvents)LogAppendEvent.NULL));
            while (!writeAllLock.hasQueuedThreads()) {
                LockSupport.parkNanos(100L);
            }
            writeAllLock.unlock();
            Future<Boolean> future2 = executorService.submit(() -> logFile.forceAfterAppend((LogForceEvents)LogAppendEvent.NULL));
            Futures.getAll(List.of(future, future2));
        }
        finally {
            if (writeAllLock.isLocked()) {
                writeAllLock.unlock();
            }
            executorService.shutdownNow();
        }
        Assertions.assertThat((int)(capturingChannel.getWriteAllCounter().get() - writesBefore)).isEqualTo(2);
        Assertions.assertThat((int)(capturingChannel.getFlushCounter().get() - flushesBefore)).isEqualTo(2);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void logFilesExternalReadersRegistration() throws IOException, ExecutionException {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        LogFile logFile = logFiles.getLogFile();
        logFile.rotate();
        logFile.rotate();
        logFile.rotate();
        org.junit.jupiter.api.Assertions.assertEquals((long)4L, (long)logFile.getHighestLogVersion());
        MutableLongObjectMap channelMap = LongObjectMaps.mutable.empty();
        channelMap.put(1L, (Object)logFile.openForVersion(1L));
        channelMap.put(2L, (Object)logFile.openForVersion(2L));
        channelMap.put(3L, (Object)logFile.openForVersion(3L));
        ExecutorService registerCalls = Executors.newFixedThreadPool(5);
        ArrayList futures = new ArrayList(10);
        try {
            for (int i = 0; i < 10; ++i) {
                futures.add(registerCalls.submit(() -> logFile.registerExternalReaders((LongObjectMap)channelMap)));
            }
        }
        finally {
            registerCalls.shutdown();
        }
        Futures.getAll(futures);
        ConcurrentMap externalFileReaders = ((TransactionLogFile)logFile).getExternalFileReaders();
        try {
            Assertions.assertThat((Map)externalFileReaders).containsOnlyKeys((Object[])new Long[]{1L, 2L, 3L});
            for (Map.Entry entry : externalFileReaders.entrySet()) {
                List channels = (List)entry.getValue();
                Assertions.assertThat((List)channels).hasSize(10);
                StoreChannel exampleChannel = (StoreChannel)channels.get(0);
                for (StoreChannel channel : channels) {
                    org.junit.jupiter.api.Assertions.assertEquals((Object)channel, (Object)exampleChannel);
                }
            }
        }
        finally {
            logFile.terminateExternalReaders(3L);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void terminateLogFilesExternalReaders() throws IOException, ExecutionException {
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        LogFile logFile = logFiles.getLogFile();
        logFile.rotate();
        logFile.rotate();
        logFile.rotate();
        logFile.rotate();
        org.junit.jupiter.api.Assertions.assertEquals((long)5L, (long)logFile.getHighestLogVersion());
        MutableLongObjectMap channelMap = LongObjectMaps.mutable.empty();
        channelMap.put(1L, (Object)logFile.openForVersion(1L));
        channelMap.put(2L, (Object)logFile.openForVersion(2L));
        channelMap.put(3L, (Object)logFile.openForVersion(3L));
        channelMap.put(4L, (Object)logFile.openForVersion(4L));
        ExecutorService registerCalls = Executors.newCachedThreadPool();
        ArrayList futures = new ArrayList(10);
        try {
            for (int i = 0; i < 10; ++i) {
                futures.add(registerCalls.submit(() -> logFile.registerExternalReaders((LongObjectMap)channelMap)));
            }
        }
        finally {
            registerCalls.shutdown();
        }
        Futures.getAll(futures);
        logFile.terminateExternalReaders(3L);
        try {
            ConcurrentMap externalFileReaders = ((TransactionLogFile)logFile).getExternalFileReaders();
            Assertions.assertThat((Map)externalFileReaders).containsOnlyKeys((Object[])new Long[]{4L});
            for (Map.Entry entry : externalFileReaders.entrySet()) {
                List channels = (List)entry.getValue();
                Assertions.assertThat((List)channels).hasSize(10);
                StoreChannel exampleChannel = (StoreChannel)channels.get(0);
                for (StoreChannel channel : channels) {
                    org.junit.jupiter.api.Assertions.assertEquals((Object)channel, (Object)exampleChannel);
                }
            }
        }
        finally {
            logFile.terminateExternalReaders(4L);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void registerUnregisterLogFilesExternalReaders() throws IOException, ExecutionException {
        int i;
        LogFiles logFiles = this.buildLogFiles();
        this.life.start();
        this.life.add((Lifecycle)logFiles);
        LogFile logFile = logFiles.getLogFile();
        logFile.rotate();
        logFile.rotate();
        logFile.rotate();
        org.junit.jupiter.api.Assertions.assertEquals((long)4L, (long)logFile.getHighestLogVersion());
        MutableLongObjectMap channelMap = LongObjectMaps.mutable.empty();
        PhysicalLogVersionedStoreChannel channel1 = logFile.openForVersion(1L);
        PhysicalLogVersionedStoreChannel channel2 = logFile.openForVersion(2L);
        channelMap.put(1L, (Object)channel1);
        channelMap.put(2L, (Object)channel2);
        ExecutorService registerCalls = Executors.newCachedThreadPool();
        ArrayList futures = new ArrayList(10);
        try {
            for (int i2 = 0; i2 < 10; ++i2) {
                futures.add(registerCalls.submit(() -> logFile.registerExternalReaders((LongObjectMap)channelMap)));
            }
        }
        finally {
            registerCalls.shutdown();
        }
        Futures.getAll(futures);
        ConcurrentMap externalFileReaders = ((TransactionLogFile)logFile).getExternalFileReaders();
        Assertions.assertThat((Map)externalFileReaders).containsOnlyKeys((Object[])new Long[]{1L, 2L});
        for (i = 0; i < 100; ++i) {
            logFile.unregisterExternalReader(1L, (StoreChannel)channel1);
        }
        Assertions.assertThat((Map)externalFileReaders).containsOnlyKeys((Object[])new Long[]{2L});
        for (i = 0; i < 19; ++i) {
            logFile.unregisterExternalReader(2L, (StoreChannel)channel1);
        }
        Assertions.assertThat((Map)externalFileReaders).containsOnlyKeys((Object[])new Long[]{2L});
        for (i = 0; i < 9; ++i) {
            logFile.unregisterExternalReader(2L, (StoreChannel)channel2);
        }
        Assertions.assertThat((Map)externalFileReaders).containsOnlyKeys((Object[])new Long[]{2L});
        Assertions.assertThat((List)((List)externalFileReaders.get(2L))).hasSize(1);
        logFile.unregisterExternalReader(2L, (StoreChannel)channel2);
        Assertions.assertThat((Map)externalFileReaders).isEmpty();
    }

    private static byte[] readBytes(ReadableChannel reader, int length) throws IOException {
        byte[] result = new byte[length];
        reader.get(result, length);
        return result;
    }

    private LogFiles buildLogFiles() throws IOException {
        return LogFilesBuilder.builder((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.wrappingFileSystem).withRotationThreshold(this.rotationThreshold).withTransactionIdStore(this.transactionIdStore).withLogVersionRepository(this.logVersionRepository).withLogEntryReader(TestLogEntryReader.logEntryReader()).withStoreId(StoreId.UNKNOWN).build();
    }

    private static byte[] someBytes(int length) {
        byte[] result = new byte[length];
        for (int i = 0; i < length; ++i) {
            result[i] = (byte)(i % 5);
        }
        return result;
    }

    private void startStop(CapturingNativeAccess capturingNativeAccess, LifeSupport lifeSupport) throws IOException {
        LogFiles logFiles = LogFilesBuilder.builder((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem).withTransactionIdStore(this.transactionIdStore).withLogVersionRepository(this.logVersionRepository).withLogEntryReader(TestLogEntryReader.logEntryReader()).withStoreId(StoreId.UNKNOWN).withNativeAccess((NativeAccess)capturingNativeAccess).build();
        lifeSupport.add((Lifecycle)logFiles);
        lifeSupport.start();
        lifeSupport.shutdown();
    }

    private void createFile(Path filePath, long version, long lastCommittedTxId) throws IOException {
        TransactionLogFilesHelper filesHelper = new TransactionLogFilesHelper(this.fileSystem, filePath);
        try (StoreChannel storeChannel = this.fileSystem.write(filesHelper.getLogFileForVersion(version));){
            LogHeaderWriter.writeLogHeader((StoreChannel)storeChannel, (LogHeader)new LogHeader(version, lastCommittedTxId, StoreId.UNKNOWN), (MemoryTracker)EmptyMemoryTracker.INSTANCE);
        }
    }

    private void createFile(Path filePath, long version) throws IOException {
        this.createFile(filePath, version, 1L);
    }

    private static class CapturingStoreChannel
    extends DelegatingStoreChannel<StoreChannel> {
        private final AtomicInteger writeAllCounter = new AtomicInteger();
        private final AtomicInteger flushCounter = new AtomicInteger();
        private final ReentrantLock writeAllLock = new ReentrantLock();

        private CapturingStoreChannel(StoreChannel delegate) {
            super(delegate);
        }

        public void writeAll(ByteBuffer src) throws IOException {
            this.writeAllLock.lock();
            try {
                this.writeAllCounter.incrementAndGet();
                super.writeAll(src);
            }
            finally {
                this.writeAllLock.unlock();
            }
        }

        public void flush() throws IOException {
            this.flushCounter.incrementAndGet();
            super.flush();
        }

        public ReentrantLock getWriteAllLock() {
            return this.writeAllLock;
        }

        public AtomicInteger getWriteAllCounter() {
            return this.writeAllCounter;
        }

        public AtomicInteger getFlushCounter() {
            return this.flushCounter;
        }
    }

    private static class CapturingChannelFileSystem
    extends DelegatingFileSystemAbstraction {
        private CapturingStoreChannel capturingChannel;

        CapturingChannelFileSystem(FileSystemAbstraction fs) {
            super(fs);
        }

        public StoreChannel write(Path fileName) throws IOException {
            if (fileName.toString().contains("neostore.transaction.db")) {
                this.capturingChannel = new CapturingStoreChannel(super.write(fileName));
                return this.capturingChannel;
            }
            return super.write(fileName);
        }

        public CapturingStoreChannel getCapturingChannel() {
            return this.capturingChannel;
        }
    }

    private static class CapturingNativeAccess
    implements NativeAccess {
        private int evictionCounter;
        private int adviseCounter;
        private int preallocateCounter;
        private int keepCounter;

        private CapturingNativeAccess() {
        }

        public boolean isAvailable() {
            return true;
        }

        public NativeCallResult tryEvictFromCache(int fd) {
            ++this.evictionCounter;
            return NativeCallResult.SUCCESS;
        }

        public NativeCallResult tryAdviseSequentialAccess(int fd) {
            ++this.adviseCounter;
            return NativeCallResult.SUCCESS;
        }

        public NativeCallResult tryAdviseToKeepInCache(int fd) {
            ++this.keepCounter;
            return NativeCallResult.SUCCESS;
        }

        public NativeCallResult tryPreallocateSpace(int fd, long bytes) {
            ++this.preallocateCounter;
            return NativeCallResult.SUCCESS;
        }

        public ErrorTranslator errorTranslator() {
            return callResult -> false;
        }

        public String describe() {
            return "Test only";
        }

        public int getEvictionCounter() {
            return this.evictionCounter;
        }

        public int getAdviseCounter() {
            return this.adviseCounter;
        }

        public int getKeepCounter() {
            return this.keepCounter;
        }

        public int getPreallocateCounter() {
            return this.preallocateCounter;
        }

        public void reset() {
            this.adviseCounter = 0;
            this.evictionCounter = 0;
            this.preallocateCounter = 0;
            this.keepCounter = 0;
        }
    }
}

