/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction.log;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Stream;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.Mockito;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.DefaultFileSystemAbstraction;
import org.neo4j.io.fs.PhysicalFlushableChannel;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.fs.StoreFileChannel;
import org.neo4j.io.memory.ByteBuffers;
import org.neo4j.io.memory.HeapScopedBuffer;
import org.neo4j.io.memory.NativeScopedBuffer;
import org.neo4j.io.memory.ScopedBuffer;
import org.neo4j.kernel.impl.api.tracer.DefaultTracer;
import org.neo4j.kernel.impl.transaction.log.LogPosition;
import org.neo4j.kernel.impl.transaction.log.LogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.PhysicalFlushableLogChannel;
import org.neo4j.kernel.impl.transaction.log.PhysicalLogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.PositionAwarePhysicalFlushableChecksumChannel;
import org.neo4j.kernel.impl.transaction.log.files.ChannelNativeAccessor;
import org.neo4j.kernel.impl.transaction.log.files.LogFileChannelNativeAccessor;
import org.neo4j.kernel.impl.transaction.tracing.DatabaseTracer;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.LocalMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.testdirectory.TestDirectoryExtension;
import org.neo4j.test.utils.TestDirectory;

@TestDirectoryExtension
class PhysicalFlushableChannelTest {
    @Inject
    private DefaultFileSystemAbstraction fileSystem;
    @Inject
    private TestDirectory directory;
    private final LogFileChannelNativeAccessor nativeChannelAccessor = (LogFileChannelNativeAccessor)Mockito.mock(LogFileChannelNativeAccessor.class);
    private final DatabaseTracer databaseTracer = DatabaseTracer.NULL;

    PhysicalFlushableChannelTest() {
    }

    @Test
    void countChannelFlushEvents() throws IOException {
        Path path = this.directory.homePath().resolve("countChannelFlushEvents");
        StoreFileChannel storeChannel = this.fileSystem.write(path);
        DefaultTracer databaseTracer = new DefaultTracer();
        try (PhysicalLogVersionedStoreChannel channel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, path, (ChannelNativeAccessor)this.nativeChannelAccessor, (DatabaseTracer)databaseTracer, true);){
            channel.flush();
            channel.flush();
            channel.flush();
        }
        org.junit.jupiter.api.Assertions.assertEquals((long)3L, (long)databaseTracer.numberOfFlushes());
    }

    @Test
    void rawChannelDoesNotEvictDataOnClose() throws IOException {
        Path rawPath = this.directory.homePath().resolve("fileRaw");
        StoreFileChannel storeChannel = this.fileSystem.write(rawPath);
        PhysicalLogVersionedStoreChannel channel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, rawPath, (ChannelNativeAccessor)this.nativeChannelAccessor, this.databaseTracer, true);
        channel.close();
        Mockito.verifyNoInteractions((Object[])new Object[]{this.nativeChannelAccessor});
    }

    @Test
    void shouldBeAbleToWriteSmallNumberOfBytes() throws IOException {
        byte[] bytes;
        Path firstFile = this.directory.homePath().resolve("file1");
        StoreFileChannel storeChannel = this.fileSystem.write(firstFile);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, firstFile, (ChannelNativeAccessor)this.nativeChannelAccessor, this.databaseTracer);
        int length = 26145;
        try (PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel, (ScopedBuffer)new HeapScopedBuffer(100, (MemoryTracker)EmptyMemoryTracker.INSTANCE));){
            bytes = PhysicalFlushableChannelTest.generateBytes(length);
            channel.put(bytes, length);
        }
        byte[] writtenBytes = Files.readAllBytes(firstFile);
        org.junit.jupiter.api.Assertions.assertArrayEquals((byte[])bytes, (byte[])writtenBytes);
    }

    @Test
    void shouldBeAbleToWriteValuesGreaterThanHalfTheBufferSize() throws IOException {
        byte[] bytes;
        Path firstFile = this.directory.homePath().resolve("file1");
        StoreFileChannel storeChannel = this.fileSystem.write(firstFile);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, firstFile, (ChannelNativeAccessor)this.nativeChannelAccessor, this.databaseTracer);
        int length = 262145;
        try (PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel, (MemoryTracker)EmptyMemoryTracker.INSTANCE);){
            bytes = PhysicalFlushableChannelTest.generateBytes(length);
            channel.put(bytes, length);
        }
        byte[] writtenBytes = Files.readAllBytes(firstFile);
        org.junit.jupiter.api.Assertions.assertArrayEquals((byte[])bytes, (byte[])writtenBytes);
    }

    @Test
    void releaseBufferMemoryOnClose() throws IOException {
        LocalMemoryTracker memoryTracker = new LocalMemoryTracker();
        Path firstFile = this.directory.homePath().resolve("file2");
        StoreFileChannel storeChannel = this.fileSystem.write(firstFile);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, firstFile, (ChannelNativeAccessor)this.nativeChannelAccessor, this.databaseTracer);
        Assertions.assertThat((long)memoryTracker.estimatedHeapMemory()).isZero();
        Assertions.assertThat((long)memoryTracker.usedNativeMemory()).isZero();
        try (PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel, (MemoryTracker)memoryTracker);){
            channel.put((byte)1);
            Assertions.assertThat((long)memoryTracker.usedNativeMemory()).isZero();
            Assertions.assertThat((long)memoryTracker.estimatedHeapMemory()).isGreaterThan(0L);
        }
        Assertions.assertThat((long)memoryTracker.estimatedHeapMemory()).isZero();
        Assertions.assertThat((long)memoryTracker.usedNativeMemory()).isZero();
    }

    @Test
    void shouldBeAbleToWriteValuesGreaterThanTheBufferSize() throws IOException {
        byte[] bytes;
        Path firstFile = this.directory.homePath().resolve("file1");
        StoreFileChannel storeChannel = this.fileSystem.write(firstFile);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, firstFile, (ChannelNativeAccessor)this.nativeChannelAccessor, this.databaseTracer);
        int length = 1000000;
        try (PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel, (MemoryTracker)EmptyMemoryTracker.INSTANCE);){
            bytes = PhysicalFlushableChannelTest.generateBytes(length);
            channel.put(bytes, length);
        }
        byte[] writtenBytes = Files.readAllBytes(firstFile);
        org.junit.jupiter.api.Assertions.assertArrayEquals((byte[])bytes, (byte[])writtenBytes);
    }

    @MethodSource(value={"bytesToChannelParameters"})
    @ParameterizedTest
    void writeBytesOverChannel(int length, ScopedBuffer buffer) throws IOException {
        Path file = this.directory.homePath().resolve("fileWithBytes");
        StoreFileChannel storeChannel = this.fileSystem.write(file);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, file, (ChannelNativeAccessor)this.nativeChannelAccessor, this.databaseTracer);
        byte[] bytes = PhysicalFlushableChannelTest.generateBytes(length);
        try (PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel, buffer);){
            channel.put(bytes, length);
        }
        byte[] writtenBytes = Files.readAllBytes(file);
        org.junit.jupiter.api.Assertions.assertArrayEquals((byte[])bytes, (byte[])writtenBytes);
    }

    @Test
    void shouldWriteThroughRotation() throws Exception {
        Path firstFile = this.directory.homePath().resolve("file1");
        Path secondFile = this.directory.homePath().resolve("file2");
        StoreFileChannel storeChannel = this.fileSystem.write(firstFile);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, firstFile, (ChannelNativeAccessor)this.nativeChannelAccessor, this.databaseTracer);
        PhysicalFlushableLogChannel channel = new PhysicalFlushableLogChannel((StoreChannel)versionedStoreChannel, (ScopedBuffer)new HeapScopedBuffer(100, (MemoryTracker)EmptyMemoryTracker.INSTANCE));
        byte byteValue = 4;
        short shortValue = 10;
        int intValue = 3545;
        long longValue = 45849589L;
        float floatValue = 45849.332f;
        double doubleValue = 4.58493343E8;
        byte[] byteArrayValue = new byte[]{1, 4, 2, 5, 3, 6};
        channel.put(byteValue);
        channel.putShort(shortValue);
        channel.putInt(intValue);
        channel.putLong(longValue);
        channel.prepareForFlush().flush();
        channel.close();
        storeChannel = this.fileSystem.write(secondFile);
        channel.setChannel((StoreChannel)new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 2L, -1, secondFile, (ChannelNativeAccessor)this.nativeChannelAccessor, this.databaseTracer));
        channel.putFloat(floatValue);
        channel.putDouble(doubleValue);
        channel.put(byteArrayValue, byteArrayValue.length);
        channel.close();
        ByteBuffer firstFileContents = this.readFile(firstFile);
        org.junit.jupiter.api.Assertions.assertEquals((byte)byteValue, (byte)firstFileContents.get());
        org.junit.jupiter.api.Assertions.assertEquals((short)shortValue, (short)firstFileContents.getShort());
        org.junit.jupiter.api.Assertions.assertEquals((int)intValue, (int)firstFileContents.getInt());
        org.junit.jupiter.api.Assertions.assertEquals((long)longValue, (long)firstFileContents.getLong());
        ByteBuffer secondFileContents = this.readFile(secondFile);
        org.junit.jupiter.api.Assertions.assertEquals((float)floatValue, (float)secondFileContents.getFloat(), (float)0.001f);
        org.junit.jupiter.api.Assertions.assertEquals((double)doubleValue, (double)secondFileContents.getDouble(), (double)0.001);
        byte[] readByteArray = new byte[byteArrayValue.length];
        secondFileContents.get(readByteArray);
        org.junit.jupiter.api.Assertions.assertArrayEquals((byte[])byteArrayValue, (byte[])readByteArray);
    }

    @Test
    void shouldSeeCorrectPositionEvenBeforeEmptyingDataIntoChannel() throws Exception {
        Path file = this.directory.homePath().resolve("file");
        StoreFileChannel storeChannel = this.fileSystem.write(file);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, file, (ChannelNativeAccessor)this.nativeChannelAccessor, this.databaseTracer);
        try (PositionAwarePhysicalFlushableChecksumChannel channel = new PositionAwarePhysicalFlushableChecksumChannel((LogVersionedStoreChannel)versionedStoreChannel, (ScopedBuffer)new NativeScopedBuffer(1024, (MemoryTracker)EmptyMemoryTracker.INSTANCE));){
            LogPosition initialPosition = channel.getCurrentPosition();
            channel.putLong(67L);
            channel.putInt(1234);
            LogPosition positionAfterSomeData = channel.getCurrentPosition();
            org.junit.jupiter.api.Assertions.assertEquals((long)12L, (long)(positionAfterSomeData.getByteOffset() - initialPosition.getByteOffset()));
        }
    }

    @Test
    void shouldThrowIllegalStateExceptionAfterClosed() throws Exception {
        Path file = this.directory.homePath().resolve("file");
        StoreFileChannel storeChannel = this.fileSystem.write(file);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, file, (ChannelNativeAccessor)this.nativeChannelAccessor, this.databaseTracer);
        PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
        channel.close();
        storeChannel.close();
        channel.put((byte)0);
        org.junit.jupiter.api.Assertions.assertThrows(IllegalStateException.class, () -> ((PhysicalFlushableChannel)channel).prepareForFlush());
    }

    @Test
    void shouldThrowClosedChannelExceptionWhenChannelUnexpectedlyClosed() throws Exception {
        Path file = this.directory.homePath().resolve("file");
        StoreFileChannel storeChannel = this.fileSystem.write(file);
        PhysicalLogVersionedStoreChannel versionedStoreChannel = new PhysicalLogVersionedStoreChannel((StoreChannel)storeChannel, 1L, -1, file, (ChannelNativeAccessor)this.nativeChannelAccessor, this.databaseTracer);
        PhysicalFlushableChannel channel = new PhysicalFlushableChannel((StoreChannel)versionedStoreChannel, (MemoryTracker)EmptyMemoryTracker.INSTANCE);
        storeChannel.close();
        channel.put((byte)0);
        org.junit.jupiter.api.Assertions.assertThrows(ClosedChannelException.class, () -> ((PhysicalFlushableChannel)channel).prepareForFlush());
    }

    private ByteBuffer readFile(Path file) throws IOException {
        try (StoreFileChannel channel = this.fileSystem.read(file);){
            ByteBuffer buffer = ByteBuffers.allocate((int)((int)channel.size()), (MemoryTracker)EmptyMemoryTracker.INSTANCE);
            channel.readAll(buffer);
            buffer.flip();
            ByteBuffer byteBuffer = buffer;
            return byteBuffer;
        }
    }

    private static Stream<Arguments> bytesToChannelParameters() {
        return Stream.of(Arguments.of((Object[])new Object[]{128, new HeapScopedBuffer(128, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{256, new HeapScopedBuffer(128, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{258, new HeapScopedBuffer(128, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{512, new HeapScopedBuffer(128, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{12, new HeapScopedBuffer(512, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{120, new HeapScopedBuffer(512, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{1200, new HeapScopedBuffer(512, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{1537, new HeapScopedBuffer(512, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{1535, new HeapScopedBuffer(512, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{(int)ByteUnit.kibiBytes((long)24L), new HeapScopedBuffer(512, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{(int)ByteUnit.kibiBytes((long)1024L), new HeapScopedBuffer(512, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{(int)ByteUnit.kibiBytes((long)5024L), new HeapScopedBuffer(512, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{(int)ByteUnit.kibiBytes((long)10024L), new HeapScopedBuffer(512, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{(int)ByteUnit.kibiBytes((long)10024L), new HeapScopedBuffer(1024, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{(int)ByteUnit.kibiBytes((long)11024L), new HeapScopedBuffer(1024, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{(int)ByteUnit.kibiBytes((long)11424L), new HeapScopedBuffer(1024, (MemoryTracker)EmptyMemoryTracker.INSTANCE)}), Arguments.of((Object[])new Object[]{(int)ByteUnit.bytes((long)ThreadLocalRandom.current().nextInt(1024, (int)ByteUnit.mebiBytes((long)100L))), new HeapScopedBuffer(ThreadLocalRandom.current().nextInt(128, (int)ByteUnit.mebiBytes((long)2L)), (MemoryTracker)EmptyMemoryTracker.INSTANCE)}));
    }

    private static byte[] generateBytes(int length) {
        Random random = new Random();
        char[] validCharacters = new char[]{'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o'};
        byte[] bytes = new byte[length];
        for (int i = 0; i < length; ++i) {
            bytes[i] = (byte)validCharacters[random.nextInt(validCharacters.length)];
        }
        return bytes;
    }
}

