/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.transaction.log.files;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Path;
import java.util.ArrayList;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.fs.StoreChannel;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.memory.ByteBuffers;
import org.neo4j.kernel.impl.api.TestCommandReaderFactory;
import org.neo4j.kernel.impl.transaction.SimpleLogVersionRepository;
import org.neo4j.kernel.impl.transaction.SimpleTransactionIdStore;
import org.neo4j.kernel.impl.transaction.log.PhysicalLogVersionedStoreChannel;
import org.neo4j.kernel.impl.transaction.log.entry.LogEntryReader;
import org.neo4j.kernel.impl.transaction.log.entry.LogHeader;
import org.neo4j.kernel.impl.transaction.log.entry.VersionAwareLogEntryReader;
import org.neo4j.kernel.impl.transaction.log.files.LogFile;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.LogFilesBuilder;
import org.neo4j.memory.EmptyMemoryTracker;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.storageengine.api.CommandReaderFactory;
import org.neo4j.storageengine.api.LogVersionRepository;
import org.neo4j.storageengine.api.StoreId;
import org.neo4j.storageengine.api.TransactionIdStore;
import org.neo4j.test.extension.Inject;
import org.neo4j.test.extension.Neo4jLayoutExtension;

@Neo4jLayoutExtension
class TransactionLogFilesTest {
    @Inject
    private FileSystemAbstraction fileSystem;
    @Inject
    private DatabaseLayout databaseLayout;

    TransactionLogFilesTest() {
    }

    @Test
    void shouldGetTheFileNameForAGivenVersion() throws Exception {
        LogFiles files = this.createLogFiles();
        int version = 12;
        Path versionFileName = files.getLogFile().getLogFileForVersion(12L);
        Path expected = TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedLogFileName(12));
        org.junit.jupiter.api.Assertions.assertEquals((Object)expected, (Object)versionFileName);
    }

    @Test
    void extractHeaderOf3_5Format() throws Exception {
        LogFiles files = this.createLogFiles();
        this.create3_5FileWithHeader(this.databaseLayout, "0");
        this.create3_5FileWithHeader(this.databaseLayout, "1", 1);
        this.create3_5FileWithHeader(this.databaseLayout, "2", 2);
        LogFile logFile = files.getLogFile();
        LogHeader logHeader = logFile.extractHeader(0L);
        org.junit.jupiter.api.Assertions.assertEquals((long)16L, (long)logHeader.getStartPosition().getByteOffset());
        org.junit.jupiter.api.Assertions.assertEquals((byte)6, (byte)logHeader.getLogFormatVersion());
        org.junit.jupiter.api.Assertions.assertEquals((long)16L, (long)logFile.extractHeader(1L).getStartPosition().getByteOffset());
        org.junit.jupiter.api.Assertions.assertEquals((long)16L, (long)logFile.extractHeader(2L).getStartPosition().getByteOffset());
    }

    @Test
    void detectEntriesIn3_5Format() throws Exception {
        LogFiles files = this.createLogFiles();
        this.create3_5FileWithHeader(this.databaseLayout, "0");
        this.create3_5FileWithHeader(this.databaseLayout, "1", 10);
        LogFile logFile = files.getLogFile();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)logFile.hasAnyEntries(0L));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)logFile.hasAnyEntries(1L));
    }

    @Test
    void shouldVisitEachLofFile() throws Throwable {
        LogFiles files = this.createLogFiles();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedLogFileName("1"))).close();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedLogFileName("some", "2"))).close();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedLogFileName("3"))).close();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, "neostore.transaction.db")).close();
        ArrayList seenFiles = new ArrayList();
        ArrayList seenVersions = new ArrayList();
        files.getLogFile().accept((file, logVersion) -> {
            seenFiles.add(file);
            seenVersions.add(logVersion);
        });
        Assertions.assertThat(seenFiles).contains((Object[])new Path[]{TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedLogFileName("neostore.transaction.db", "1")), TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedLogFileName("neostore.transaction.db", "3"))});
        Assertions.assertThat(seenVersions).contains((Object[])new Long[]{1L, 3L});
        files.shutdown();
    }

    @Test
    void shouldBeAbleToRetrieveTheHighestLogVersion() throws Throwable {
        LogFiles files = this.createLogFiles();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedLogFileName("1"))).close();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedLogFileName("some", "4"))).close();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedLogFileName("3"))).close();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, "neostore.transaction.db")).close();
        long highestLogVersion = files.getLogFile().getHighestLogVersion();
        org.junit.jupiter.api.Assertions.assertEquals((long)3L, (long)highestLogVersion);
        files.shutdown();
    }

    @Test
    void checkpointAndLogFilesAreIncludedInTheListOfFiles() throws Exception {
        LogFiles files = this.createLogFiles();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedLogFileName(1))).close();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedLogFileName(2))).close();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedCheckpointLogFileName(3))).close();
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, TransactionLogFilesTest.getVersionedCheckpointLogFileName(4))).close();
        Object[] logFiles = files.logFiles();
        Assertions.assertThat((Object[])logFiles).hasSize(4);
    }

    @Test
    void shouldReturnANegativeValueIfThereAreNoLogFiles() throws Throwable {
        LogFiles files = this.createLogFiles();
        this.fileSystem.write(this.databaseLayout.file(TransactionLogFilesTest.getVersionedLogFileName("some", "4"))).close();
        this.fileSystem.write(this.databaseLayout.file("neostore.transaction.db")).close();
        long highestLogVersion = files.getLogFile().getHighestLogVersion();
        org.junit.jupiter.api.Assertions.assertEquals((long)-1L, (long)highestLogVersion);
        files.shutdown();
    }

    @Test
    void shouldFindTheVersionBasedOnTheFilename() throws Throwable {
        LogFiles logFiles = this.createLogFiles();
        Path file = Path.of("v....2", new String[0]);
        long logVersion = logFiles.getLogFile().getLogVersion(file);
        org.junit.jupiter.api.Assertions.assertEquals((long)2L, (long)logVersion);
        logFiles.shutdown();
    }

    @Test
    void shouldThrowIfThereIsNoVersionInTheFileName() throws Exception {
        LogFiles logFiles = this.createLogFiles();
        Path file = Path.of("wrong", new String[0]);
        RuntimeException exception = (RuntimeException)org.junit.jupiter.api.Assertions.assertThrows(RuntimeException.class, () -> logFiles.getLogFile().getLogVersion(file));
        org.junit.jupiter.api.Assertions.assertEquals((Object)("Invalid log file '" + file.getFileName() + "'"), (Object)exception.getMessage());
    }

    @Test
    void shouldThrowIfVersionIsNotANumber() throws Exception {
        LogFiles logFiles = this.createLogFiles();
        Path file = Path.of(TransactionLogFilesTest.getVersionedLogFileName("aa", "A"), new String[0]);
        org.junit.jupiter.api.Assertions.assertThrows(NumberFormatException.class, () -> logFiles.getLogFile().getLogVersion(file));
    }

    @Test
    void isLogFile() throws Exception {
        LogFiles logFiles = this.createLogFiles();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)logFiles.isLogFile(Path.of("aaa.tx.log", new String[0])));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)logFiles.isLogFile(Path.of("neostore.transaction.db.0", new String[0])));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)logFiles.isLogFile(Path.of("neostore.transaction.db.17", new String[0])));
        org.junit.jupiter.api.Assertions.assertTrue((boolean)logFiles.isLogFile(Path.of("checkpoint.17", new String[0])));
        org.junit.jupiter.api.Assertions.assertFalse((boolean)logFiles.isLogFile(Path.of("thecheckpoint.17", new String[0])));
    }

    @Test
    void emptyFileWithoutEntriesDoesNotHaveThem() throws Exception {
        LogFiles logFiles = this.createLogFiles();
        String file = TransactionLogFilesTest.getVersionedLogFileName("1");
        this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(this.databaseLayout, file)).close();
        org.junit.jupiter.api.Assertions.assertFalse((boolean)logFiles.getLogFile().hasAnyEntries(1L));
    }

    @Test
    void fileWithoutEntriesDoesNotHaveThemIndependentlyOfItsSize() throws Exception {
        LogFiles logFiles = this.createLogFiles();
        try (PhysicalLogVersionedStoreChannel channel = logFiles.getLogFile().createLogChannelForVersion(1L, () -> 1L);){
            Assertions.assertThat((long)channel.size()).isGreaterThanOrEqualTo(64L);
            org.junit.jupiter.api.Assertions.assertFalse((boolean)logFiles.getLogFile().hasAnyEntries(1L));
        }
    }

    private void create3_5FileWithHeader(DatabaseLayout databaseLayout, String version, int bytesOfData) throws IOException {
        try (StoreChannel storeChannel = this.fileSystem.write(TransactionLogFilesTest.createTransactionLogFile(databaseLayout, TransactionLogFilesTest.getVersionedLogFileName(version)));){
            ByteBuffer byteBuffer = ByteBuffers.allocate((int)(16 + bytesOfData), (MemoryTracker)EmptyMemoryTracker.INSTANCE);
            while (byteBuffer.hasRemaining()) {
                byteBuffer.put((byte)6);
            }
            byteBuffer.flip();
            storeChannel.writeAll(byteBuffer);
        }
    }

    private void create3_5FileWithHeader(DatabaseLayout databaseLayout, String version) throws IOException {
        this.create3_5FileWithHeader(databaseLayout, version, 0);
    }

    private static Path createTransactionLogFile(DatabaseLayout databaseLayout, String fileName) {
        Path transactionLogsDirectory = databaseLayout.getTransactionLogsDirectory();
        return transactionLogsDirectory.resolve(fileName);
    }

    private LogFiles createLogFiles() throws Exception {
        LogFiles files = LogFilesBuilder.builder((DatabaseLayout)this.databaseLayout, (FileSystemAbstraction)this.fileSystem).withTransactionIdStore((TransactionIdStore)new SimpleTransactionIdStore()).withLogVersionRepository((LogVersionRepository)new SimpleLogVersionRepository()).withLogEntryReader((LogEntryReader)new VersionAwareLogEntryReader((CommandReaderFactory)new TestCommandReaderFactory())).withStoreId(StoreId.UNKNOWN).build();
        files.init();
        return files;
    }

    private static String getVersionedLogFileName(int version) {
        return TransactionLogFilesTest.getVersionedLogFileName("neostore.transaction.db", String.valueOf(version));
    }

    private static String getVersionedCheckpointLogFileName(int version) {
        return TransactionLogFilesTest.getVersionedLogFileName("checkpoint", String.valueOf(version));
    }

    private static String getVersionedLogFileName(String version) {
        return TransactionLogFilesTest.getVersionedLogFileName("neostore.transaction.db", version);
    }

    private static String getVersionedLogFileName(String filename, String version) {
        return filename + "." + version;
    }
}

