/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.impl.storemigration;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;
import org.neo4j.common.ProgressReporter;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.exceptions.KernelException;
import org.neo4j.internal.batchimport.IndexImporterFactory;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.pagecache.context.CursorContext;
import org.neo4j.io.pagecache.tracing.PageCacheTracer;
import org.neo4j.kernel.impl.index.schema.IndexImporterFactoryImpl;
import org.neo4j.kernel.impl.storemigration.LogsUpgrader;
import org.neo4j.kernel.impl.storemigration.MigrationStatus;
import org.neo4j.kernel.impl.transaction.log.files.LogTailInformation;
import org.neo4j.kernel.internal.Version;
import org.neo4j.logging.Log;
import org.neo4j.logging.LogProvider;
import org.neo4j.storageengine.api.IndexCapabilities;
import org.neo4j.storageengine.api.StorageEngineFactory;
import org.neo4j.storageengine.api.StoreVersion;
import org.neo4j.storageengine.api.StoreVersionCheck;
import org.neo4j.storageengine.api.format.Capability;
import org.neo4j.storageengine.migration.MigrationProgressMonitor;
import org.neo4j.storageengine.migration.StoreMigrationParticipant;
import org.neo4j.storageengine.migration.UpgradeNotAllowedException;
import org.neo4j.util.Preconditions;

public class StoreUpgrader {
    private static final String STORE_UPGRADE_TAG = "storeUpgrade";
    public static final String MIGRATION_DIRECTORY = "upgrade";
    public static final String MIGRATION_LEFT_OVERS_DIRECTORY = "upgrade_backup";
    private static final String MIGRATION_STATUS_FILE = "_status";
    private static final Pattern MIGRATION_LEFTOVERS_PATTERN = Pattern.compile("upgrade_backup(_\\d*)?");
    private final StorageEngineFactory storageEngineFactory;
    private final StoreVersionCheck storeVersionCheck;
    private final MigrationProgressMonitor progressMonitor;
    private final LinkedHashMap<String, StoreMigrationParticipant> participants = new LinkedHashMap();
    private final Config config;
    private final FileSystemAbstraction fileSystem;
    private final Log log;
    private final LogsUpgrader logsUpgrader;
    private final String configuredFormat;
    private final PageCacheTracer pageCacheTracer;

    public StoreUpgrader(StorageEngineFactory storageEngineFactory, StoreVersionCheck storeVersionCheck, MigrationProgressMonitor progressMonitor, Config config, FileSystemAbstraction fileSystem, LogProvider logProvider, LogsUpgrader logsUpgrader, PageCacheTracer pageCacheTracer) {
        this.storageEngineFactory = storageEngineFactory;
        this.storeVersionCheck = storeVersionCheck;
        this.progressMonitor = progressMonitor;
        this.fileSystem = fileSystem;
        this.config = config;
        this.logsUpgrader = logsUpgrader;
        this.log = logProvider.getLog(this.getClass());
        this.configuredFormat = storeVersionCheck.configuredVersion();
        this.pageCacheTracer = pageCacheTracer;
    }

    public void addParticipant(StoreMigrationParticipant participant) {
        assert (participant != null);
        if (!StoreMigrationParticipant.NOT_PARTICIPATING.equals(participant)) {
            String newParticipantName = participant.getName();
            Preconditions.checkState((!this.participants.containsKey(newParticipantName) ? 1 : 0) != 0, (String)"Migration participants should have unique names. Participant with name: `%s` is already registered.", (Object[])new Object[]{newParticipantName});
            this.participants.put(newParticipantName, participant);
        }
    }

    public void migrateIfNeeded(DatabaseLayout layout, boolean forceUpgrade) throws IOException {
        if (!Files.exists(layout.databaseDirectory(), new LinkOption[0])) {
            return;
        }
        try (CursorContext cursorContext = new CursorContext(this.pageCacheTracer.createPageCursorTracer(STORE_UPGRADE_TAG));){
            DatabaseLayout migrationStructure = DatabaseLayout.ofFlat((Path)layout.file(MIGRATION_DIRECTORY));
            this.cleanupLegacyLeftOverDirsIn(layout.databaseDirectory());
            Path migrationStateFile = migrationStructure.file(MIGRATION_STATUS_FILE);
            if (StoreUpgrader.hasCurrentVersion(this.storeVersionCheck, cursorContext) && !this.fileSystem.fileExists(migrationStateFile)) {
                return;
            }
            if (this.isUpgradeAllowed() || forceUpgrade) {
                this.migrate(layout, migrationStructure, migrationStateFile, cursorContext);
            } else {
                Optional storeVersion = this.storeVersionCheck.storeVersion(cursorContext);
                if (storeVersion.isPresent()) {
                    StoreVersion version = this.storageEngineFactory.versionInformation((String)storeVersion.get());
                    if (version.hasCapability((Capability)IndexCapabilities.LuceneCapability.LUCENE_5)) {
                        throw new UpgradeNotAllowedException("Upgrade is required to migrate store to new major version.");
                    }
                    String configuredVersion = this.storeVersionCheck.configuredVersion();
                    if (configuredVersion != null && !version.isCompatibleWith(this.storageEngineFactory.versionInformation(configuredVersion))) {
                        throw new UpgradeNotAllowedException();
                    }
                }
            }
        }
    }

    private static boolean hasCurrentVersion(StoreVersionCheck storeVersionCheck, CursorContext cursorContext) {
        String configuredVersion = storeVersionCheck.configuredVersion();
        StoreVersionCheck.Result versionResult = storeVersionCheck.checkUpgrade(configuredVersion, cursorContext);
        if (versionResult.outcome == StoreVersionCheck.Outcome.missingStoreFile) {
            return true;
        }
        return versionResult.outcome.isSuccessful() && versionResult.actualVersion.equals(configuredVersion);
    }

    private void migrate(DatabaseLayout dbDirectoryLayout, DatabaseLayout migrationLayout, Path migrationStateFile, CursorContext cursorContext) {
        this.progressMonitor.started(this.participants.size());
        MigrationStatus migrationStatus = MigrationStatus.readMigrationStatus(this.fileSystem, migrationStateFile);
        String versionToMigrateFrom = null;
        if (MigrationStatus.migrating.isNeededFor(migrationStatus)) {
            StoreVersionCheck.Result upgradeCheck = this.storeVersionCheck.checkUpgrade(this.storeVersionCheck.configuredVersion(), cursorContext);
            versionToMigrateFrom = this.getVersionFromResult(upgradeCheck);
            this.logsUpgrader.assertCleanlyShutDown(dbDirectoryLayout);
            this.cleanMigrationDirectory(migrationLayout.databaseDirectory());
            MigrationStatus.migrating.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            this.migrateToIsolatedDirectory(dbDirectoryLayout, migrationLayout, versionToMigrateFrom);
            MigrationStatus.moving.setMigrationStatus(this.fileSystem, migrationStateFile, versionToMigrateFrom);
        }
        if (MigrationStatus.moving.isNeededFor(migrationStatus)) {
            versionToMigrateFrom = MigrationStatus.moving.maybeReadInfo(this.fileSystem, migrationStateFile, versionToMigrateFrom);
            String versionToMigrateTo = this.storeVersionCheck.configuredVersion();
            StoreUpgrader.moveMigratedFilesToStoreDirectory(this.participants.values(), migrationLayout, dbDirectoryLayout, versionToMigrateFrom, versionToMigrateTo);
        }
        this.progressMonitor.startTransactionLogsMigration();
        this.logsUpgrader.upgrade(dbDirectoryLayout);
        this.progressMonitor.completeTransactionLogsMigration();
        StoreUpgrader.cleanup(this.participants.values(), migrationLayout);
        this.progressMonitor.completed();
    }

    private String getVersionFromResult(StoreVersionCheck.Result result) {
        switch (result.outcome) {
            case ok: {
                return result.actualVersion;
            }
            case missingStoreFile: {
                throw new UpgradeMissingStoreFilesException(result.storeFilename);
            }
            case storeVersionNotFound: {
                throw new UpgradingStoreVersionNotFoundException(result.storeFilename);
            }
            case attemptedStoreDowngrade: {
                throw new AttemptedDowngradeException();
            }
            case unexpectedStoreVersion: {
                throw new UnexpectedUpgradingStoreVersionException(result.actualVersion, this.configuredFormat);
            }
            case unexpectedUpgradingVersion: {
                throw new UnexpectedUpgradingStoreFormatException();
            }
        }
        throw new IllegalArgumentException("Unexpected outcome: " + result.outcome.name());
    }

    List<StoreMigrationParticipant> getParticipants() {
        return List.copyOf(this.participants.values());
    }

    private boolean isUpgradeAllowed() {
        return (Boolean)this.config.get(GraphDatabaseSettings.allow_upgrade);
    }

    private void cleanupLegacyLeftOverDirsIn(Path databaseDirectory) throws IOException {
        try (DirectoryStream<Path> paths = Files.newDirectoryStream(databaseDirectory);){
            for (Path path : paths) {
                if (!MIGRATION_LEFTOVERS_PATTERN.matcher(path.getFileName().toString()).matches()) continue;
                this.deleteSilently(path);
            }
        }
    }

    private static void cleanup(Iterable<StoreMigrationParticipant> participants, DatabaseLayout migrationStructure) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.cleanup(migrationStructure);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Failure cleaning up after migration", e);
        }
    }

    private static void moveMigratedFilesToStoreDirectory(Iterable<StoreMigrationParticipant> participants, DatabaseLayout migrationLayout, DatabaseLayout directoryLayout, String versionToMigrateFrom, String versionToMigrateTo) {
        try {
            for (StoreMigrationParticipant participant : participants) {
                participant.moveMigratedFiles(migrationLayout, directoryLayout, versionToMigrateFrom, versionToMigrateTo);
            }
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Unable to move migrated files into place", e);
        }
    }

    private void migrateToIsolatedDirectory(DatabaseLayout directoryLayout, DatabaseLayout migrationLayout, String versionToMigrateFrom) {
        try {
            for (Map.Entry<String, StoreMigrationParticipant> participantEntry : this.participants.entrySet()) {
                ProgressReporter progressReporter = this.progressMonitor.startSection(participantEntry.getKey());
                String versionToMigrateTo = this.storeVersionCheck.configuredVersion();
                IndexImporterFactoryImpl indexImporterFactory = new IndexImporterFactoryImpl(this.config);
                participantEntry.getValue().migrate(directoryLayout, migrationLayout, progressReporter, versionToMigrateFrom, versionToMigrateTo, (IndexImporterFactory)indexImporterFactory);
                progressReporter.completed();
            }
        }
        catch (IOException | UncheckedIOException | KernelException e) {
            throw new UnableToUpgradeException("Failure doing migration", e);
        }
    }

    private void cleanMigrationDirectory(Path migrationDirectory) {
        try {
            if (this.fileSystem.fileExists(migrationDirectory)) {
                this.fileSystem.deleteRecursively(migrationDirectory);
            }
        }
        catch (IOException | UncheckedIOException e) {
            throw new UnableToUpgradeException("Failure deleting upgrade directory " + migrationDirectory, e);
        }
        try {
            this.fileSystem.mkdir(migrationDirectory);
        }
        catch (IOException e) {
            throw new UnableToUpgradeException("Failure to create directory " + migrationDirectory, e);
        }
    }

    private void deleteSilently(Path dir) {
        try {
            this.fileSystem.deleteRecursively(dir);
        }
        catch (IOException e) {
            this.log.error("Unable to delete directory: " + dir, (Throwable)e);
        }
    }

    public static class DatabaseNotCleanlyShutDownException
    extends UnableToUpgradeException {
        private static final String DEFAULT_MESSAGE = "The database is not cleanly shutdown. The database needs recovery, in order to recover the database, please run the old version of the database on this store.";

        DatabaseNotCleanlyShutDownException() {
            super(DEFAULT_MESSAGE);
        }

        DatabaseNotCleanlyShutDownException(LogTailInformation logTail) {
            super("The database is not cleanly shutdown. The database needs recovery, in order to recover the database, please run the old version of the database on this store. Log tail: " + logTail);
        }
    }

    public static class UnexpectedUpgradingStoreFormatException
    extends UnableToUpgradeException {
        static final String MESSAGE = "This is an enterprise-only store. Please configure '%s' to open.";

        UnexpectedUpgradingStoreFormatException() {
            super(String.format(MESSAGE, GraphDatabaseSettings.record_format.name()));
        }
    }

    public static class AttemptedDowngradeException
    extends UnableToUpgradeException {
        static final String MESSAGE = "Downgrading stores are not supported.";

        AttemptedDowngradeException() {
            super(MESSAGE);
        }
    }

    public static class UnexpectedUpgradingStoreVersionException
    extends UnableToUpgradeException {
        static final String MESSAGE = "Not possible to upgrade a store with version '%s' to current store version `%s` (Neo4j %s).";

        UnexpectedUpgradingStoreVersionException(String fileVersion, String currentVersion) {
            super(String.format(MESSAGE, fileVersion, currentVersion, Version.getNeo4jVersion()));
        }
    }

    static class UpgradingStoreVersionNotFoundException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "'%s' does not contain a store version, please ensure that the original database was shut down in a clean state.";

        UpgradingStoreVersionNotFoundException(String filenameWithoutStoreVersion) {
            super(String.format(MESSAGE, filenameWithoutStoreVersion));
        }
    }

    static class UpgradeMissingStoreFilesException
    extends UnableToUpgradeException {
        private static final String MESSAGE = "Missing required store file '%s'.";

        UpgradeMissingStoreFilesException(String filenameExpectedToExist) {
            super(String.format(MESSAGE, filenameExpectedToExist));
        }
    }

    public static class UnableToUpgradeException
    extends RuntimeException {
        public UnableToUpgradeException(String message, Throwable cause) {
            super(message, cause);
        }

        UnableToUpgradeException(String message) {
            super(message);
        }
    }

    static class TransactionLogsRelocationException
    extends RuntimeException {
        TransactionLogsRelocationException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

