/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [https://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.neo4j.kernel.recovery;

import static org.neo4j.kernel.recovery.RecoveryStartInformation.MISSING_LOGS;
import static org.neo4j.kernel.recovery.RecoveryStartInformation.NO_RECOVERY_REQUIRED;
import static org.neo4j.storageengine.api.LogVersionRepository.INITIAL_LOG_VERSION;

import java.io.IOException;
import org.neo4j.exceptions.UnderlyingStorageException;
import org.neo4j.function.ThrowingSupplier;
import org.neo4j.kernel.impl.transaction.log.CheckpointInfo;
import org.neo4j.kernel.impl.transaction.log.LogPosition;
import org.neo4j.kernel.impl.transaction.log.files.LogFiles;
import org.neo4j.kernel.impl.transaction.log.files.LogTailInformation;

/**
 * Utility class to find the log position to start recovery from
 */
public class RecoveryStartInformationProvider implements ThrowingSupplier<RecoveryStartInformation, IOException> {
    public interface Monitor {
        /**
         * There's a check point log entry as the last entry in the transaction log.
         *
         * @param logPosition {@link LogPosition} of the last check point.
         */
        default void noCommitsAfterLastCheckPoint(LogPosition logPosition) { // no-op by default
        }

        /**
         * There's a check point log entry, but there are other log entries after it.
         *
         * @param logPosition {@link LogPosition} pointing to the first log entry after the last
         * check pointed transaction.
         * @param firstTxIdAfterLastCheckPoint transaction id of the first transaction after the last check point.
         */
        default void logsAfterLastCheckPoint(
                LogPosition logPosition, long firstTxIdAfterLastCheckPoint) { // no-op by default
        }

        /**
         * No check point log entry found in the transaction log.
         */
        default void noCheckPointFound() { // no-op by default
        }

        /**
         * Failure to read initial header of initial log file
         */
        default void failToExtractInitialFileHeader(Exception e) {
            // no-op by default
        }
    }

    public static final Monitor NO_MONITOR = new Monitor() {};

    private final LogFiles logFiles;
    private final Monitor monitor;

    public RecoveryStartInformationProvider(LogFiles logFiles, Monitor monitor) {
        this.logFiles = logFiles;
        this.monitor = monitor;
    }

    @Override
    public RecoveryStartInformation get() {
        var logTailInformation = (LogTailInformation) logFiles.getTailMetadata();
        CheckpointInfo lastCheckPoint = logTailInformation.lastCheckPoint;
        long txIdAfterLastCheckPoint = logTailInformation.firstTxIdAfterLastCheckPoint;

        if (!logTailInformation.isRecoveryRequired()) {
            monitor.noCommitsAfterLastCheckPoint(
                    lastCheckPoint != null ? lastCheckPoint.transactionLogPosition() : null);
            return NO_RECOVERY_REQUIRED;
        }
        if (logTailInformation.logsMissing()) {
            return MISSING_LOGS;
        }
        if (logTailInformation.logsAfterLastCheckpoint()) {
            if (lastCheckPoint == null) {
                long lowestLogVersion = logFiles.getLogFile().getLowestLogVersion();
                if (lowestLogVersion != INITIAL_LOG_VERSION) {
                    throw new UnderlyingStorageException("No check point found in any log file and transaction log "
                            + "files do not exist from expected version " + INITIAL_LOG_VERSION
                            + ". Lowest found log file is "
                            + lowestLogVersion + ".");
                }
                monitor.noCheckPointFound();
                LogPosition position = tryExtractHeaderAndGetStartPosition();
                return createRecoveryInformation(position, LogPosition.UNSPECIFIED, txIdAfterLastCheckPoint);
            }
            LogPosition transactionLogPosition = lastCheckPoint.transactionLogPosition();
            monitor.logsAfterLastCheckPoint(transactionLogPosition, txIdAfterLastCheckPoint);
            return createRecoveryInformation(
                    transactionLogPosition, lastCheckPoint.checkpointEntryPosition(), txIdAfterLastCheckPoint);
        } else {
            throw new UnderlyingStorageException(
                    "Fail to determine recovery information Log tail info: " + logTailInformation);
        }
    }

    private LogPosition tryExtractHeaderAndGetStartPosition() {
        try {
            return logFiles.getLogFile().extractHeader(INITIAL_LOG_VERSION).getStartPosition();
        } catch (IOException e) {
            monitor.failToExtractInitialFileHeader(e);
            throw new UnderlyingStorageException(
                    "Unable to read header from log file with version " + INITIAL_LOG_VERSION, e);
        }
    }

    private static RecoveryStartInformation createRecoveryInformation(
            LogPosition transactionLogPosition, LogPosition checkpointLogPosition, long firstTxId) {
        return new RecoveryStartInformation(transactionLogPosition, checkpointLogPosition, firstTxId);
    }
}
