/*
 * Copyright (c) 2002-2019 "Neo4j,"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.neo4j.ogm.support;

/**
 *  The following methods are a 1:1 copy of {@code org.springframework.util.ClassUtils} in version 5.1.3:
 *
 *  <ul>
 *      <li>{@link #getDefaultClassLoader()}</li>
 *  </ul>
 */
public final class ClassUtils {

    /**
     * Return the default ClassLoader to use: typically the thread context
     * ClassLoader, if available; the ClassLoader that loaded the ClassUtils
     * class will be used as fallback.
     * <p>Call this method if you intend to use the thread context ClassLoader
     * in a scenario where you clearly prefer a non-null ClassLoader reference:
     * for example, for class path resource loading (but not necessarily for
     * {@code Class.forName}, which accepts a {@code null} ClassLoader
     * reference as well).
     *
     * @return the default ClassLoader (only {@code null} if even the system
     * ClassLoader isn't accessible)
     * @see Thread#getContextClassLoader()
     * @see ClassLoader#getSystemClassLoader()
     */
    public static ClassLoader getDefaultClassLoader() {
        ClassLoader cl = null;
        try {
            cl = Thread.currentThread().getContextClassLoader();
        } catch (Throwable ex) {
            // Cannot access thread context ClassLoader - falling back...
        }
        if (cl == null) {
            // No thread context class loader -> use class loader of this class.
            cl = ClassUtils.class.getClassLoader();
            if (cl == null) {
                // getClassLoader() returning null indicates the bootstrap ClassLoader
                try {
                    cl = ClassLoader.getSystemClassLoader();
                } catch (Throwable ex) {
                    // Cannot access system ClassLoader - oh well, maybe the caller can live with null...
                }
            }
        }
        return cl;
    }

    /**
     * See https://github.com/neo4j/neo4j-ogm/issues/643. An enum instance that overrides methods of the enum itself
     * is realized as an anonymous inner class for which {@link Class#isEnum()} returns false.
     *
     * @param clazz The class to check whether it is an enum or not.
     * @return True, if {@code clazz} is an enum.
     */
    public static boolean isEnum(Class<?> clazz) {

        return clazz.isEnum() || Enum.class.isAssignableFrom(clazz);
    }

    /**
     * @param object
     * @return True, if the object is an enum instance.
     * @see #isEnum(Class)
     */
    public static boolean isEnum(Object object) {

        if (object == null) {
            return false;
        }

        return isEnum(object.getClass());
    }

    private ClassUtils() {
    }
}
