/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.ogm.autoindex;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Result;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.schema.ConstraintDefinition;
import org.neo4j.graphdb.schema.IndexDefinition;
import org.neo4j.ogm.autoindex.MissingIndexException;
import org.neo4j.ogm.config.Configuration;
import org.neo4j.ogm.session.SessionFactory;
import org.neo4j.ogm.testutil.MultiDriverTestClass;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class BaseAutoIndexManagerTestClass
extends MultiDriverTestClass {
    private static final Logger logger = LoggerFactory.getLogger(BaseAutoIndexManagerTestClass.class);
    private static final String[] COMMUNITY_INDEXES = new String[]{"INDEX ON :User(email)"};
    private static final String[] COMMUNITY_CONSTRAINTS = new String[]{"CONSTRAINT ON (user:User) ASSERT user.login IS UNIQUE"};
    private static final String[] ENTERPRISE_INDEXES = new String[]{"INDEX ON :User(email)", "INDEX ON :User(lat, lon)"};
    private static final String[] ENTERPRISE_CONSTRAINTS = new String[]{"CONSTRAINT ON (user:User) ASSERT user.login IS UNIQUE", "CONSTRAINT ON (user:User) ASSERT (user.key, user.key2) IS NODE KEY", "CONSTRAINT ON (user:User) ASSERT exists(user.address)", "CONSTRAINT ON ()-[rating:RATING]-() ASSERT exists(rating.stars)"};
    private String[] indexes;
    private String[] constraints;
    private String[] statements;
    private String[] expectedIndexDefinitions;
    private GraphDatabaseService service;
    protected SessionFactory sessionFactory;

    public BaseAutoIndexManagerTestClass(String[] expectedIndexDefinitions, Class<?> ... packages) {
        this.sessionFactory = new SessionFactory(driver, (String[])Arrays.stream(packages).map(Class::getName).toArray(String[]::new));
        this.expectedIndexDefinitions = expectedIndexDefinitions;
    }

    @Before
    public void setUp() {
        this.service = BaseAutoIndexManagerTestClass.getGraphDatabaseService();
        this.service.execute("MATCH (n) DETACH DELETE n");
        String[] existingConstraints = (String[])this.service.execute("CALL db.constraints()").stream().map(r -> r.get("description")).toArray(String[]::new);
        this.executeDrop(existingConstraints);
        if (BaseAutoIndexManagerTestClass.isEnterpriseEdition() && BaseAutoIndexManagerTestClass.isVersionOrGreater((String)"3.2.0")) {
            this.indexes = ENTERPRISE_INDEXES;
            this.constraints = ENTERPRISE_CONSTRAINTS;
            this.statements = (String[])Stream.of(ENTERPRISE_INDEXES, ENTERPRISE_CONSTRAINTS).flatMap(Stream::of).toArray(String[]::new);
        } else {
            this.indexes = COMMUNITY_INDEXES;
            this.constraints = COMMUNITY_CONSTRAINTS;
            this.statements = (String[])Stream.of(COMMUNITY_INDEXES, COMMUNITY_CONSTRAINTS).flatMap(Stream::of).toArray(String[]::new);
        }
    }

    @After
    public void tearDown() throws Exception {
        this.executeDrop(this.expectedIndexDefinitions);
        this.executeDrop(this.statements);
    }

    @Test
    public void testAutoIndexNoneNoIndexIsCreated() {
        this.runAutoIndex("none");
        this.executeForIndexes(indexes -> Assertions.assertThat((List)indexes).isEmpty());
        this.executeForConstraints(constraints -> Assertions.assertThat((List)constraints).isEmpty());
    }

    @Test
    public void testAutoIndexNoneNoIndexIsDropped() {
        this.executeCreate(this.statements);
        this.runAutoIndex("none");
        this.executeForIndexes(indexes -> Assertions.assertThat((List)indexes).hasSize(this.indexes.length));
        this.executeForConstraints(constraints -> Assertions.assertThat((List)constraints).hasSize(this.constraints.length));
    }

    @Test
    public void testIndexesAreSuccessfullyValidated() {
        this.executeCreate(this.expectedIndexDefinitions);
        this.runAutoIndex("validate");
    }

    @Test
    public void testIndexValidationFailsOnMissingIndex() {
        AbstractThrowableAssert assertThatException = ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.runAutoIndex("validate")).isInstanceOf(MissingIndexException.class)).hasMessageContaining("Validation of Constraints and Indexes failed. Could not find the following :");
        for (String definition : this.expectedIndexDefinitions) {
            assertThatException.hasMessageContaining(definition);
        }
    }

    @Test
    public void testAutoIndexAssertDropsAllIndexesAndCreatesExisting() {
        this.executeCreate(this.statements);
        this.runAutoIndex("assert");
        ArrayList all = new ArrayList();
        this.executeForIndexes(all::addAll);
        this.executeForConstraints(all::addAll);
        Assertions.assertThat(all).hasSize(this.expectedIndexDefinitions.length);
    }

    @Test
    public void testAutoIndexUpdateKeepIndexesAndCreateNew() {
        this.executeCreate(this.statements);
        this.runAutoIndex("update");
        ArrayList all = new ArrayList();
        this.executeForIndexes(all::addAll);
        this.executeForConstraints(all::addAll);
        int expectedNumberOfIndexes = this.indexes.length + this.constraints.length + this.expectedIndexDefinitions.length;
        Assertions.assertThat(all).hasSize(expectedNumberOfIndexes);
    }

    @Test
    public void testAutoIndexUpdateIndexExistsDoNothing() {
        this.executeCreate(this.expectedIndexDefinitions);
        this.runAutoIndex("update");
        ArrayList all = new ArrayList();
        this.executeForIndexes(all::addAll);
        this.executeForConstraints(all::addAll);
        Assertions.assertThat(all).hasSize(this.expectedIndexDefinitions.length);
    }

    @Test
    public void testAutoIndexDumpCreatesIndex() throws IOException {
        File file = File.createTempFile("test", ".cql");
        file.deleteOnExit();
        Configuration configuration = BaseAutoIndexManagerTestClass.getBaseConfiguration().autoIndex("dump").generatedIndexesOutputDir(file.getParent()).generatedIndexesOutputFilename(file.getName()).build();
        this.sessionFactory.runAutoIndexManager(configuration);
        Assertions.assertThat((File)file).exists().canRead();
        Assertions.assertThat((String)Assertions.contentOf((File)file)).contains((CharSequence[])Arrays.stream(this.expectedIndexDefinitions).map(d -> "CREATE " + d).toArray(String[]::new));
    }

    void runAutoIndex(String mode) {
        Configuration configuration = BaseAutoIndexManagerTestClass.getBaseConfiguration().autoIndex(mode).build();
        this.sessionFactory.runAutoIndexManager(configuration);
    }

    void executeForIndexes(Consumer<List<IndexDefinition>> consumer) {
        try (Transaction tx = this.service.beginTx();){
            Iterable indexes = this.service.schema().getIndexes();
            List pureIndexes = StreamSupport.stream(indexes.spliterator(), false).filter(indexDefinition -> !indexDefinition.isConstraintIndex()).collect(Collectors.toList());
            consumer.accept(pureIndexes);
            tx.success();
        }
    }

    void executeForConstraints(Consumer<List<ConstraintDefinition>> consumer) {
        try (Transaction tx = this.service.beginTx();){
            List constraints = StreamSupport.stream(this.service.schema().getConstraints().spliterator(), false).collect(Collectors.toList());
            consumer.accept(constraints);
            tx.success();
        }
    }

    void executeCreate(String ... statements) {
        for (String statement : statements) {
            logger.info("Execute CREATE " + statement);
            Result execute = this.service.execute("CREATE " + statement);
            execute.close();
        }
    }

    void executeDrop(String ... statements) {
        for (String statement : statements) {
            Transaction tx = this.service.beginTx();
            try {
                this.service.execute("DROP " + statement);
                tx.success();
            }
            catch (Exception e) {
                logger.trace("Could not execute drop for statement (this is likely expected) {}", (Object)statement, (Object)e);
                tx.failure();
            }
            tx.close();
        }
    }
}

