/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.internal.unsafe;

import com.sun.jna.Native;
import com.sun.jna.Pointer;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.invoke.VarHandle;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.SortedMap;
import java.util.concurrent.ConcurrentNavigableMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicLong;
import org.neo4j.internal.helpers.Exceptions;
import org.neo4j.internal.helpers.VarHandleUtils;
import org.neo4j.internal.unsafe.NativeMemoryAllocationRefusedError;
import org.neo4j.internal.unsafe.UnsafeAccessor;
import org.neo4j.memory.MemoryTracker;
import org.neo4j.util.FeatureToggles;
import sun.misc.Unsafe;

public final class UnsafeUtil {
    private static final boolean DIRTY_MEMORY;
    private static final boolean CHECK_NATIVE_ACCESS;
    private static final int NERFED_BUFFER_MARK = -2;
    private static boolean nativeAccessCheckEnabled;
    private static final Unsafe unsafe;
    private static final String allowUnalignedMemoryAccessProperty = "org.neo4j.internal.unsafe.UnsafeUtil.allowUnalignedMemoryAccess";
    private static final ConcurrentSkipListMap<Long, Allocation> allocations;
    private static final ThreadLocal<Allocation> lastUsedAllocation;
    private static final FreeTrace[] freeTraces;
    private static final AtomicLong freeCounter;
    public static final Class<?> DIRECT_BYTE_BUFFER_CLASS;
    private static final VarHandle BYTE_BUFFER_MARK;
    private static final VarHandle BYTE_BUFFER_POSITION;
    private static final VarHandle BYTE_BUFFER_LIMIT;
    private static final VarHandle BYTE_BUFFER_CAPACITY;
    private static final VarHandle BYTE_BUFFER_ADDRESS;
    private static final MethodHandle DIRECT_BYTE_BUFFER_CONSTRUCTOR;
    private static final int pageSize;
    public static final boolean allowUnalignedMemoryAccess;
    public static final boolean nativeByteOrderIsLittleEndian;

    private UnsafeUtil() {
    }

    private static MethodHandle tryLookupConstructor(Class<?> dbbClass) {
        if (dbbClass != null) {
            try {
                MethodHandles.Lookup directByteBufferLookup = MethodHandles.privateLookupIn(dbbClass, MethodHandles.lookup());
                return directByteBufferLookup.findConstructor(dbbClass, MethodType.methodType(Void.TYPE, Long.TYPE, Long.TYPE));
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        }
        return null;
    }

    private static boolean findUnalignedMemoryAccess() {
        String alignmentProperty = System.getProperty(allowUnalignedMemoryAccessProperty);
        if (alignmentProperty != null && (alignmentProperty.equalsIgnoreCase("true") || alignmentProperty.equalsIgnoreCase("false"))) {
            return Boolean.parseBoolean(alignmentProperty);
        }
        try {
            Class<?> bits = Class.forName("java.nio.Bits");
            Method unaligned = bits.getDeclaredMethod("unaligned", new Class[0]);
            unaligned.setAccessible(true);
            return (Boolean)unaligned.invoke(null, new Object[0]);
        }
        catch (Throwable t) {
            return UnsafeUtil.findUnalignedMemoryAccessFromArch();
        }
    }

    private static boolean findUnalignedMemoryAccessFromArch() {
        String arch;
        return switch (arch = System.getProperty("os.arch", "?")) {
            case "x86_64", "i386", "x86", "amd64", "ppc64", "ppc64le", "ppc64be", "aarch64" -> true;
            default -> false;
        };
    }

    public static void assertHasUnsafe() {
        if (unsafe == null) {
            throw new LinkageError("Unsafe not available");
        }
    }

    public static long getFieldOffset(Class<?> type, String field) {
        try {
            return unsafe.objectFieldOffset(type.getDeclaredField(field));
        }
        catch (NoSuchFieldException e) {
            String message = "Could not get offset of '" + field + "' field on type " + String.valueOf(type);
            throw new LinkageError(message, e);
        }
    }

    public static long getFieldOffset(Field field) {
        return unsafe.objectFieldOffset(field);
    }

    public static long getAndAddLong(Object obj, long offset, long delta) {
        UnsafeUtil.checkAccess(obj, offset, 8L);
        return unsafe.getAndAddLong(obj, offset, delta);
    }

    public static boolean compareAndSwapLong(Object obj, long offset, long expected, long update) {
        UnsafeUtil.checkAccess(obj, offset, 8L);
        return unsafe.compareAndSwapLong(obj, offset, expected, update);
    }

    public static boolean compareAndSwapInt(Object obj, long offset, int expected, int update) {
        UnsafeUtil.checkAccess(obj, offset, 4L);
        return unsafe.compareAndSwapInt(obj, offset, expected, update);
    }

    public static long getAndSetLong(Object obj, long offset, long newValue) {
        UnsafeUtil.checkAccess(obj, offset, 8L);
        return unsafe.getAndSetLong(obj, offset, newValue);
    }

    public static void compareAndSetMaxLong(Object object, long fieldOffset, long newValue) {
        long currentValue;
        UnsafeUtil.checkAccess(object, fieldOffset, 8L);
        do {
            if ((currentValue = UnsafeUtil.getLongVolatile(object, fieldOffset)) < newValue) continue;
            return;
        } while (!UnsafeUtil.compareAndSwapLong(object, fieldOffset, currentValue, newValue));
    }

    public static long allocateMemory(long bytes, MemoryTracker memoryTracker) throws NativeMemoryAllocationRefusedError {
        memoryTracker.allocateNative(bytes);
        long pointer = Native.malloc((long)bytes);
        if (pointer == 0L) {
            memoryTracker.releaseNative(bytes);
            throw new NativeMemoryAllocationRefusedError(bytes, memoryTracker.usedNativeMemory());
        }
        UnsafeUtil.addAllocatedPointer(pointer, bytes);
        if (DIRTY_MEMORY) {
            UnsafeUtil.setMemory(pointer, bytes, (byte)-91);
        }
        return pointer;
    }

    public static void free(long pointer, long bytes, MemoryTracker memoryTracker) {
        UnsafeUtil.checkFree(pointer);
        Native.free((long)pointer);
        memoryTracker.releaseNative(bytes);
    }

    public static boolean isCheckNativeAccessEnabled() {
        return CHECK_NATIVE_ACCESS;
    }

    public static void addAllocatedPointer(long pointer, long sizeInBytes) {
        if (CHECK_NATIVE_ACCESS) {
            allocations.put(pointer, new Allocation(pointer, sizeInBytes));
        }
    }

    public static void removeAllocatedPointer(long pointer) {
        Allocation allocation;
        if (CHECK_NATIVE_ACCESS && (allocation = allocations.remove(pointer)) == null) {
            StringBuilder sb = new StringBuilder(String.format("Bad free: 0x%x, valid pointers are:", pointer));
            allocations.forEach((k, v) -> sb.append('\n').append("0x").append(Long.toHexString(k)));
            throw new AssertionError((Object)sb.toString());
        }
    }

    private static void checkFree(long pointer) {
        if (CHECK_NATIVE_ACCESS) {
            UnsafeUtil.doCheckFree(pointer);
        }
    }

    private static void doCheckFree(long pointer) {
        long count = freeCounter.getAndIncrement();
        Allocation allocation = allocations.remove(pointer);
        if (allocation == null) {
            StringBuilder sb = new StringBuilder(String.format("Bad free: 0x%x, valid pointers are:", pointer));
            allocations.forEach((k, v) -> sb.append('\n').append("0x").append(Long.toHexString(k)));
            throw new AssertionError((Object)sb.toString());
        }
        allocation.freed = true;
        int idx = (int)(count & 0xFFFL);
        UnsafeUtil.freeTraces[idx] = new FreeTrace(pointer, allocation, count);
    }

    static void checkAccess(long pointer, long size) {
        if (CHECK_NATIVE_ACCESS && nativeAccessCheckEnabled) {
            UnsafeUtil.doCheckAccess(pointer, size);
        }
    }

    private static void checkAccess(Object object, long pointer, long size) {
        if (CHECK_NATIVE_ACCESS && nativeAccessCheckEnabled && object == null) {
            UnsafeUtil.doCheckAccess(pointer, size);
        }
    }

    private static void doCheckAccess(long pointer, long size) {
        long boundary = pointer + size;
        Allocation allocation = lastUsedAllocation.get();
        if (allocation != null && !allocation.freed && Long.compareUnsigned(allocation.pointer, pointer) <= 0 && Long.compareUnsigned(allocation.boundary, boundary) >= 0) {
            return;
        }
        Map.Entry<Long, Allocation> floorEntry = allocations.floorEntry(pointer);
        if (floorEntry == null) {
            SortedMap allocationsInRange = allocations.headMap((Object)boundary);
            UnsafeUtil.throwBadAccess(pointer, size, (ConcurrentNavigableMap<Long, Allocation>)allocationsInRange);
            return;
        }
        Allocation floorAllocation = floorEntry.getValue();
        if (Long.compareUnsigned(floorAllocation.pointer, pointer) > 0 || Long.compareUnsigned(floorAllocation.boundary, boundary) < 0) {
            SortedMap allocationsInRange = allocations.subMap((Object)floorAllocation.pointer, (Object)(pointer + size));
            long lastAllocatedAddress = ((Allocation)allocationsInRange.lastEntry().getValue()).boundary;
            if (Long.compareUnsigned(boundary, lastAllocatedAddress) <= 0 && UnsafeUtil.isContinuous((ConcurrentNavigableMap<Long, Allocation>)allocationsInRange)) {
                return;
            }
            UnsafeUtil.throwBadAccess(pointer, size, (ConcurrentNavigableMap<Long, Allocation>)allocationsInRange);
        }
        lastUsedAllocation.set(floorAllocation);
    }

    private static boolean isContinuous(ConcurrentNavigableMap<Long, Allocation> allocationsInRange) {
        if (allocationsInRange.size() < 2) {
            return false;
        }
        Iterator iterator = allocationsInRange.values().iterator();
        Allocation previous = (Allocation)iterator.next();
        while (iterator.hasNext()) {
            Allocation next = (Allocation)iterator.next();
            if (next.pointer != previous.boundary) {
                return false;
            }
            previous = next;
        }
        return true;
    }

    private static void throwBadAccess(long pointer, long size, ConcurrentNavigableMap<Long, Allocation> allocationsInRange) {
        long now = System.nanoTime();
        String accessMap = UnsafeUtil.buildAccessMap(pointer, size, allocationsInRange);
        List<FreeTrace> recentFrees = Arrays.stream(freeTraces).filter(Objects::nonNull).filter(trace -> trace.contains(pointer)).sorted().toList();
        AssertionError error = new AssertionError((Object)String.format("Bad access to address 0x%x with size %s. Access map: '%s'. Recent relevant frees (of %s) are attached as suppressed exceptions.", pointer, size, accessMap, freeCounter.get()));
        for (FreeTrace recentFree : recentFrees) {
            recentFree.referenceTime = now;
            ((Throwable)((Object)error)).addSuppressed(recentFree);
        }
        throw error;
    }

    private static String buildAccessMap(long pointer, long size, ConcurrentNavigableMap<Long, Allocation> allocationsInRange) {
        Map.Entry<Long, Allocation> longAllocationEntry;
        if (allocationsInRange.isEmpty() && (longAllocationEntry = allocations.ceilingEntry(pointer)) != null) {
            StringBuilder sb = new StringBuilder();
            UnsafeUtil.appendAccessInfo(sb, pointer, size, 0L, -1L);
            sb.append("[0x%x - 0x%x]".formatted(longAllocationEntry.getValue().pointer, longAllocationEntry.getValue().boundary));
            return sb.toString();
        }
        StringBuilder sb = new StringBuilder();
        long previousBoundary = 0L;
        for (Allocation value : allocationsInRange.values()) {
            if (previousBoundary != 0L && previousBoundary != value.pointer) {
                sb.append(" GAP! ");
            }
            previousBoundary = UnsafeUtil.appendAccessInfo(sb, pointer, size, previousBoundary, value.pointer);
            sb.append("[0x%x -".formatted(value.pointer));
            previousBoundary = UnsafeUtil.appendAccessInfo(sb, pointer, size, previousBoundary, value.boundary);
            sb.append(" 0x%x]".formatted(value.boundary));
        }
        UnsafeUtil.appendAccessInfo(sb, pointer, size, previousBoundary, -1L);
        return sb.toString();
    }

    private static long appendAccessInfo(StringBuilder sb, long startAccess, long size, long start, long end) {
        long endAccess;
        if (Long.compareUnsigned(startAccess, start) >= 0 && Long.compareUnsigned(startAccess, end) < 0) {
            sb.append(" <access start(0x%x)> ".formatted(startAccess));
        }
        if (Long.compareUnsigned(endAccess = startAccess + size, start) >= 0 && Long.compareUnsigned(endAccess, end) < 0) {
            sb.append(" <access end(0x%x)> ".formatted(endAccess));
        }
        return end;
    }

    public static int pageSize() {
        return pageSize;
    }

    public static void putByte(long address, byte value) {
        UnsafeUtil.checkAccess(address, 1L);
        unsafe.putByte(address, value);
    }

    public static byte getByte(long address) {
        UnsafeUtil.checkAccess(address, 1L);
        return unsafe.getByte(address);
    }

    public static void putByte(Object obj, long offset, byte value) {
        UnsafeUtil.checkAccess(obj, offset, 1L);
        unsafe.putByte(obj, offset, value);
    }

    public static byte getByte(Object obj, long offset) {
        UnsafeUtil.checkAccess(obj, offset, 1L);
        return unsafe.getByte(obj, offset);
    }

    public static void putShort(long address, short value) {
        UnsafeUtil.checkAccess(address, 2L);
        unsafe.putShort(address, value);
    }

    public static short getShort(long address) {
        UnsafeUtil.checkAccess(address, 2L);
        return unsafe.getShort(address);
    }

    public static void putInt(long address, int value) {
        UnsafeUtil.checkAccess(address, 4L);
        unsafe.putInt(address, value);
    }

    public static int getInt(long address) {
        UnsafeUtil.checkAccess(address, 4L);
        return unsafe.getInt(address);
    }

    public static void putLongVolatile(long address, long value) {
        UnsafeUtil.checkAccess(address, 8L);
        unsafe.putLongVolatile(null, address, value);
    }

    public static long getLongVolatile(long address) {
        UnsafeUtil.checkAccess(address, 8L);
        return unsafe.getLongVolatile(null, address);
    }

    public static void putLong(long address, long value) {
        UnsafeUtil.checkAccess(address, 8L);
        unsafe.putLong(address, value);
    }

    public static long getLong(long address) {
        UnsafeUtil.checkAccess(address, 8L);
        return unsafe.getLong(address);
    }

    public static long getLongVolatile(Object obj, long offset) {
        UnsafeUtil.checkAccess(obj, offset, 8L);
        return unsafe.getLongVolatile(obj, offset);
    }

    public static int arrayBaseOffset(Class<?> klass) {
        return unsafe.arrayBaseOffset(klass);
    }

    public static int arrayIndexScale(Class<?> klass) {
        int scale = unsafe.arrayIndexScale(klass);
        if (scale == 0) {
            throw new AssertionError((Object)("Array type too narrow for unsafe access: " + String.valueOf(klass)));
        }
        return scale;
    }

    public static int arrayOffset(int index, int base, int scale) {
        return base + index * scale;
    }

    public static void setMemory(long address, long bytes, byte value) {
        UnsafeUtil.checkAccess(address, bytes);
        new Pointer(address).setMemory(0L, bytes, value);
    }

    public static void copyMemory(long srcAddress, long destAddress, long bytes) {
        UnsafeUtil.checkAccess(srcAddress, bytes);
        UnsafeUtil.checkAccess(destAddress, bytes);
        unsafe.copyMemory(srcAddress, destAddress, bytes);
    }

    public static void copyMemory(Object srcBase, long srcOffset, Object destBase, long destOffset, long bytes) {
        UnsafeUtil.checkAccess(srcBase, srcOffset, bytes);
        UnsafeUtil.checkAccess(destBase, destOffset, bytes);
        unsafe.copyMemory(srcBase, srcOffset, destBase, destOffset, bytes);
    }

    public static boolean exchangeNativeAccessCheckEnabled(boolean newSetting) {
        boolean previousSetting = nativeAccessCheckEnabled;
        nativeAccessCheckEnabled = newSetting;
        return previousSetting;
    }

    public static short getShortByteWiseLittleEndian(long p) {
        short a = (short)(UnsafeUtil.getByte(p) & 0xFF);
        short b = (short)(UnsafeUtil.getByte(p + 1L) & 0xFF);
        return (short)(b << 8 | a);
    }

    public static int getIntByteWiseLittleEndian(long p) {
        int a = UnsafeUtil.getByte(p) & 0xFF;
        int b = UnsafeUtil.getByte(p + 1L) & 0xFF;
        int c = UnsafeUtil.getByte(p + 2L) & 0xFF;
        int d = UnsafeUtil.getByte(p + 3L) & 0xFF;
        return d << 24 | c << 16 | b << 8 | a;
    }

    public static long getLongByteWiseLittleEndian(long p) {
        long a = UnsafeUtil.getByte(p) & 0xFF;
        long b = UnsafeUtil.getByte(p + 1L) & 0xFF;
        long c = UnsafeUtil.getByte(p + 2L) & 0xFF;
        long d = UnsafeUtil.getByte(p + 3L) & 0xFF;
        long e = UnsafeUtil.getByte(p + 4L) & 0xFF;
        long f = UnsafeUtil.getByte(p + 5L) & 0xFF;
        long g = UnsafeUtil.getByte(p + 6L) & 0xFF;
        long h = UnsafeUtil.getByte(p + 7L) & 0xFF;
        return h << 56 | g << 48 | f << 40 | e << 32 | d << 24 | c << 16 | b << 8 | a;
    }

    public static void putShortByteWiseLittleEndian(long p, short value) {
        UnsafeUtil.putByte(p, (byte)value);
        UnsafeUtil.putByte(p + 1L, (byte)(value >> 8));
    }

    public static void putIntByteWiseLittleEndian(long p, int value) {
        UnsafeUtil.putByte(p, (byte)value);
        UnsafeUtil.putByte(p + 1L, (byte)(value >> 8));
        UnsafeUtil.putByte(p + 2L, (byte)(value >> 16));
        UnsafeUtil.putByte(p + 3L, (byte)(value >> 24));
    }

    public static void putLongByteWiseLittleEndian(long p, long value) {
        UnsafeUtil.putByte(p, (byte)value);
        UnsafeUtil.putByte(p + 1L, (byte)(value >> 8));
        UnsafeUtil.putByte(p + 2L, (byte)(value >> 16));
        UnsafeUtil.putByte(p + 3L, (byte)(value >> 24));
        UnsafeUtil.putByte(p + 4L, (byte)(value >> 32));
        UnsafeUtil.putByte(p + 5L, (byte)(value >> 40));
        UnsafeUtil.putByte(p + 6L, (byte)(value >> 48));
        UnsafeUtil.putByte(p + 7L, (byte)(value >> 56));
    }

    public static boolean unsafeByteBufferAccessAvailable() {
        return DIRECT_BYTE_BUFFER_CLASS != null;
    }

    private static void assertUnsafeByteBufferAccess() {
        if (!UnsafeUtil.unsafeByteBufferAccessAvailable()) {
            throw new IllegalStateException("java.nio.DirectByteBuffer is not available. Start with --add-opens=java.base/java.nio=ALL-UNNAMED");
        }
    }

    public static ByteBuffer allocateByteBuffer(int size, MemoryTracker memoryTracker) {
        UnsafeUtil.assertUnsafeByteBufferAccess();
        try {
            long addr = UnsafeUtil.allocateMemory(size, memoryTracker);
            UnsafeUtil.setMemory(addr, size, (byte)0);
            return UnsafeUtil.newDirectByteBuffer(addr, size);
        }
        catch (Throwable e) {
            Exceptions.throwIfUnchecked((Throwable)e);
            throw new RuntimeException(e);
        }
    }

    public static void freeByteBuffer(ByteBuffer byteBuffer, MemoryTracker memoryTracker) {
        UnsafeUtil.assertUnsafeByteBufferAccess();
        int bytes = byteBuffer.capacity();
        long addr = UnsafeUtil.getDirectByteBufferAddress(byteBuffer);
        if (addr == 0L) {
            return;
        }
        UnsafeUtil.nerfBuffer(byteBuffer);
        UnsafeUtil.free(addr, bytes, memoryTracker);
    }

    private static void nerfBuffer(Buffer byteBuffer) {
        UnsafeUtil.assertUnsafeByteBufferAccess();
        BYTE_BUFFER_MARK.set(byteBuffer, -2);
        BYTE_BUFFER_POSITION.set(byteBuffer, 0);
        BYTE_BUFFER_LIMIT.set(byteBuffer, 0);
        BYTE_BUFFER_CAPACITY.set(byteBuffer, 0);
        BYTE_BUFFER_ADDRESS.set(byteBuffer, 0L);
    }

    public static ByteBuffer newDirectByteBuffer(long addr, int cap) throws Throwable {
        UnsafeUtil.assertUnsafeByteBufferAccess();
        UnsafeUtil.checkAccess(addr, cap);
        return DIRECT_BYTE_BUFFER_CONSTRUCTOR.invoke(addr, cap);
    }

    public static void initDirectByteBuffer(ByteBuffer dbb, long addr, int cap) {
        UnsafeUtil.assertUnsafeByteBufferAccess();
        UnsafeUtil.checkAccess(addr, cap);
        dbb.order(ByteOrder.LITTLE_ENDIAN);
        ByteBuffer bb = dbb;
        BYTE_BUFFER_MARK.set(bb, -1);
        BYTE_BUFFER_POSITION.set(bb, 0);
        BYTE_BUFFER_LIMIT.set(bb, cap);
        BYTE_BUFFER_CAPACITY.set(bb, cap);
        BYTE_BUFFER_ADDRESS.set(bb, addr);
    }

    public static long getDirectByteBufferAddress(Buffer dbb) {
        UnsafeUtil.assertUnsafeByteBufferAccess();
        return BYTE_BUFFER_ADDRESS.get(dbb);
    }

    public static void invokeCleaner(ByteBuffer byteBuffer) {
        unsafe.invokeCleaner(byteBuffer);
    }

    public static void releaseBuffer(ByteBuffer byteBuffer, MemoryTracker memoryTracker) {
        if (!byteBuffer.isDirect()) {
            UnsafeUtil.freeHeapByteBuffer(byteBuffer, memoryTracker);
            return;
        }
        UnsafeUtil.freeByteBuffer(byteBuffer, memoryTracker);
    }

    private static void freeHeapByteBuffer(Buffer byteBuffer, MemoryTracker memoryTracker) {
        if (BYTE_BUFFER_MARK.get(byteBuffer) == -2) {
            return;
        }
        int capacity = byteBuffer.capacity();
        UnsafeUtil.nerfBuffer(byteBuffer);
        memoryTracker.releaseHeap((long)capacity);
    }

    static {
        MethodHandle dbbCtor;
        DIRTY_MEMORY = FeatureToggles.flag(UnsafeUtil.class, (String)"DIRTY_MEMORY", (boolean)false);
        CHECK_NATIVE_ACCESS = FeatureToggles.flag(UnsafeUtil.class, (String)"CHECK_NATIVE_ACCESS", (boolean)false);
        nativeAccessCheckEnabled = true;
        allocations = new ConcurrentSkipListMap(Long::compareUnsigned);
        lastUsedAllocation = new ThreadLocal();
        freeTraces = CHECK_NATIVE_ACCESS ? new FreeTrace[4096] : null;
        freeCounter = new AtomicLong();
        unsafe = UnsafeAccessor.getUnsafe();
        pageSize = unsafe.pageSize();
        allowUnalignedMemoryAccess = UnsafeUtil.findUnalignedMemoryAccess();
        nativeByteOrderIsLittleEndian = ByteOrder.nativeOrder() == ByteOrder.LITTLE_ENDIAN;
        Class<?> dbbClass = null;
        VarHandle bbMark = null;
        VarHandle bbPosition = null;
        VarHandle bbLimit = null;
        VarHandle bbCapacity = null;
        VarHandle bbAddress = null;
        try {
            MethodHandles.Lookup bufferLookup = MethodHandles.privateLookupIn(Buffer.class, MethodHandles.lookup());
            bbMark = VarHandleUtils.getVarHandle((MethodHandles.Lookup)bufferLookup, Buffer.class, (String)"mark", Integer.TYPE);
            bbPosition = VarHandleUtils.getVarHandle((MethodHandles.Lookup)bufferLookup, Buffer.class, (String)"position", Integer.TYPE);
            bbCapacity = bbLimit = VarHandleUtils.getVarHandle((MethodHandles.Lookup)bufferLookup, Buffer.class, (String)"limit", Integer.TYPE);
            bbAddress = VarHandleUtils.getVarHandle((MethodHandles.Lookup)bufferLookup, Buffer.class, (String)"address", Long.TYPE);
            dbbClass = Class.forName("java.nio.DirectByteBuffer");
            dbbCtor = UnsafeUtil.tryLookupConstructor(dbbClass);
        }
        catch (Throwable e) {
            dbbCtor = UnsafeUtil.tryLookupConstructor(dbbClass);
        }
        DIRECT_BYTE_BUFFER_CLASS = dbbClass;
        BYTE_BUFFER_MARK = bbMark;
        BYTE_BUFFER_POSITION = bbPosition;
        BYTE_BUFFER_LIMIT = bbLimit;
        BYTE_BUFFER_CAPACITY = bbCapacity;
        BYTE_BUFFER_ADDRESS = bbAddress;
        DIRECT_BYTE_BUFFER_CONSTRUCTOR = dbbCtor;
    }

    private static final class Allocation {
        private final long pointer;
        private final long sizeInBytes;
        private final long boundary;
        public volatile boolean freed;

        Allocation(long pointer, long sizeInBytes) {
            this.pointer = pointer;
            this.sizeInBytes = sizeInBytes;
            this.boundary = pointer + sizeInBytes;
        }

        public String toString() {
            return String.format("Allocation[pointer=%s (%x), size=%s, boundary=%s (%x)]", this.pointer, this.pointer, this.sizeInBytes, this.boundary, this.boundary);
        }
    }

    private static final class FreeTrace
    extends Throwable
    implements Comparable<FreeTrace> {
        private final long pointer;
        private final Allocation allocation;
        private final long id;
        private final long nanoTime;
        private long referenceTime;

        private FreeTrace(long pointer, Allocation allocation, long id) {
            this.pointer = pointer;
            this.allocation = allocation;
            this.id = id;
            this.nanoTime = System.nanoTime();
        }

        private boolean contains(long pointer) {
            return this.pointer <= pointer && pointer <= this.pointer + this.allocation.sizeInBytes;
        }

        @Override
        public int compareTo(FreeTrace that) {
            return Long.compare(this.id, that.id);
        }

        @Override
        public String getMessage() {
            return String.format("0x%x of %6d bytes, freed %s \u00b5s ago at", this.pointer, this.allocation.sizeInBytes, (this.referenceTime - this.nanoTime) / 1000L);
        }
    }
}

