package apoc.index;

import apoc.meta.Meta;
import apoc.result.WeightedNodeResult;
import apoc.result.WeightedRelationshipResult;
import org.neo4j.graphdb.*;
import org.neo4j.graphdb.index.Index;
import org.neo4j.graphdb.index.IndexHits;
import org.neo4j.graphdb.index.IndexManager;
import org.neo4j.graphdb.index.RelationshipIndex;
import org.neo4j.index.impl.lucene.explicit.LuceneIndexImplementation;
import org.neo4j.logging.Log;
import org.neo4j.procedure.*;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

/**
 * @author mh
 * @since 25.03.16
 */
public class FulltextIndex {
    private static final Map<String, String> FULL_TEXT = LuceneIndexImplementation.FULLTEXT_CONFIG;
    public static final String NODE = Meta.Types.NODE.name();
    public static final String RELATIONSHIP = Meta.Types.RELATIONSHIP.name();

    @Context
    public GraphDatabaseService db;

    @Context
    public Log log;

    @Description("apoc.index.nodes('Label','prop:value*') YIELD node - lucene query on node index with the given label name")
    @Procedure(mode = Mode.READ)
    public Stream<WeightedNodeResult> nodes(@Name("label") String label, @Name("query") String query) throws Exception {
        if (!db.index().existsForNodes(label)) return Stream.empty();
        return toWeightedNodeResult(db.index().forNodes(label).query(query));
    }

    @UserFunction("apoc.index.nodes.count")
    @Description("apoc.index.nodes.count('Label','prop:value*') YIELD value - lucene query on node index with the given label name")
    public long nodesCount(@Name("label") String label, @Name("query") String query) throws Exception {
        if (!db.index().existsForNodes(label)) return 0L;
        return (long) db.index().forNodes(label).query(query).size();
    }

    public static class IndexInfo {
        public final String type;
        public final String name;
        public final Map<String,String> config;

        public IndexInfo(String type, String name, Map<String, String> config) {
            this.type = type;
            this.name = name;
            this.config = config;
        }
    }

    @Description("apoc.index.forNodes('name',{config}) YIELD type,name,config - gets or creates node index")
    @Procedure(mode = Mode.WRITE)
    public Stream<IndexInfo> forNodes(@Name("name") String name, @Name(value="config",defaultValue="") Map<String,String> config) {
        Index<Node> index = getNodeIndex(name, config);
        return Stream.of(new IndexInfo(NODE, name, db.index().getConfiguration(index)));
    }

    private Index<Node> getNodeIndex(String name, Map<String, String> config) {
        IndexManager mgr = db.index();
        return config == null ? mgr.forNodes(name) : mgr.forNodes(name, config);
    }

    @Description("apoc.index.forRelationships('name',{config}) YIELD type,name,config - gets or creates relationship index")
    @Procedure(mode = Mode.WRITE)
    public Stream<IndexInfo> forRelationships(@Name("name") String name, @Name(value="config",defaultValue="") Map<String,String> config) {
        RelationshipIndex index = getRelationshipIndex(name, config);
        return Stream.of(new IndexInfo(RELATIONSHIP, name, db.index().getConfiguration(index)));
    }

    private RelationshipIndex getRelationshipIndex(String name, Map<String, String> config) {
        IndexManager mgr = db.index();
        return config == null ? mgr.forRelationships(name) : mgr.forRelationships(name, config);
    }

    @Description("apoc.index.remove('name') YIELD type,name,config - removes an manual index")
    @Procedure(mode = Mode.WRITE)
    public Stream<IndexInfo> remove(@Name("name") String name) {
        IndexManager mgr = db.index();
        List<IndexInfo> indexInfos = new ArrayList<>(2);
        if (mgr.existsForNodes(name)) {
            Index<Node> index = mgr.forNodes(name);
            indexInfos.add(new IndexInfo(NODE, name, mgr.getConfiguration(index)));
            index.delete();
        }
        if (mgr.existsForRelationships(name)) {
            RelationshipIndex index = mgr.forRelationships(name);
            indexInfos.add(new IndexInfo(RELATIONSHIP, name, mgr.getConfiguration(index)));
            index.delete();
        }
        return indexInfos.stream();
    }

    @Description("apoc.index.list() - YIELD type,name,config - lists all manual indexes")
    @Procedure(mode = Mode.READ)
    public Stream<IndexInfo> list() {
        IndexManager mgr = db.index();
        List<IndexInfo> indexInfos = new ArrayList<>(100);
        for (String name : mgr.nodeIndexNames()) {
            Index<Node> index = mgr.forNodes(name);
            indexInfos.add(new IndexInfo(NODE,name,mgr.getConfiguration(index)));
        }
        for (String name : mgr.relationshipIndexNames()) {
            RelationshipIndex index = mgr.forRelationships(name);
            indexInfos.add(new IndexInfo(RELATIONSHIP,name,mgr.getConfiguration(index)));
        }
        return indexInfos.stream();
    }

    private Stream<WeightedNodeResult> toWeightedNodeResult(IndexHits<Node> hits) {
        List<WeightedNodeResult> results = new ArrayList<>(hits.size());
        while (hits.hasNext()) {
            results.add(new WeightedNodeResult(hits.next(),(double)hits.currentScore()));
        }
        return results.stream();
    }
    private Stream<WeightedRelationshipResult> toWeightedRelationshipResult(IndexHits<Relationship> hits) {
        List<WeightedRelationshipResult> results = new ArrayList<>(hits.size());
        while (hits.hasNext()) {
            results.add(new WeightedRelationshipResult(hits.next(),(double)hits.currentScore()));
        }
        return results.stream();
    }

    // CALL apoc.index.relationships('CHECKIN','on:2010-*')
    @Description("apoc.index.relationships('TYPE','prop:value*') YIELD rel - lucene query on relationship index with the given type name")
    @Procedure(mode = Mode.READ)
    public Stream<WeightedRelationshipResult> relationships(@Name("type") String type, @Name("query") String query) throws Exception {
        if (!db.index().existsForRelationships(type)) return Stream.empty();
        return toWeightedRelationshipResult(db.index().forRelationships(type).query(query,null,null));
    }

    @UserFunction("apoc.index.relationships.count")
    @Description("apoc.index.relationships.count('Type','prop:value*') YIELD value - lucene query on relationship index with the given type name")
    public long relationshipsCount(@Name("type") String type, @Name("query") String query) throws Exception {
        if (!db.index().existsForRelationships(type)) return 0L;
        return (long) db.index().forRelationships(type).query(query, null, null).size();
    }

    // CALL apoc.index.between(joe, 'KNOWS', null, 'since:2010-*')
    // CALL apoc.index.between(joe, 'CHECKIN', philz, 'on:2016-01-*')
    @Description("apoc.index.between(node1,'TYPE',node2,'prop:value*') YIELD rel - lucene query on relationship index with the given type name bound by either or both sides (each node parameter can be null)")
    @Procedure(mode = Mode.READ)
    public Stream<WeightedRelationshipResult> between(@Name("from") Node from, @Name("type") String type, @Name("to") Node to, @Name("query") String query) throws Exception {
        if (!db.index().existsForRelationships(type)) return Stream.empty();
        return toWeightedRelationshipResult(db.index().forRelationships(type).query(query,from,to));
    }

    @UserFunction("apoc.index.between.count")
    @Description("apoc.index.between.count(node1,'TYPE',node2,'prop:value*') YIELD value - lucene query on relationship index with the given type name bound by either or both sides (each node parameter can be null)")
    public long betweenCount(@Name("from") Node from, @Name("type") String type, @Name("to") Node to, @Name("query") String query) throws Exception {
        if (!db.index().existsForRelationships(type)) return 0L;
        return (long) db.index().forRelationships(type).query(query,from,to).size();
    }

    @Procedure(mode = Mode.READ)
    @Description("out(node,'TYPE','prop:value*') YIELD node - lucene query on relationship index with the given type name for *outgoing* relationship of the given node, *returns end-nodes*")
    public Stream<WeightedNodeResult> out(@Name("from") Node from, @Name("type") String type, @Name("query") String query) throws Exception {
        if (!db.index().existsForRelationships(type)) return Stream.empty();
        return toWeightedRelationshipResult(db.index().forRelationships(type).query(query,from,null)).map((w) -> new WeightedNodeResult(w.rel.getEndNode(), w.weight));
    }

    @UserFunction("apoc.index.out.count")
    @Description("apoc.index.out.count(node,'TYPE','prop:value*') YIELD value - lucene query on relationship index with the given type name for *outgoing* relationship of the given node, *returns count-end-nodes*")
    public long outCount(@Name("from") Node from, @Name("type") String type, @Name("query") String query) throws Exception {
        if (!db.index().existsForRelationships(type)) return 0L;
        return (long) db.index().forRelationships(type).query(query,from,null).size();
    }

    // CALL apoc.index.in(philz, 'CHECKIN', 'on:2010-*')
    @Procedure(mode = Mode.READ)
    @Description("apoc.index.in(node,'TYPE','prop:value*') YIELD node lucene query on relationship index with the given type name for *incoming* relationship of the given node, *returns start-nodes*")
    public Stream<WeightedNodeResult> in(@Name("to") Node to, @Name("type") String type, @Name("query") String query) throws Exception {
        if (!db.index().existsForRelationships(type)) return Stream.empty();
        return toWeightedRelationshipResult(db.index().forRelationships(type).query(query,null,to)).map((w) -> new WeightedNodeResult(w.rel.getStartNode(), w.weight));
    }

    @UserFunction("apoc.index.in.count")
    @Description("apoc.index.in.count(node1,'TYPE',node2,'prop:value*') YIELD value - lucene query on relationship index with the given type name for *incoming* relationship of the given node, *returns count-start-nodes*")
    public long inCount(@Name("to") Node to, @Name("type") String type, @Name("query") String query) throws Exception {
        if (!db.index().existsForRelationships(type)) return 0L;
        return (long) db.index().forRelationships(type).query(query,null,to).size();
    }

    // CALL apoc.index.addNode(joe, ['name','age','city'])
    @Procedure(mode = Mode.WRITE)
    @Description("apoc.index.addNode(node,['prop1',...]) add node to an index for each label it has")
    public void addNode(@Name("node") Node node, @Name("properties") List<String> propKeys) {
        for (Label label : node.getLabels()) {
            addNodeByLabel(label.name(),node,propKeys);
        }
    }

    @Procedure(mode = Mode.WRITE)
    @Description("apoc.index.addNodeMap(node,{key:value}) add node to an index for each label it has with the given attributes which can also be computed")
    public void addNodeMap(@Name("node") Node node, @Name("properties") Map<String, Object> document) {
        for (Label label : node.getLabels()) {
            addNodeMapByName(label.name(), node, document);
        }
    }

    @Procedure(mode = Mode.WRITE)
    @Description("apoc.index.addNodeMapByName(index, node,{key:value}) add node to an index for each label it has with the given attributes which can also be computed")
    public void addNodeMapByName(@Name("index") String index, @Name("node") Node node, @Name("properties") Map<String, Object> document) {
        indexEntityWithMap(node, document, getNodeIndex(index, FULL_TEXT));
    }

    // CALL apoc.index.addNode(joe, 'Person', ['name','age','city'])
    @Procedure(mode = Mode.WRITE)
    @Description("apoc.index.addNodeByLabel(node,'Label',['prop1',...]) add node to an index for the given label")
    public void addNodeByLabel(@Name("label") String label, @Name("node") Node node, @Name("properties") List<String> propKeys) {
        indexEntityProperties(node, propKeys, getNodeIndex(label,FULL_TEXT));
    }

    // CALL apoc.index.addNodeByName('name', joe, ['name','age','city'])
    @Procedure(mode = Mode.WRITE)
    @Description("apoc.index.addNodeByName('name',node,['prop1',...]) add node to an index for the given name")
    public void addNodeByName(@Name("name") String name, @Name("node") Node node, @Name("properties") List<String> propKeys) {
        Index<Node> index = getNodeIndex(name, null);
        indexEntityProperties(node, propKeys, index);
    }

    @Procedure(mode = Mode.WRITE)
    @Description("apoc.index.addRelationship(rel,['prop1',...]) add relationship to an index for its type")
    public void addRelationship(@Name("relationship") Relationship rel, @Name("properties") List<String> propKeys) {
        RelationshipIndex index = getRelationshipIndex(rel.getType().name(), FULL_TEXT);
        indexEntityProperties(rel, propKeys, index);
    }

    @Procedure(mode = Mode.WRITE)
    @Description("apoc.index.addRelationshipMap(rel,{key:value}) add relationship to an index for its type indexing the given document which can be computed")
    public void addRelationshipMap(@Name("relationship") Relationship rel, @Name("docuemnt") Map<String,Object> document) {
        addRelationshipMapByName(rel.getType().name(), rel, document);
    }

    @Procedure(mode = Mode.WRITE)
    @Description("apoc.index.addRelationshipMapByName(index, rel,{key:value}) add relationship to an index for its type indexing the given document which can be computed")
    public void addRelationshipMapByName(@Name("index") String indexName, @Name("relationship") Relationship rel, @Name("docuemnt") Map<String,Object> document) {
        indexEntityWithMap(rel, document, getRelationshipIndex(indexName, FULL_TEXT));
    }

    @Procedure(mode = Mode.WRITE)
    @Description("apoc.index.addRelationshipByName('name',rel,['prop1',...]) add relationship to an index for the given name")
    public void addRelationshipByName(@Name("name") String name, @Name("relationship") Relationship rel, @Name("properties") List<String> propKeys) {
        RelationshipIndex index = getRelationshipIndex(name, null);
        indexEntityProperties(rel, propKeys, index);
    }

    private <T extends PropertyContainer> void indexEntityProperties(T pc, List<String> propKeys, org.neo4j.graphdb.index.Index<T> index) {
        Map<String, Object> properties = pc.getProperties(propKeys.toArray(new String[propKeys.size()]));
        indexEntityWithMap(pc, properties, index);
    }

    private <T extends PropertyContainer> void indexEntityWithMap(T pc, Map<String, Object> document, Index<T> index) {
        index.remove(pc);
        document.forEach((key, value) -> {
            index.remove(pc, key);
            index.add(pc, key, collectionToArray(value));
        });
    }


    public static Object collectionToArray(Object value) {
        if (value == null) return null;
        if (!(value instanceof Collection)) return value;
        Collection coll = (Collection) value;
        if (coll.isEmpty()) return EMPTY_ARRAY;
        return coll.toArray(new Object[coll.size()]);
    }

    final static Object[] EMPTY_ARRAY=new Object[0];


    // CALL apoc.index.removeNodeByName('name', joe)
    @Procedure(mode = Mode.WRITE)
    @Description("apoc.index.removeNodeByName('name',node) remove node from an index for the given name")
    public void removeNodeByName(@Name("name") String name, @Name("node") Node node) {
        Index<Node> index = getNodeIndex(name, null);
        index.remove(node);
    }

    // CALL apoc.index.removeRelationshipByName('name', checkin)
    @Procedure(mode = Mode.WRITE)
    @Description("apoc.index.removeRelationshipByName('name',rel) remove relationship from an index for the given name")
    public void removeRelationshipByName(@Name("name") String name, @Name("relationship") Relationship rel) {
        RelationshipIndex index = getRelationshipIndex(name, null);
        index.remove(rel);
    }

    /* WIP
    private TermQuery query(Map<String,Object> params) {
        Object sort = params.remove("sort");
        if (sort != null) {
            if (sort instanceof String) {
                new SortField(sort,null);
            }
        }
        Object top = params.remove("top");
        params.remove("score");
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            Object value = entry.getValue();
            if (value == null) continue;
            if (value instanceof String) {
                Query.class
            }
            if (value instanceof Number) {
                Number num = (Number) value;
                if (value instanceof Double || value instanceof Float) {
                    builder.add(NumericRangeQuery.newDoubleRange(entry.getKey(), num.doubleValue(), num.doubleValue(), true, true));
                }
                builder.add(NumericRangeQuery.newLongRange(entry.getKey(), num.longValue(), num.longValue(), true, true));
            }
            if (value.getClass().isArray()) {

            }
        }
        NumericRangeQuery.newDoubleRange(field,min, max, minInclusive,maxInclusive);
        NumericRangeQuery.newLongRange(field,min, max, minInclusive,maxInclusive);
        new Sort(new SortField())

    }
    */

}
