
// line 1 "WarcParser.rl"
// recompile: ragel -J WarcParser.rl -o WarcParser.java
// diagram:   ragel -Vp WarcParser.rl | dot -Tpng | feh -

package org.netpreserve.jwarc;

import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.*;

import static java.nio.charset.StandardCharsets.ISO_8859_1;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.charset.StandardCharsets.US_ASCII;


// line 185 "WarcParser.rl"


/**
 * Low-level WARC record parser.
 * <p>
 * Unless you're doing something advanced (like non-blocking IO) you should use the higher-level {@link WarcReader}
 * class instead.
 */
public class WarcParser extends MessageParser {
    private int entryState;
    private int cs;
    private long position;
    private byte[] buf = new byte[256];
    private int bufPos;
    private int endOfText;
    private int major;
    private int minor;
    private String name;
    private String protocol = "WARC";
    private Map<String,List<String>> headerMap;
    private static final DateTimeFormatter arcTimeFormat = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");

    public static WarcParser newWarcFieldsParser() {
        return new WarcParser(warc_en_warc_fields);
    }

    public WarcParser() {
        this(warc_start);
    }

    private WarcParser(int entryState) {
        this.entryState = entryState;
        reset();
    }

    public void reset() {
        cs = entryState;
        position = 0;
        bufPos = 0;
        endOfText = 0;
        major = 0;
        minor = 0;
        name = null;
        headerMap = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        if (buf.length > 4096) {
            buf = new byte[4096];
        }
    }

	/**
	 * Sets the lenient mode for the WarcParser.
	 * <p>
	 * When enabled, this causes the parser to follow the specification less strictly,
	 * allowing reading of non-compliant records by:
	 * <ul>
	 *   <li>permitting ASCII control characters in header field names and values
	 *   <li>allowing lines to end with LF instead of CRLF
	 *   <li>permitting multi-digit WARC minor versions like "0.18"
	 * </ul>
	 * Calling this method also resets the state of the parser.
	 */
    public void setLenient(boolean lenient) {
        if (warcFieldsMode()) {
            entryState = lenient ? warc_en_warc_fields_lenient : warc_en_warc_fields;
        } else {
            entryState = lenient ? warc_en_any_header_lenient : warc_start;
        }
        reset();
    }

    private boolean warcFieldsMode() {
        return entryState == warc_en_warc_fields || entryState == warc_en_warc_fields_lenient;
    }

    public boolean isFinished() {
        return cs >= warc_first_final;
    }

    public boolean isError() {
        return cs == warc_error;
    }

    public void parse(ByteBuffer data) {
        int p = data.position();
        int pe = data.limit();

        
// line 113 "WarcParser.java"
	{
	int _klen;
	int _trans = 0;
	int _acts;
	int _nacts;
	int _keys;
	int _goto_targ = 0;

	_goto: while (true) {
	switch ( _goto_targ ) {
	case 0:
	if ( p == pe ) {
		_goto_targ = 4;
		continue _goto;
	}
	if ( cs == 0 ) {
		_goto_targ = 5;
		continue _goto;
	}
case 1:
	_match: do {
	_keys = _warc_key_offsets[cs];
	_trans = _warc_index_offsets[cs];
	_klen = _warc_single_lengths[cs];
	if ( _klen > 0 ) {
		int _lower = _keys;
		int _mid;
		int _upper = _keys + _klen - 1;
		while (true) {
			if ( _upper < _lower )
				break;

			_mid = _lower + ((_upper-_lower) >> 1);
			if ( ( (data.get(p) & 0xff)) < _warc_trans_keys[_mid] )
				_upper = _mid - 1;
			else if ( ( (data.get(p) & 0xff)) > _warc_trans_keys[_mid] )
				_lower = _mid + 1;
			else {
				_trans += (_mid - _keys);
				break _match;
			}
		}
		_keys += _klen;
		_trans += _klen;
	}

	_klen = _warc_range_lengths[cs];
	if ( _klen > 0 ) {
		int _lower = _keys;
		int _mid;
		int _upper = _keys + (_klen<<1) - 2;
		while (true) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( ( (data.get(p) & 0xff)) < _warc_trans_keys[_mid] )
				_upper = _mid - 2;
			else if ( ( (data.get(p) & 0xff)) > _warc_trans_keys[_mid+1] )
				_lower = _mid + 2;
			else {
				_trans += ((_mid - _keys)>>1);
				break _match;
			}
		}
		_trans += _klen;
	}
	} while (false);

	_trans = _warc_indicies[_trans];
	cs = _warc_trans_targs[_trans];

	if ( _warc_trans_actions[_trans] != 0 ) {
		_acts = _warc_trans_actions[_trans];
		_nacts = (int) _warc_actions[_acts++];
		while ( _nacts-- > 0 )
	{
			switch ( _warc_actions[_acts++] )
			{
	case 0:
// line 27 "WarcParser.rl"
	{ push(data.get(p)); }
	break;
	case 1:
// line 28 "WarcParser.rl"
	{ major = major * 10 + data.get(p) - '0'; }
	break;
	case 2:
// line 29 "WarcParser.rl"
	{ minor = minor * 10 + data.get(p) - '0'; }
	break;
	case 3:
// line 30 "WarcParser.rl"
	{ endOfText = bufPos; }
	break;
	case 4:
// line 32 "WarcParser.rl"
	{
    if (bufPos > 0) {
        bufPos = endOfText;
        push((byte)' ');
    }
}
	break;
	case 5:
// line 39 "WarcParser.rl"
	{
    name = new String(buf, 0, bufPos, US_ASCII);
    bufPos = 0;
}
	break;
	case 6:
// line 44 "WarcParser.rl"
	{
    String value = new String(buf, 0, endOfText, UTF_8);
    headerMap.computeIfAbsent(name, n -> new ArrayList<>()).add(value);
    bufPos = 0;
    endOfText = 0;
}
	break;
	case 7:
// line 51 "WarcParser.rl"
	{
    String url = new String(buf, 0, bufPos, ISO_8859_1);
    if (url.startsWith("filedesc://")) {
        setHeader("WARC-Type", "warcinfo");
        setHeader("WARC-Filename", url.substring("filedesc://".length()));
        setHeader("Content-Type", "text/plain");
    } else if (url.startsWith("dns:")) {
        setHeader("WARC-Type", "response");
        setHeader("Content-Type", "text/dns");
        setHeader("WARC-Target-URI", url);
     } else {
        setHeader("WARC-Type", "response");
        setHeader("Content-Type", "application/http;msgtype=response");
        setHeader("WARC-Target-URI", url);
    }
    bufPos = 0;
}
	break;
	case 8:
// line 69 "WarcParser.rl"
	{
    setHeader("WARC-IP-Address", new String(buf, 0, bufPos, US_ASCII));
    bufPos = 0;
}
	break;
	case 9:
// line 74 "WarcParser.rl"
	{
    String arcDate = new String(buf, 0, bufPos, US_ASCII);
    // Some WARC files have been seen in the wild with truncated dates
    if (arcDate.length() < 14) {
        emitWarning("ARC date too short (" + arcDate.length() + " digits)");
        arcDate = arcDate + "00000000000000".substring(arcDate.length());
    } else if (arcDate.length() > 14) {
        emitWarning("ARC date too long (" + arcDate.length() + " digits)");
        arcDate = arcDate.substring(0, 14);
    }
    try {
        Instant instant = LocalDateTime.parse(arcDate, arcTimeFormat).toInstant(ZoneOffset.UTC);
        setHeader("WARC-Date", instant.toString());
    } catch (DateTimeParseException e) {
        emitWarning("ARC date not parsable");
    }
    bufPos = 0;
}
	break;
	case 10:
// line 93 "WarcParser.rl"
	{
    setHeader("Content-Length", new String(buf, 0, bufPos, US_ASCII));
    bufPos = 0;
}
	break;
	case 11:
// line 98 "WarcParser.rl"
	{
    bufPos = 0;
}
	break;
	case 12:
// line 102 "WarcParser.rl"
	{
    protocol = "ARC";
    major = 1;
    minor = 1;
}
	break;
	case 13:
// line 182 "WarcParser.rl"
	{ { p += 1; _goto_targ = 5; if (true)  continue _goto;} }
	break;
	case 14:
// line 183 "WarcParser.rl"
	{ { p += 1; _goto_targ = 5; if (true)  continue _goto;} }
	break;
// line 311 "WarcParser.java"
			}
		}
	}

case 2:
	if ( cs == 0 ) {
		_goto_targ = 5;
		continue _goto;
	}
	if ( ++p != pe ) {
		_goto_targ = 1;
		continue _goto;
	}
case 4:
case 5:
	}
	break; }
	}

// line 272 "WarcParser.rl"

        position += p - data.position();
        data.position(p);
    }

    public boolean parse(ReadableByteChannel channel, ByteBuffer buffer) throws IOException {
        while (true) {
            parse(buffer);
            if (isFinished()) {
                return true;
            }
            if (isError()) {
                throw new ParsingException("invalid WARC record at position " + position + ": "
                        + getErrorContext(buffer, buffer.position(), 40));
            }
            buffer.compact();
            int n = channel.read(buffer);
            buffer.flip();
            if (n < 0) {
                if (position > 0) {
                    throw new EOFException();
                }
                return false;
            }
        }
    }

    private void push(byte b) {
        if (bufPos >= buf.length) {
            buf = Arrays.copyOf(buf, buf.length * 2);
        }
        buf[bufPos++] = b;
    }

    public MessageHeaders headers() {
        return new MessageHeaders(headerMap);
    }

    public MessageVersion version() {
        return new MessageVersion(protocol, major, minor);
    }

    public long position() {
        return position;
    }

    private void setHeader(String name, String value) {
        List<String> list = new ArrayList<>();
        list.add(value);
        headerMap.put(name, list);
    }

    
// line 385 "WarcParser.java"
private static byte[] init__warc_actions_0()
{
	return new byte [] {
	    0,    1,    0,    1,    1,    1,    2,    1,    3,    1,    4,    1,
	    5,    1,    6,    1,    7,    1,    8,    1,    9,    1,   11,    1,
	   13,    1,   14,    2,    3,    0,    2,    4,    0,    2,    6,    0,
	    2,    6,   13,    3,   10,   12,   13,    3,   10,   12,   14
	};
}

private static final byte _warc_actions[] = init__warc_actions_0();


private static short[] init__warc_key_offsets_0()
{
	return new short [] {
	    0,    0,    4,    7,   10,   12,   15,   17,   20,   24,   26,   28,
	   30,   32,   34,   36,   38,   40,   43,   60,   62,   64,   67,   70,
	   74,   76,   78,   80,   82,   84,   87,   89,   91,   93,   96,  112,
	  129,  148,  152,  157,  160,  177,  193,  208,  226,  233,  236,  240,
	  258,  275,  292,  309,  327,  344,  362,  379,  388,  399,  411,  423,
	  436,  445,  454,  463,  472,  483,  495,  504,  513,  524,  536,  540,
	  544,  549,  567,  584,  602,  619,  638,  644,  648,  652,  657,  675,
	  692,  709,  727,  744,  763,  768,  772,  776,  780,  782,  784,  788,
	  792,  796,  802,  806,  810,  814,  832,  850,  868,  885,  904,  910,
	  914,  918,  922,  926,  930,  932,  934,  936,  942,  948,  951,  954,
	  957,  960,  963,  966,  969,  972,  975,  978,  981,  984,  987,  990,
	  993,  996,  999, 1002, 1005, 1006, 1008, 1011, 1013, 1016, 1018, 1021,
	 1024, 1025, 1028, 1029, 1032, 1033, 1036, 1037, 1040, 1041, 1042, 1043,
	 1044, 1046, 1049, 1051, 1054, 1055, 1071, 1072, 1088, 1094, 1095, 1113,
	 1119, 1125, 1131, 1136, 1139, 1143, 1147, 1151, 1156, 1160, 1164, 1168,
	 1170, 1186, 1187, 1203, 1209, 1210, 1228, 1234, 1240, 1246, 1250, 1253,
	 1256, 1258, 1261, 1263, 1266, 1270, 1272, 1274, 1276, 1278, 1280, 1282,
	 1284, 1286, 1289, 1306, 1308, 1310, 1313, 1316, 1320, 1322, 1324, 1326,
	 1328, 1330, 1333, 1335, 1337, 1339, 1342, 1358, 1375, 1394, 1398, 1403,
	 1406, 1423, 1439, 1454, 1472, 1479, 1482, 1486, 1504, 1521, 1538, 1555,
	 1573, 1590, 1608, 1625, 1634, 1645, 1657, 1669, 1682, 1691, 1700, 1709,
	 1718, 1729, 1741, 1750, 1759, 1770, 1782, 1786, 1790, 1795, 1813, 1830,
	 1848, 1865, 1884, 1890, 1894, 1898, 1903, 1921, 1938, 1955, 1973, 1990,
	 2009, 2014, 2018, 2022, 2026, 2028, 2030, 2034, 2038, 2042, 2048, 2052,
	 2056, 2060, 2078, 2096, 2114, 2131, 2150, 2156, 2160, 2164, 2168, 2172,
	 2176, 2178, 2180, 2182, 2188, 2194, 2197, 2200, 2203, 2206, 2209, 2212,
	 2215, 2218, 2221, 2224, 2227, 2230, 2233, 2236, 2239, 2242, 2245, 2248,
	 2251, 2252, 2254, 2257, 2259, 2262, 2264, 2267, 2270, 2271, 2274, 2275,
	 2278, 2279, 2282, 2283, 2286, 2287, 2288, 2289, 2290, 2292, 2295, 2297,
	 2301, 2306, 2309, 2313, 2317, 2321, 2326, 2330, 2334, 2338, 2340, 2342,
	 2342, 2342, 2342
	};
}

private static final short _warc_key_offsets[] = init__warc_key_offsets_0();


private static char[] init__warc_trans_keys_0()
{
	return new char [] {
	   10,   87,   97,  122,   10,   97,  122,   10,   97,  122,   97,  122,
	   58,   97,  122,   10,   32,   48,   49,   57,   32,   46,   48,   57,
	   48,   57,   48,   57,   48,   57,   48,   57,   48,   57,   48,   57,
	   48,   57,   48,   57,   32,   48,   57,   10,   32,   33,  124,  126,
	   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,
	   10,   32,   48,   57,   10,   48,   57,   10,   48,   57,   10,   32,
	   48,   57,   10,   32,   10,   32,   10,   32,   10,   32,   48,   57,
	   32,   48,   57,   10,   32,   10,   32,   48,   57,   10,   48,   57,
	   10,   32,   33,   47,  124,  126,   35,   39,   42,   43,   45,   57,
	   65,   90,   94,  122,   10,   32,   33,  124,  126,   35,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,
	   33,   59,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   10,   32,   59,    9,   32,   59,   48,
	   57,    9,   32,   59,    9,   32,   33,  124,  126,   35,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,   33,   61,  124,
	  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,
	  122,   34,  124,  126,   33,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   32,   33,   59,  124,  126,   35,   39,
	   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,    9,   34,
	   92,   32,  126,  128,  255,    9,   32,   59,    0,  191,  194,  244,
	    9,   10,   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,
	   48,   57,   65,   90,   94,  122,    9,   32,   33,  124,  126,   35,
	   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,
	   33,   61,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,   10,   33,   61,  124,  126,   35,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,   33,
	   61,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,
	   90,   94,  122,   10,   33,   61,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,   10,   32,   34,  124,  126,   33,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,   34,   92,
	   33,  126,  128,  255,    9,   34,   92,   32,   47,   48,   57,   58,
	  126,  128,  255,    9,   10,   34,   92,   32,   47,   48,   57,   58,
	  126,  128,  255,    9,   10,   34,   92,   32,   47,   48,   57,   58,
	  126,  128,  255,    9,   10,   32,   34,   92,   33,   47,   48,   57,
	   58,  126,  128,  255,    9,   10,   32,   34,   92,   33,  126,  128,
	  255,    9,   10,   32,   34,   92,   33,  126,  128,  255,    9,   10,
	   32,   34,   92,   33,  126,  128,  255,    9,   10,   32,   34,   92,
	   33,  126,  128,  255,    9,   34,   92,   32,   47,   48,   57,   58,
	  126,  128,  255,    9,   32,   34,   92,   33,   47,   48,   57,   58,
	  126,  128,  255,    9,   10,   32,   34,   92,   33,  126,  128,  255,
	    9,   10,   32,   34,   92,   33,  126,  128,  255,    9,   34,   92,
	   32,   47,   48,   57,   58,  126,  128,  255,    9,   10,   34,   92,
	   32,   47,   48,   57,   58,  126,  128,  255,    9,   10,   32,   59,
	    9,   10,   32,   59,    9,   32,   59,   48,   57,    9,   10,   32,
	   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,
	   90,   94,  122,    9,   32,   33,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,   10,   32,   34,  124,  126,   33,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,   33,   59,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,   10,   32,    0,  191,  194,  244,    9,   10,   32,   59,
	    9,   10,   32,   59,    9,   32,   59,   48,   57,    9,   10,   32,
	   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,
	   90,   94,  122,    9,   32,   33,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,   32,   33,   61,  124,
	  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,
	  122,   10,   32,   33,   61,  124,  126,   35,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,   10,   32,   34,  124,  126,
	   33,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,
	    9,   10,   32,   33,   59,  124,  126,   35,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,    9,   32,   59,   48,   57,
	   10,   32,   48,   57,   10,   32,   48,   57,   10,   32,   48,   57,
	   10,   32,   10,   32,   10,   32,   48,   57,   10,   32,   48,   57,
	   10,   32,   48,   57,   10,   32,    0,  191,  194,  244,    9,   10,
	   32,   59,    9,   10,   32,   59,    9,   10,   32,   59,    9,   10,
	   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   10,   32,   33,  124,  126,   35,   39,
	   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,
	   33,   61,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,   10,   32,   34,  124,  126,   33,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,
	   33,   59,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   10,   32,   59,   48,   57,   10,   32,
	   48,   57,   10,   32,   48,   57,   10,   32,   48,   57,   10,   32,
	   48,   57,   10,   32,   48,   57,   10,   32,   10,   32,   48,   57,
	   10,   32,    0,  191,  194,  244,   10,   32,    0,  191,  194,  244,
	   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,
	   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,
	   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,
	   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,
	   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,
	   46,   48,   57,   48,   57,   46,   48,   57,   48,   57,   32,   48,
	   57,   32,   48,   57,   32,   46,   48,   57,   46,   46,   48,   57,
	   46,   46,   48,   57,   46,   46,   48,   57,   65,   82,   67,   47,
	   48,   57,   46,   48,   57,   48,   57,   13,   48,   57,   10,   13,
	   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,
	   90,   94,  122,   10,   33,   58,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,    9,   13,   32,  127,
	    0,   31,   10,    9,   13,   32,   33,  124,  126,   35,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,    9,   13,   32,
	  127,    0,   31,    9,   13,   32,  127,    0,   31,    9,   13,   32,
	  127,    0,   31,    9,   10,   13,   32,   58,   10,   13,   58,    9,
	   10,   13,   32,    9,   10,   13,   32,    9,   10,   13,   32,    9,
	   10,   13,   32,   58,    9,   10,   13,   32,    9,   10,   13,   32,
	    9,   10,   13,   32,   10,   13,   13,   33,  124,  126,   35,   39,
	   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,   33,
	   58,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,
	   90,   94,  122,    9,   13,   32,  127,    0,   31,   10,    9,   13,
	   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   13,   32,  127,    0,   31,    9,   13,
	   32,  127,    0,   31,    9,   13,   32,  127,    0,   31,   10,   87,
	   97,  122,   10,   97,  122,   10,   97,  122,   97,  122,   58,   97,
	  122,   10,   32,   48,   49,   57,   32,   46,   48,   57,   48,   57,
	   48,   57,   48,   57,   48,   57,   48,   57,   48,   57,   48,   57,
	   48,   57,   32,   48,   57,   10,   32,   33,  124,  126,   35,   39,
	   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,
	   48,   57,   10,   48,   57,   10,   48,   57,   10,   32,   48,   57,
	   10,   32,   10,   32,   10,   32,   10,   32,   48,   57,   32,   48,
	   57,   10,   32,   10,   32,   48,   57,   10,   48,   57,   10,   32,
	   33,   47,  124,  126,   35,   39,   42,   43,   45,   57,   65,   90,
	   94,  122,   10,   32,   33,  124,  126,   35,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,   33,   59,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,    9,   10,   32,   59,    9,   32,   59,   48,   57,    9,
	   32,   59,    9,   32,   33,  124,  126,   35,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,   33,   61,  124,  126,   35,
	   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   34,
	  124,  126,   33,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,    9,   32,   33,   59,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,    9,   34,   92,   32,
	  126,  128,  255,    9,   32,   59,    0,  191,  194,  244,    9,   10,
	   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   32,   33,  124,  126,   35,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,   33,   61,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,   10,   33,   61,  124,  126,   35,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,  124,
	  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,
	  122,   10,   33,   61,  124,  126,   35,   39,   42,   43,   45,   46,
	   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,  124,  126,
	   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,
	   10,   32,   34,  124,  126,   33,   39,   42,   43,   45,   46,   48,
	   57,   65,   90,   94,  122,    9,   10,   32,   34,   92,   33,  126,
	  128,  255,    9,   34,   92,   32,   47,   48,   57,   58,  126,  128,
	  255,    9,   10,   34,   92,   32,   47,   48,   57,   58,  126,  128,
	  255,    9,   10,   34,   92,   32,   47,   48,   57,   58,  126,  128,
	  255,    9,   10,   32,   34,   92,   33,   47,   48,   57,   58,  126,
	  128,  255,    9,   10,   32,   34,   92,   33,  126,  128,  255,    9,
	   10,   32,   34,   92,   33,  126,  128,  255,    9,   10,   32,   34,
	   92,   33,  126,  128,  255,    9,   10,   32,   34,   92,   33,  126,
	  128,  255,    9,   34,   92,   32,   47,   48,   57,   58,  126,  128,
	  255,    9,   32,   34,   92,   33,   47,   48,   57,   58,  126,  128,
	  255,    9,   10,   32,   34,   92,   33,  126,  128,  255,    9,   10,
	   32,   34,   92,   33,  126,  128,  255,    9,   34,   92,   32,   47,
	   48,   57,   58,  126,  128,  255,    9,   10,   34,   92,   32,   47,
	   48,   57,   58,  126,  128,  255,    9,   10,   32,   59,    9,   10,
	   32,   59,    9,   32,   59,   48,   57,    9,   10,   32,   33,  124,
	  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,
	  122,    9,   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,
	   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,  124,  126,
	   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,
	   10,   32,   34,  124,  126,   33,   39,   42,   43,   45,   46,   48,
	   57,   65,   90,   94,  122,    9,   10,   32,   33,   59,  124,  126,
	   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,
	   10,   32,    0,  191,  194,  244,    9,   10,   32,   59,    9,   10,
	   32,   59,    9,   32,   59,   48,   57,    9,   10,   32,   33,  124,
	  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,
	  122,    9,   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,
	   48,   57,   65,   90,   94,  122,   32,   33,   61,  124,  126,   35,
	   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,
	   32,   33,   61,  124,  126,   35,   39,   42,   43,   45,   46,   48,
	   57,   65,   90,   94,  122,   10,   32,   34,  124,  126,   33,   39,
	   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,    9,   10,
	   32,   33,   59,  124,  126,   35,   39,   42,   43,   45,   46,   48,
	   57,   65,   90,   94,  122,    9,   32,   59,   48,   57,   10,   32,
	   48,   57,   10,   32,   48,   57,   10,   32,   48,   57,   10,   32,
	   10,   32,   10,   32,   48,   57,   10,   32,   48,   57,   10,   32,
	   48,   57,   10,   32,    0,  191,  194,  244,    9,   10,   32,   59,
	    9,   10,   32,   59,    9,   10,   32,   59,    9,   10,   32,   33,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,    9,   10,   32,   33,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,   10,   32,   34,  124,  126,   33,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,   33,   59,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,    9,   10,   32,   59,   48,   57,   10,   32,   48,   57,
	   10,   32,   48,   57,   10,   32,   48,   57,   10,   32,   48,   57,
	   10,   32,   48,   57,   10,   32,   10,   32,   48,   57,   10,   32,
	    0,  191,  194,  244,   10,   32,    0,  191,  194,  244,   32,   48,
	   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,
	   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,
	   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,
	   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,
	   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,   46,   48,
	   57,   48,   57,   46,   48,   57,   48,   57,   32,   48,   57,   32,
	   48,   57,   32,   46,   48,   57,   46,   46,   48,   57,   46,   46,
	   48,   57,   46,   46,   48,   57,   65,   82,   67,   47,   48,   57,
	   46,   48,   57,   48,   57,   10,   13,   48,   57,    9,   10,   13,
	   32,   58,   10,   13,   58,    9,   10,   13,   32,    9,   10,   13,
	   32,    9,   10,   13,   32,    9,   10,   13,   32,   58,    9,   10,
	   13,   32,    9,   10,   13,   32,    9,   10,   13,   32,   10,   13,
	   10,   13,    0
	};
}

private static final char _warc_trans_keys[] = init__warc_trans_keys_0();


private static byte[] init__warc_single_lengths_0()
{
	return new byte [] {
	    0,    2,    1,    1,    0,    1,    2,    1,    2,    0,    0,    0,
	    0,    0,    0,    0,    0,    1,    5,    2,    0,    1,    1,    2,
	    2,    2,    2,    2,    0,    1,    2,    2,    0,    1,    6,    5,
	    7,    4,    3,    3,    5,    4,    3,    6,    3,    3,    0,    6,
	    5,    5,    5,    6,    5,    6,    5,    5,    3,    4,    4,    5,
	    5,    5,    5,    5,    3,    4,    5,    5,    3,    4,    4,    4,
	    3,    6,    5,    6,    5,    7,    2,    4,    4,    3,    6,    5,
	    5,    6,    5,    7,    3,    2,    2,    2,    2,    2,    2,    2,
	    2,    2,    4,    4,    4,    6,    6,    6,    5,    7,    4,    2,
	    2,    2,    2,    2,    2,    2,    0,    2,    2,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    0,    1,    0,    1,    0,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    0,    1,    0,    1,    1,    4,    1,    4,    4,    1,    6,    4,
	    4,    4,    5,    3,    4,    4,    4,    5,    4,    4,    4,    2,
	    4,    1,    4,    4,    1,    6,    4,    4,    4,    2,    1,    1,
	    0,    1,    2,    1,    2,    0,    0,    0,    0,    0,    0,    0,
	    0,    1,    5,    2,    0,    1,    1,    2,    2,    2,    2,    2,
	    0,    1,    2,    2,    0,    1,    6,    5,    7,    4,    3,    3,
	    5,    4,    3,    6,    3,    3,    0,    6,    5,    5,    5,    6,
	    5,    6,    5,    5,    3,    4,    4,    5,    5,    5,    5,    5,
	    3,    4,    5,    5,    3,    4,    4,    4,    3,    6,    5,    6,
	    5,    7,    2,    4,    4,    3,    6,    5,    5,    6,    5,    7,
	    3,    2,    2,    2,    2,    2,    2,    2,    2,    2,    4,    4,
	    4,    6,    6,    6,    5,    7,    4,    2,    2,    2,    2,    2,
	    2,    2,    0,    2,    2,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    0,    1,    0,    1,    0,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    0,    1,    0,    2,
	    5,    3,    4,    4,    4,    5,    4,    4,    4,    2,    2,    0,
	    0,    0,    0
	};
}

private static final byte _warc_single_lengths[] = init__warc_single_lengths_0();


private static byte[] init__warc_range_lengths_0()
{
	return new byte [] {
	    0,    1,    1,    1,    1,    1,    0,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    6,    0,    1,    1,    1,    1,
	    0,    0,    0,    0,    1,    1,    0,    0,    1,    1,    5,    6,
	    6,    0,    1,    0,    6,    6,    6,    6,    2,    0,    2,    6,
	    6,    6,    6,    6,    6,    6,    6,    2,    4,    4,    4,    4,
	    2,    2,    2,    2,    4,    4,    2,    2,    4,    4,    0,    0,
	    1,    6,    6,    6,    6,    6,    2,    0,    0,    1,    6,    6,
	    6,    6,    6,    6,    1,    1,    1,    1,    0,    0,    1,    1,
	    1,    2,    0,    0,    0,    6,    6,    6,    6,    6,    1,    1,
	    1,    1,    1,    1,    0,    0,    1,    2,    2,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    0,    1,    1,    1,    1,    1,    1,    1,
	    0,    1,    0,    1,    0,    1,    0,    1,    0,    0,    0,    0,
	    1,    1,    1,    1,    0,    6,    0,    6,    1,    0,    6,    1,
	    1,    1,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    6,    0,    6,    1,    0,    6,    1,    1,    1,    1,    1,    1,
	    1,    1,    0,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    6,    0,    1,    1,    1,    1,    0,    0,    0,    0,
	    1,    1,    0,    0,    1,    1,    5,    6,    6,    0,    1,    0,
	    6,    6,    6,    6,    2,    0,    2,    6,    6,    6,    6,    6,
	    6,    6,    6,    2,    4,    4,    4,    4,    2,    2,    2,    2,
	    4,    4,    2,    2,    4,    4,    0,    0,    1,    6,    6,    6,
	    6,    6,    2,    0,    0,    1,    6,    6,    6,    6,    6,    6,
	    1,    1,    1,    1,    0,    0,    1,    1,    1,    2,    0,    0,
	    0,    6,    6,    6,    6,    6,    1,    1,    1,    1,    1,    1,
	    0,    0,    1,    2,    2,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    0,    1,    1,    1,    1,    1,    1,    1,    0,    1,    0,    1,
	    0,    1,    0,    1,    0,    0,    0,    0,    1,    1,    1,    1,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    0
	};
}

private static final byte _warc_range_lengths[] = init__warc_range_lengths_0();


private static short[] init__warc_index_offsets_0()
{
	return new short [] {
	    0,    0,    4,    7,   10,   12,   15,   18,   21,   25,   27,   29,
	   31,   33,   35,   37,   39,   41,   44,   56,   59,   61,   64,   67,
	   71,   74,   77,   80,   83,   85,   88,   91,   94,   96,   99,  111,
	  123,  137,  142,  147,  151,  163,  174,  184,  197,  203,  207,  210,
	  223,  235,  247,  259,  272,  284,  297,  309,  317,  325,  334,  343,
	  353,  361,  369,  377,  385,  393,  402,  410,  418,  426,  435,  440,
	  445,  450,  463,  475,  488,  500,  514,  519,  524,  529,  534,  547,
	  559,  571,  584,  596,  610,  615,  619,  623,  627,  630,  633,  637,
	  641,  645,  650,  655,  660,  665,  678,  691,  704,  716,  730,  736,
	  740,  744,  748,  752,  756,  759,  762,  764,  769,  774,  777,  780,
	  783,  786,  789,  792,  795,  798,  801,  804,  807,  810,  813,  816,
	  819,  822,  825,  828,  831,  833,  835,  838,  840,  843,  845,  848,
	  851,  853,  856,  858,  861,  863,  866,  868,  871,  873,  875,  877,
	  879,  881,  884,  886,  889,  891,  902,  904,  915,  921,  923,  936,
	  942,  948,  954,  960,  964,  969,  974,  979,  985,  990,  995, 1000,
	 1003, 1014, 1016, 1027, 1033, 1035, 1048, 1054, 1060, 1066, 1070, 1073,
	 1076, 1078, 1081, 1084, 1087, 1091, 1093, 1095, 1097, 1099, 1101, 1103,
	 1105, 1107, 1110, 1122, 1125, 1127, 1130, 1133, 1137, 1140, 1143, 1146,
	 1149, 1151, 1154, 1157, 1160, 1162, 1165, 1177, 1189, 1203, 1208, 1213,
	 1217, 1229, 1240, 1250, 1263, 1269, 1273, 1276, 1289, 1301, 1313, 1325,
	 1338, 1350, 1363, 1375, 1383, 1391, 1400, 1409, 1419, 1427, 1435, 1443,
	 1451, 1459, 1468, 1476, 1484, 1492, 1501, 1506, 1511, 1516, 1529, 1541,
	 1554, 1566, 1580, 1585, 1590, 1595, 1600, 1613, 1625, 1637, 1650, 1662,
	 1676, 1681, 1685, 1689, 1693, 1696, 1699, 1703, 1707, 1711, 1716, 1721,
	 1726, 1731, 1744, 1757, 1770, 1782, 1796, 1802, 1806, 1810, 1814, 1818,
	 1822, 1825, 1828, 1830, 1835, 1840, 1843, 1846, 1849, 1852, 1855, 1858,
	 1861, 1864, 1867, 1870, 1873, 1876, 1879, 1882, 1885, 1888, 1891, 1894,
	 1897, 1899, 1901, 1904, 1906, 1909, 1911, 1914, 1917, 1919, 1922, 1924,
	 1927, 1929, 1932, 1934, 1937, 1939, 1941, 1943, 1945, 1947, 1950, 1952,
	 1956, 1962, 1966, 1971, 1976, 1981, 1987, 1992, 1997, 2002, 2005, 2008,
	 2009, 2010, 2011
	};
}

private static final short _warc_index_offsets[] = init__warc_index_offsets_0();


private static short[] init__warc_indicies_0()
{
	return new short [] {
	    0,    2,    3,    1,    4,    3,    1,    5,    3,    1,    3,    1,
	    6,    3,    1,    1,    7,    6,    8,    9,    1,   10,   11,   12,
	    1,   13,    1,   14,    1,   15,    1,   16,    1,   17,    1,   18,
	    1,   19,    1,   20,    1,   21,   22,    1,    1,   24,   25,   25,
	   25,   25,   25,   25,   25,   25,   25,   23,    1,   24,   23,   26,
	    1,   27,   28,    1,   27,   29,    1,   27,   30,   31,    1,    1,
	    1,   32,    1,   33,   32,    1,    1,   34,    1,   35,   34,   36,
	    1,   37,   36,    1,    1,    1,   38,    1,   39,   38,   31,    1,
	   27,   31,    1,    1,   24,   25,   40,   25,   25,   25,   25,   25,
	   25,   25,   23,    1,   24,   41,   41,   41,   41,   41,   41,   41,
	   41,   41,   23,   42,    1,   43,   41,   44,   41,   41,   41,   41,
	   41,   41,   41,   41,   23,   42,    1,   43,   44,   23,   45,   45,
	   46,   26,    1,   45,   45,   46,    1,   46,   46,   47,   47,   47,
	   47,   47,   47,   47,   47,   47,    1,   47,   48,   47,   47,   47,
	   47,   47,   47,   47,   47,    1,   50,   49,   49,   49,   49,   49,
	   49,   49,   49,    1,   45,   43,   49,   46,   49,   49,   49,   49,
	   49,   49,   49,   49,    1,   50,   51,   52,   50,   50,    1,   45,
	   43,   46,    1,   50,   50,    1,   44,    1,   53,   54,   54,   54,
	   54,   54,   54,   54,   54,   54,   23,   46,   46,   47,   47,   47,
	   47,   47,   47,   55,   47,   47,    1,   27,   47,   48,   47,   47,
	   47,   47,   47,   56,   47,   47,    1,   27,   47,   48,   47,   47,
	   47,   47,   47,   57,   47,   47,    1,   27,   30,   47,   48,   47,
	   47,   47,   47,   47,   58,   47,   47,    1,   27,   47,   48,   47,
	   47,   47,   47,   47,   58,   47,   47,    1,    1,   24,   54,   59,
	   54,   54,   54,   54,   54,   54,   54,   54,   23,    1,   24,   60,
	   41,   41,   41,   41,   41,   41,   41,   41,   23,   60,    1,   61,
	   42,   62,   60,   60,   23,   50,   51,   52,   50,   63,   50,   50,
	    1,   50,   27,   51,   52,   50,   64,   50,   50,    1,   50,   27,
	   51,   52,   50,   65,   50,   50,    1,   50,   27,   66,   51,   52,
	   50,   67,   50,   50,    1,   68,    1,   50,   69,   70,   68,   68,
	   32,   68,    1,   71,   69,   70,   68,   68,   32,   72,    1,   50,
	   73,   74,   72,   72,   34,   72,    1,   75,   73,   74,   72,   72,
	   34,   50,   51,   52,   50,   76,   50,   50,    1,   50,   77,   51,
	   52,   50,   76,   50,   50,    1,   78,    1,   50,   79,   80,   78,
	   78,   38,   78,    1,   81,   79,   80,   78,   78,   38,   50,   51,
	   52,   50,   67,   50,   50,    1,   50,   27,   51,   52,   50,   67,
	   50,   50,    1,   82,    1,   43,   83,   38,   82,    1,   84,   83,
	   38,   45,   45,   46,   31,    1,   83,    1,   85,   86,   86,   86,
	   86,   86,   86,   86,   86,   86,   38,   46,   46,   47,   47,   47,
	   47,   47,   47,   58,   47,   47,    1,    1,   39,   86,   87,   86,
	   86,   86,   86,   86,   86,   86,   86,   38,    1,   39,   78,   88,
	   88,   88,   88,   88,   88,   88,   88,   38,   82,    1,   43,   88,
	   83,   88,   88,   88,   88,   88,   88,   88,   88,   38,   50,   81,
	   78,   78,   38,   89,    1,   90,   91,   34,   89,    1,   92,   91,
	   34,   45,   45,   46,   36,    1,   91,    1,   93,   94,   94,   94,
	   94,   94,   94,   94,   94,   94,   34,   46,   46,   47,   47,   47,
	   47,   47,   47,   95,   47,   47,    1,   37,   47,   48,   47,   47,
	   47,   47,   47,   95,   47,   47,    1,    1,   35,   94,   96,   94,
	   94,   94,   94,   94,   94,   94,   94,   34,    1,   35,   72,   97,
	   97,   97,   97,   97,   97,   97,   97,   34,   89,    1,   90,   97,
	   91,   97,   97,   97,   97,   97,   97,   97,   97,   34,   45,   45,
	   46,   98,    1,   27,   37,   99,    1,   27,   37,  100,    1,   27,
	  101,  102,    1,    1,    1,  103,    1,  104,  103,    1,    1,  105,
	   34,   27,   35,  105,   34,   27,   37,  102,    1,   50,   75,   72,
	   72,   34,  106,    1,  107,  108,   32,  106,    1,  109,  108,   32,
	   89,    1,   45,   91,   34,  108,    1,  110,  111,  111,  111,  111,
	  111,  111,  111,  111,  111,   32,   91,    1,   46,   94,   94,   94,
	   94,   94,   94,   94,   94,   94,   34,    1,   33,  111,  112,  111,
	  111,  111,  111,  111,  111,  111,  111,   32,    1,   33,   68,  113,
	  113,  113,  113,  113,  113,  113,  113,   32,  106,    1,  107,  113,
	  108,  113,  113,  113,  113,  113,  113,  113,  113,   32,   89,    1,
	   45,   91,  114,   34,   27,   35,  115,   34,   27,   35,  116,   34,
	   27,  117,  105,   34,    1,    1,  118,   32,    1,  119,  118,   32,
	    1,    1,  120,    1,  121,  120,  102,    1,   50,   71,   68,   68,
	   32,   50,   61,   60,   60,   23,   21,  122,    1,   21,  123,    1,
	   21,  124,    1,   21,  125,    1,   21,  126,    1,   21,  127,    1,
	   21,  128,    1,   21,  129,    1,   21,  130,    1,   21,  131,    1,
	   21,  132,    1,   21,  133,    1,   21,  134,    1,   21,  135,    1,
	   21,  136,    1,   21,  137,    1,   21,  138,    1,   21,  139,    1,
	   21,  140,    1,   21,    1,  141,    1,  142,  143,    1,  144,    1,
	  145,  146,    1,  147,    1,   10,  148,    1,   10,  149,    1,   10,
	    1,  145,  150,    1,  145,    1,  142,  151,    1,  142,    1,   11,
	  152,    1,   11,    1,   11,   12,    1,  153,    1,  154,    1,  155,
	    1,  156,    1,  157,    1,  158,  157,    1,  159,    1,  160,  159,
	    1,  161,    1,  162,  163,  163,  163,  163,  163,  163,  163,  163,
	  163,    1,  164,    1,  163,  165,  163,  163,  163,  163,  163,  163,
	  163,  163,    1,  166,  167,  166,    1,    1,  168,  169,    1,  170,
	  171,  170,  172,  172,  172,  172,  172,  172,  172,  172,  172,    1,
	  170,  173,  170,    1,    1,  174,  175,  176,  175,    1,    1,  168,
	  177,  167,  177,    1,    1,  168,    1,  179,  180,    1,    1,  178,
	    1,    1,  181,  178,  183,  184,  185,  183,  182,  186,  187,  188,
	  186,  182,  189,  184,  185,  189,  182,  191,  192,  193,  191,    1,
	  190,  191,  195,  196,  191,  194,  197,  184,  185,  197,  182,  197,
	    1,  197,  197,  182,  179,  180,    1,  198,  199,  199,  199,  199,
	  199,  199,  199,  199,  199,    1,  200,    1,  199,  201,  199,  199,
	  199,  199,  199,  199,  199,  199,    1,  202,  203,  202,    1,    1,
	  204,  205,    1,  206,  207,  206,  208,  208,  208,  208,  208,  208,
	  208,  208,  208,    1,  206,  209,  206,    1,    1,  210,  211,  212,
	  211,    1,    1,  204,  213,  203,  213,    1,    1,  204,  214,  215,
	  216,    1,  217,  216,    1,  218,  216,    1,  216,    1,  219,  216,
	    1,    1,  220,  219,  221,  222,    1,  223,  224,  225,    1,  226,
	    1,  227,    1,  228,    1,  229,    1,  230,    1,  231,    1,  232,
	    1,  233,    1,  234,  235,    1,    1,  237,  238,  238,  238,  238,
	  238,  238,  238,  238,  238,  236,    1,  237,  236,  239,    1,  240,
	  241,    1,  240,  242,    1,  240,  243,  244,    1,    1,    1,  245,
	    1,  246,  245,    1,    1,  247,    1,  248,  247,  249,    1,  250,
	  249,    1,    1,    1,  251,    1,  252,  251,  244,    1,  240,  244,
	    1,    1,  237,  238,  253,  238,  238,  238,  238,  238,  238,  238,
	  236,    1,  237,  254,  254,  254,  254,  254,  254,  254,  254,  254,
	  236,  255,    1,  256,  254,  257,  254,  254,  254,  254,  254,  254,
	  254,  254,  236,  255,    1,  256,  257,  236,  258,  258,  259,  239,
	    1,  258,  258,  259,    1,  259,  259,  260,  260,  260,  260,  260,
	  260,  260,  260,  260,    1,  260,  261,  260,  260,  260,  260,  260,
	  260,  260,  260,    1,  263,  262,  262,  262,  262,  262,  262,  262,
	  262,    1,  258,  256,  262,  259,  262,  262,  262,  262,  262,  262,
	  262,  262,    1,  263,  264,  265,  263,  263,    1,  258,  256,  259,
	    1,  263,  263,    1,  257,    1,  266,  267,  267,  267,  267,  267,
	  267,  267,  267,  267,  236,  259,  259,  260,  260,  260,  260,  260,
	  260,  268,  260,  260,    1,  240,  260,  261,  260,  260,  260,  260,
	  260,  269,  260,  260,    1,  240,  260,  261,  260,  260,  260,  260,
	  260,  270,  260,  260,    1,  240,  243,  260,  261,  260,  260,  260,
	  260,  260,  271,  260,  260,    1,  240,  260,  261,  260,  260,  260,
	  260,  260,  271,  260,  260,    1,    1,  237,  267,  272,  267,  267,
	  267,  267,  267,  267,  267,  267,  236,    1,  237,  273,  254,  254,
	  254,  254,  254,  254,  254,  254,  236,  273,    1,  274,  255,  275,
	  273,  273,  236,  263,  264,  265,  263,  276,  263,  263,    1,  263,
	  240,  264,  265,  263,  277,  263,  263,    1,  263,  240,  264,  265,
	  263,  278,  263,  263,    1,  263,  240,  279,  264,  265,  263,  280,
	  263,  263,    1,  281,    1,  263,  282,  283,  281,  281,  245,  281,
	    1,  284,  282,  283,  281,  281,  245,  285,    1,  263,  286,  287,
	  285,  285,  247,  285,    1,  288,  286,  287,  285,  285,  247,  263,
	  264,  265,  263,  289,  263,  263,    1,  263,  290,  264,  265,  263,
	  289,  263,  263,    1,  291,    1,  263,  292,  293,  291,  291,  251,
	  291,    1,  294,  292,  293,  291,  291,  251,  263,  264,  265,  263,
	  280,  263,  263,    1,  263,  240,  264,  265,  263,  280,  263,  263,
	    1,  295,    1,  256,  296,  251,  295,    1,  297,  296,  251,  258,
	  258,  259,  244,    1,  296,    1,  298,  299,  299,  299,  299,  299,
	  299,  299,  299,  299,  251,  259,  259,  260,  260,  260,  260,  260,
	  260,  271,  260,  260,    1,    1,  252,  299,  300,  299,  299,  299,
	  299,  299,  299,  299,  299,  251,    1,  252,  291,  301,  301,  301,
	  301,  301,  301,  301,  301,  251,  295,    1,  256,  301,  296,  301,
	  301,  301,  301,  301,  301,  301,  301,  251,  263,  294,  291,  291,
	  251,  302,    1,  303,  304,  247,  302,    1,  305,  304,  247,  258,
	  258,  259,  249,    1,  304,    1,  306,  307,  307,  307,  307,  307,
	  307,  307,  307,  307,  247,  259,  259,  260,  260,  260,  260,  260,
	  260,  308,  260,  260,    1,  250,  260,  261,  260,  260,  260,  260,
	  260,  308,  260,  260,    1,    1,  248,  307,  309,  307,  307,  307,
	  307,  307,  307,  307,  307,  247,    1,  248,  285,  310,  310,  310,
	  310,  310,  310,  310,  310,  247,  302,    1,  303,  310,  304,  310,
	  310,  310,  310,  310,  310,  310,  310,  247,  258,  258,  259,  311,
	    1,  240,  250,  312,    1,  240,  250,  313,    1,  240,  314,  315,
	    1,    1,    1,  316,    1,  317,  316,    1,    1,  318,  247,  240,
	  248,  318,  247,  240,  250,  315,    1,  263,  288,  285,  285,  247,
	  319,    1,  320,  321,  245,  319,    1,  322,  321,  245,  302,    1,
	  258,  304,  247,  321,    1,  323,  324,  324,  324,  324,  324,  324,
	  324,  324,  324,  245,  304,    1,  259,  307,  307,  307,  307,  307,
	  307,  307,  307,  307,  247,    1,  246,  324,  325,  324,  324,  324,
	  324,  324,  324,  324,  324,  245,    1,  246,  281,  326,  326,  326,
	  326,  326,  326,  326,  326,  245,  319,    1,  320,  326,  321,  326,
	  326,  326,  326,  326,  326,  326,  326,  245,  302,    1,  258,  304,
	  327,  247,  240,  248,  328,  247,  240,  248,  329,  247,  240,  330,
	  318,  247,    1,    1,  331,  245,    1,  332,  331,  245,    1,    1,
	  333,    1,  334,  333,  315,    1,  263,  284,  281,  281,  245,  263,
	  274,  273,  273,  236,  234,  335,    1,  234,  336,    1,  234,  337,
	    1,  234,  338,    1,  234,  339,    1,  234,  340,    1,  234,  341,
	    1,  234,  342,    1,  234,  343,    1,  234,  344,    1,  234,  345,
	    1,  234,  346,    1,  234,  347,    1,  234,  348,    1,  234,  349,
	    1,  234,  350,    1,  234,  351,    1,  234,  352,    1,  234,  353,
	    1,  234,    1,  354,    1,  355,  356,    1,  357,    1,  358,  359,
	    1,  360,    1,  223,  361,    1,  223,  362,    1,  223,    1,  358,
	  363,    1,  358,    1,  355,  364,    1,  355,    1,  224,  365,    1,
	  224,    1,  224,  225,    1,  366,    1,  367,    1,  368,    1,  369,
	    1,  370,    1,  371,  370,    1,  372,    1,  373,  374,  372,    1,
	    1,  376,  377,    1,    1,  375,    1,    1,  378,  375,  380,  381,
	  382,  380,  379,  383,  384,  385,  383,  379,  386,  381,  382,  386,
	  379,  388,  389,  390,  388,    1,  387,  388,  392,  393,  388,  391,
	  394,  381,  382,  394,  379,  394,    1,  394,  394,  379,  376,  377,
	    1,  373,  374,    1,    1,    1,    1,    1,    0
	};
}

private static final short _warc_indicies[] = init__warc_indicies_0();


private static short[] init__warc_trans_targs_0()
{
	return new short [] {
	    2,    0,  152,    5,    3,    4,    6,    7,    8,  151,    9,  137,
	  149,   10,   11,   12,   13,   14,   15,   16,   17,   18,  117,   19,
	   20,   34,   21,  359,   22,   23,   24,   33,   25,   26,   27,   28,
	   29,   30,   31,   32,   35,   36,   37,   38,   47,   39,   40,   41,
	   42,   43,   44,   45,   46,   48,   53,   49,   50,   51,   52,   54,
	   55,   56,  116,   57,   58,   59,   60,   69,   61,   98,  115,   62,
	   63,   79,   97,   64,   65,   66,   67,   70,   78,   68,   71,   73,
	   72,   74,   75,   76,   77,   80,   88,   82,   81,   83,   85,   84,
	   86,   87,   89,   90,   91,   92,   96,   93,   94,   95,   99,  106,
	  101,  100,  102,  103,  104,  105,  107,  108,  109,  110,  111,  112,
	  113,  114,  118,  119,  120,  121,  122,  123,  124,  125,  126,  127,
	  128,  129,  130,  131,  132,  133,  134,  135,  136,  138,  139,  147,
	  140,  141,  145,  142,  143,  144,  146,  148,  150,  153,  154,  155,
	  156,  157,  158,  159,  160,  161,  162,  163,  359,  164,  164,  165,
	  168,  166,  167,  162,  163,  165,  168,  169,  165,  169,  171,  360,
	  179,  172,  173,  172,  175,  177,  174,  175,  177,  174,  171,  176,
	  360,  179,  173,  175,  177,  178,  181,  182,  361,  183,  183,  184,
	  187,  185,  186,  181,  182,  184,  187,  188,  184,  188,  190,  340,
	  193,  191,  192,  194,  195,  196,  339,  197,  325,  337,  198,  199,
	  200,  201,  202,  203,  204,  205,  206,  305,  207,  208,  222,  209,
	  362,  210,  211,  212,  221,  213,  214,  215,  216,  217,  218,  219,
	  220,  223,  224,  225,  226,  235,  227,  228,  229,  230,  231,  232,
	  233,  234,  236,  241,  237,  238,  239,  240,  242,  243,  244,  304,
	  245,  246,  247,  248,  257,  249,  286,  303,  250,  251,  267,  285,
	  252,  253,  254,  255,  258,  266,  256,  259,  261,  260,  262,  263,
	  264,  265,  268,  276,  270,  269,  271,  273,  272,  274,  275,  277,
	  278,  279,  280,  284,  281,  282,  283,  287,  294,  289,  288,  290,
	  291,  292,  293,  295,  296,  297,  298,  299,  300,  301,  302,  306,
	  307,  308,  309,  310,  311,  312,  313,  314,  315,  316,  317,  318,
	  319,  320,  321,  322,  323,  324,  326,  327,  335,  328,  329,  333,
	  330,  331,  332,  334,  336,  338,  341,  342,  343,  344,  345,  346,
	  347,  348,  358,  349,  362,  357,  350,  351,  350,  353,  355,  352,
	  353,  355,  352,  349,  354,  362,  357,  351,  353,  355,  356
	};
}

private static final short _warc_trans_targs[] = init__warc_trans_targs_0();


private static byte[] init__warc_trans_actions_0()
{
	return new byte [] {
	    0,    0,    0,    1,    0,    0,    1,   15,    1,    1,   17,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,   19,    1,    0,
	    0,    0,    1,   43,    1,    1,   21,    1,    0,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,    0,    1,    1,    1,    1,    0,
	    0,    0,    0,    1,    1,    1,   21,    1,    0,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    1,    1,    1,   21,    1,    0,    0,    1,    0,    0,
	    0,    0,    0,    0,    0,    0,    1,    1,    1,   21,    0,    0,
	    0,    0,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    0,    0,    0,
	    0,    3,    0,    5,    0,    0,    0,    1,   25,   11,    0,    0,
	    1,    0,    0,   13,   33,    9,   30,   27,    7,    1,    1,    0,
	    0,   11,    1,    0,    0,    1,   27,    7,   27,    1,   33,    0,
	   13,   13,   30,    9,   30,    1,    0,    1,    0,   11,    0,    0,
	    1,    0,    0,   13,   33,    9,   30,   27,    7,    1,    0,    0,
	    1,    0,    0,    1,   15,    1,    1,   17,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,   19,    1,    0,    0,    0,    1,
	   39,    1,    1,   21,    1,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    0,    0,    1,    1,    1,    1,    0,    0,    0,    0,
	    1,    1,    1,   21,    1,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    1,
	    1,    1,   21,    1,    0,    0,    1,    0,    0,    0,    0,    0,
	    0,    0,    0,    1,    1,    1,   21,    0,    0,    0,    0,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    0,    0,    0,    0,    3,    0,
	    5,    0,    0,    1,   23,    0,   11,    1,    0,    0,    1,   27,
	    7,   27,    1,   33,    0,   36,   13,   30,    9,   30,    1
	};
}

private static final byte _warc_trans_actions[] = init__warc_trans_actions_0();


static final int warc_start = 1;
static final int warc_first_final = 359;
static final int warc_error = 0;

static final int warc_en_warc_fields_lenient = 170;
static final int warc_en_warc_fields = 180;
static final int warc_en_any_header_lenient = 189;
static final int warc_en_any_header = 1;


// line 325 "WarcParser.rl"
}