
// line 1 "WarcParser.rl"
// recompile: ragel -J WarcParser.rl -o WarcParser.java
// diagram:   ragel -Vp WarcParser.rl | dot -Tpng | feh -

package org.netpreserve.jwarc;

import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.*;

import static java.nio.charset.StandardCharsets.ISO_8859_1;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.charset.StandardCharsets.US_ASCII;


// line 195 "WarcParser.rl"


/**
 * Low-level WARC record parser.
 * <p>
 * Unless you're doing something advanced (like non-blocking IO) you should use the higher-level {@link WarcReader}
 * class instead.
 */
public class WarcParser extends MessageParser {
    private int entryState;
    private int cs;
    private long position;
    private byte[] buf = new byte[256];
    private int bufPos;
    private int endOfText;
    private int major;
    private int minor;
    private String name;
    private String protocol = "WARC";
    private Map<String,List<String>> headerMap;
    private static final DateTimeFormatter arcTimeFormat = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");

    public static WarcParser newWarcFieldsParser() {
        return new WarcParser(warc_en_warc_fields);
    }

    public WarcParser() {
        this(warc_start);
    }

    private WarcParser(int entryState) {
        this.entryState = entryState;
        reset();
    }

    public void reset() {
        cs = entryState;
        position = 0;
        bufPos = 0;
        endOfText = 0;
        major = 0;
        minor = 0;
        name = null;
        headerMap = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        if (buf.length > 4096) {
            buf = new byte[4096];
        }
    }

	/**
	 * Sets the lenient mode for the WarcParser.
	 * <p>
	 * When enabled, this causes the parser to follow the specification less strictly,
	 * allowing reading of non-compliant records by:
	 * <ul>
	 *   <li>permitting ASCII control characters in header field names and values
	 *   <li>allowing lines to end with LF instead of CRLF
	 *   <li>permitting multi-digit WARC minor versions like "0.18"
	 * </ul>
	 * Calling this method also resets the state of the parser.
	 */
    public void setLenient(boolean lenient) {
        if (warcFieldsMode()) {
            entryState = lenient ? warc_en_warc_fields_lenient : warc_en_warc_fields;
        } else {
            entryState = lenient ? warc_en_any_header_lenient : warc_start;
        }
        reset();
    }

    private boolean warcFieldsMode() {
        return entryState == warc_en_warc_fields || entryState == warc_en_warc_fields_lenient;
    }

    public boolean isFinished() {
        return cs >= warc_first_final;
    }

    public boolean isError() {
        return cs == warc_error;
    }

    public void parse(ByteBuffer data) {
        int p = data.position();
        int pe = data.limit();

        
// line 113 "WarcParser.java"
	{
	int _klen;
	int _trans = 0;
	int _acts;
	int _nacts;
	int _keys;
	int _goto_targ = 0;

	_goto: while (true) {
	switch ( _goto_targ ) {
	case 0:
	if ( p == pe ) {
		_goto_targ = 4;
		continue _goto;
	}
	if ( cs == 0 ) {
		_goto_targ = 5;
		continue _goto;
	}
case 1:
	_match: do {
	_keys = _warc_key_offsets[cs];
	_trans = _warc_index_offsets[cs];
	_klen = _warc_single_lengths[cs];
	if ( _klen > 0 ) {
		int _lower = _keys;
		int _mid;
		int _upper = _keys + _klen - 1;
		while (true) {
			if ( _upper < _lower )
				break;

			_mid = _lower + ((_upper-_lower) >> 1);
			if ( ( (data.get(p) & 0xff)) < _warc_trans_keys[_mid] )
				_upper = _mid - 1;
			else if ( ( (data.get(p) & 0xff)) > _warc_trans_keys[_mid] )
				_lower = _mid + 1;
			else {
				_trans += (_mid - _keys);
				break _match;
			}
		}
		_keys += _klen;
		_trans += _klen;
	}

	_klen = _warc_range_lengths[cs];
	if ( _klen > 0 ) {
		int _lower = _keys;
		int _mid;
		int _upper = _keys + (_klen<<1) - 2;
		while (true) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( ( (data.get(p) & 0xff)) < _warc_trans_keys[_mid] )
				_upper = _mid - 2;
			else if ( ( (data.get(p) & 0xff)) > _warc_trans_keys[_mid+1] )
				_lower = _mid + 2;
			else {
				_trans += ((_mid - _keys)>>1);
				break _match;
			}
		}
		_trans += _klen;
	}
	} while (false);

	_trans = _warc_indicies[_trans];
	cs = _warc_trans_targs[_trans];

	if ( _warc_trans_actions[_trans] != 0 ) {
		_acts = _warc_trans_actions[_trans];
		_nacts = (int) _warc_actions[_acts++];
		while ( _nacts-- > 0 )
	{
			switch ( _warc_actions[_acts++] )
			{
	case 0:
// line 27 "WarcParser.rl"
	{ push(data.get(p)); }
	break;
	case 1:
// line 28 "WarcParser.rl"
	{ major = major * 10 + data.get(p) - '0'; }
	break;
	case 2:
// line 29 "WarcParser.rl"
	{ minor = minor * 10 + data.get(p) - '0'; }
	break;
	case 3:
// line 30 "WarcParser.rl"
	{ endOfText = bufPos; }
	break;
	case 4:
// line 32 "WarcParser.rl"
	{
    if (bufPos > 0) {
        bufPos = endOfText;
        push((byte)' ');
    }
}
	break;
	case 5:
// line 39 "WarcParser.rl"
	{
    name = new String(buf, 0, bufPos, US_ASCII);
    bufPos = 0;
}
	break;
	case 6:
// line 44 "WarcParser.rl"
	{
    String value = new String(buf, 0, endOfText, UTF_8);
    headerMap.computeIfAbsent(name, n -> new ArrayList<>()).add(value);
    bufPos = 0;
    endOfText = 0;
}
	break;
	case 7:
// line 51 "WarcParser.rl"
	{
    String url = new String(buf, 0, bufPos, ISO_8859_1);
    if (url.startsWith("filedesc://")) {
        setHeader("WARC-Type", "warcinfo");
        setHeader("WARC-Filename", url.substring("filedesc://".length()));
        setHeader("Content-Type", "text/plain");
    } else if (url.startsWith("dns:")) {
        setHeader("WARC-Type", "response");
        setHeader("Content-Type", "text/dns");
        setHeader("WARC-Target-URI", url);
     } else {
        setHeader("WARC-Type", "response");
        setHeader("Content-Type", "application/http;msgtype=response");
        setHeader("WARC-Target-URI", url);
    }
    bufPos = 0;
}
	break;
	case 8:
// line 69 "WarcParser.rl"
	{
    setHeader("WARC-IP-Address", new String(buf, 0, bufPos, US_ASCII));
    bufPos = 0;
}
	break;
	case 9:
// line 74 "WarcParser.rl"
	{
    String arcDate = new String(buf, 0, bufPos, US_ASCII);
    // Some WARC files have been seen in the wild with truncated dates
    if (arcDate.length() < 14) {
        emitWarning("ARC date too short (" + arcDate.length() + " digits)");
        arcDate = arcDate + "00000000000000".substring(arcDate.length());
    } else if (arcDate.length() > 14) {
        emitWarning("ARC date too long (" + arcDate.length() + " digits)");
        arcDate = arcDate.substring(0, 14);
    }
    try {
        Instant instant = LocalDateTime.parse(arcDate, arcTimeFormat).toInstant(ZoneOffset.UTC);
        setHeader("WARC-Date", instant.toString());
    } catch (DateTimeParseException e) {
        emitWarning("ARC date not parsable");
    }
    bufPos = 0;
}
	break;
	case 10:
// line 93 "WarcParser.rl"
	{
    String contentType = new String(buf, 0, bufPos, US_ASCII);
    // Normally the ARC content-type header is copied from the HTTP header, but as a special case "alexa/dat"
    // indicates the body is not a HTTP message.
    if (contentType.equals("alexa/dat")) {
        setHeader("Content-Type", "alexa/dat");
    }
    bufPos = 0;
}
	break;
	case 11:
// line 103 "WarcParser.rl"
	{
    setHeader("Content-Length", new String(buf, 0, bufPos, US_ASCII));
    bufPos = 0;
}
	break;
	case 12:
// line 108 "WarcParser.rl"
	{
    bufPos = 0;
}
	break;
	case 13:
// line 112 "WarcParser.rl"
	{
    protocol = "ARC";
    major = 1;
    minor = 1;
}
	break;
	case 14:
// line 192 "WarcParser.rl"
	{ { p += 1; _goto_targ = 5; if (true)  continue _goto;} }
	break;
	case 15:
// line 193 "WarcParser.rl"
	{ { p += 1; _goto_targ = 5; if (true)  continue _goto;} }
	break;
// line 323 "WarcParser.java"
			}
		}
	}

case 2:
	if ( cs == 0 ) {
		_goto_targ = 5;
		continue _goto;
	}
	if ( ++p != pe ) {
		_goto_targ = 1;
		continue _goto;
	}
case 4:
case 5:
	}
	break; }
	}

// line 282 "WarcParser.rl"

        position += p - data.position();
        data.position(p);
    }

    public boolean parse(ReadableByteChannel channel, ByteBuffer buffer) throws IOException {
        while (true) {
            parse(buffer);
            if (isFinished()) {
                return true;
            }
            if (isError()) {
                throw new ParsingException("invalid WARC record at position " + position + ": "
                        + getErrorContext(buffer, buffer.position(), 40));
            }
            buffer.compact();
            int n = channel.read(buffer);
            buffer.flip();
            if (n < 0) {
                if (position > 0) {
                    throw new EOFException();
                }
                return false;
            }
        }
    }

    private void push(byte b) {
        if (bufPos >= buf.length) {
            buf = Arrays.copyOf(buf, buf.length * 2);
        }
        buf[bufPos++] = b;
    }

    public MessageHeaders headers() {
        return new MessageHeaders(headerMap);
    }

    public MessageVersion version() {
        return new MessageVersion(protocol, major, minor);
    }

    public long position() {
        return position;
    }

    private void setHeader(String name, String value) {
        List<String> list = new ArrayList<>();
        list.add(value);
        headerMap.put(name, list);
    }

    
// line 397 "WarcParser.java"
private static byte[] init__warc_actions_0()
{
	return new byte [] {
	    0,    1,    0,    1,    1,    1,    2,    1,    3,    1,    4,    1,
	    5,    1,    6,    1,    7,    1,    8,    1,    9,    1,   10,    1,
	   12,    1,   14,    1,   15,    2,    3,    0,    2,    4,    0,    2,
	    6,    0,    2,    6,   14,    3,   11,   13,   14,    3,   11,   13,
	   15
	};
}

private static final byte _warc_actions[] = init__warc_actions_0();


private static short[] init__warc_key_offsets_0()
{
	return new short [] {
	    0,    0,    4,    7,   10,   12,   15,   17,   20,   24,   26,   28,
	   30,   32,   34,   36,   38,   40,   43,   60,   62,   64,   67,   70,
	   74,   76,   78,   80,   82,   84,   87,   89,   91,   93,   96,  112,
	  129,  148,  152,  157,  160,  177,  193,  208,  226,  233,  236,  240,
	  258,  275,  292,  309,  327,  344,  362,  379,  388,  399,  411,  423,
	  436,  445,  454,  463,  472,  483,  495,  504,  513,  524,  536,  540,
	  544,  549,  567,  584,  602,  619,  638,  644,  648,  652,  657,  675,
	  692,  709,  727,  744,  763,  768,  772,  776,  780,  782,  784,  788,
	  792,  796,  802,  806,  810,  814,  832,  850,  868,  885,  904,  910,
	  914,  918,  922,  926,  930,  932,  934,  936,  942,  948,  951,  954,
	  957,  960,  963,  966,  969,  972,  975,  978,  981,  984,  987,  990,
	  993,  996,  999, 1002, 1005, 1006, 1008, 1011, 1013, 1016, 1018, 1021,
	 1024, 1025, 1028, 1029, 1032, 1033, 1036, 1037, 1040, 1041, 1042, 1043,
	 1044, 1046, 1049, 1051, 1054, 1055, 1071, 1072, 1088, 1094, 1095, 1113,
	 1119, 1125, 1131, 1136, 1139, 1143, 1147, 1151, 1156, 1160, 1164, 1168,
	 1170, 1186, 1187, 1203, 1209, 1210, 1228, 1234, 1240, 1246, 1250, 1253,
	 1256, 1258, 1261, 1263, 1266, 1270, 1272, 1274, 1276, 1278, 1280, 1282,
	 1284, 1286, 1289, 1306, 1308, 1310, 1313, 1316, 1320, 1322, 1324, 1326,
	 1328, 1330, 1333, 1335, 1337, 1339, 1342, 1358, 1375, 1394, 1398, 1403,
	 1406, 1423, 1439, 1454, 1472, 1479, 1482, 1486, 1504, 1521, 1538, 1555,
	 1573, 1590, 1608, 1625, 1634, 1645, 1657, 1669, 1682, 1691, 1700, 1709,
	 1718, 1729, 1741, 1750, 1759, 1770, 1782, 1786, 1790, 1795, 1813, 1830,
	 1848, 1865, 1884, 1890, 1894, 1898, 1903, 1921, 1938, 1955, 1973, 1990,
	 2009, 2014, 2018, 2022, 2026, 2028, 2030, 2034, 2038, 2042, 2048, 2052,
	 2056, 2060, 2078, 2096, 2114, 2131, 2150, 2156, 2160, 2164, 2168, 2172,
	 2176, 2178, 2180, 2182, 2188, 2194, 2197, 2200, 2203, 2206, 2209, 2212,
	 2215, 2218, 2221, 2224, 2227, 2230, 2233, 2236, 2239, 2242, 2245, 2248,
	 2251, 2252, 2254, 2257, 2259, 2262, 2264, 2267, 2270, 2271, 2274, 2275,
	 2278, 2279, 2282, 2283, 2286, 2287, 2288, 2289, 2290, 2292, 2295, 2297,
	 2301, 2306, 2309, 2313, 2317, 2321, 2326, 2330, 2334, 2338, 2340, 2342,
	 2342, 2342, 2342
	};
}

private static final short _warc_key_offsets[] = init__warc_key_offsets_0();


private static char[] init__warc_trans_keys_0()
{
	return new char [] {
	   10,   87,   97,  122,   10,   97,  122,   10,   97,  122,   97,  122,
	   58,   97,  122,   10,   32,   48,   49,   57,   32,   46,   48,   57,
	   48,   57,   48,   57,   48,   57,   48,   57,   48,   57,   48,   57,
	   48,   57,   48,   57,   32,   48,   57,   10,   32,   33,  124,  126,
	   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,
	   10,   32,   48,   57,   10,   48,   57,   10,   48,   57,   10,   32,
	   48,   57,   10,   32,   10,   32,   10,   32,   10,   32,   48,   57,
	   32,   48,   57,   10,   32,   10,   32,   48,   57,   10,   48,   57,
	   10,   32,   33,   47,  124,  126,   35,   39,   42,   43,   45,   57,
	   65,   90,   94,  122,   10,   32,   33,  124,  126,   35,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,
	   33,   59,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   10,   32,   59,    9,   32,   59,   48,
	   57,    9,   32,   59,    9,   32,   33,  124,  126,   35,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,   33,   61,  124,
	  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,
	  122,   34,  124,  126,   33,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   32,   33,   59,  124,  126,   35,   39,
	   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,    9,   34,
	   92,   32,  126,  128,  255,    9,   32,   59,    0,  191,  194,  244,
	    9,   10,   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,
	   48,   57,   65,   90,   94,  122,    9,   32,   33,  124,  126,   35,
	   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,
	   33,   61,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,   10,   33,   61,  124,  126,   35,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,   33,
	   61,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,
	   90,   94,  122,   10,   33,   61,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,   10,   32,   34,  124,  126,   33,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,   34,   92,
	   33,  126,  128,  255,    9,   34,   92,   32,   47,   48,   57,   58,
	  126,  128,  255,    9,   10,   34,   92,   32,   47,   48,   57,   58,
	  126,  128,  255,    9,   10,   34,   92,   32,   47,   48,   57,   58,
	  126,  128,  255,    9,   10,   32,   34,   92,   33,   47,   48,   57,
	   58,  126,  128,  255,    9,   10,   32,   34,   92,   33,  126,  128,
	  255,    9,   10,   32,   34,   92,   33,  126,  128,  255,    9,   10,
	   32,   34,   92,   33,  126,  128,  255,    9,   10,   32,   34,   92,
	   33,  126,  128,  255,    9,   34,   92,   32,   47,   48,   57,   58,
	  126,  128,  255,    9,   32,   34,   92,   33,   47,   48,   57,   58,
	  126,  128,  255,    9,   10,   32,   34,   92,   33,  126,  128,  255,
	    9,   10,   32,   34,   92,   33,  126,  128,  255,    9,   34,   92,
	   32,   47,   48,   57,   58,  126,  128,  255,    9,   10,   34,   92,
	   32,   47,   48,   57,   58,  126,  128,  255,    9,   10,   32,   59,
	    9,   10,   32,   59,    9,   32,   59,   48,   57,    9,   10,   32,
	   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,
	   90,   94,  122,    9,   32,   33,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,   10,   32,   34,  124,  126,   33,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,   33,   59,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,   10,   32,    0,  191,  194,  244,    9,   10,   32,   59,
	    9,   10,   32,   59,    9,   32,   59,   48,   57,    9,   10,   32,
	   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,
	   90,   94,  122,    9,   32,   33,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,   32,   33,   61,  124,
	  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,
	  122,   10,   32,   33,   61,  124,  126,   35,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,   10,   32,   34,  124,  126,
	   33,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,
	    9,   10,   32,   33,   59,  124,  126,   35,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,    9,   32,   59,   48,   57,
	   10,   32,   48,   57,   10,   32,   48,   57,   10,   32,   48,   57,
	   10,   32,   10,   32,   10,   32,   48,   57,   10,   32,   48,   57,
	   10,   32,   48,   57,   10,   32,    0,  191,  194,  244,    9,   10,
	   32,   59,    9,   10,   32,   59,    9,   10,   32,   59,    9,   10,
	   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   10,   32,   33,  124,  126,   35,   39,
	   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,
	   33,   61,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,   10,   32,   34,  124,  126,   33,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,
	   33,   59,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   10,   32,   59,   48,   57,   10,   32,
	   48,   57,   10,   32,   48,   57,   10,   32,   48,   57,   10,   32,
	   48,   57,   10,   32,   48,   57,   10,   32,   10,   32,   48,   57,
	   10,   32,    0,  191,  194,  244,   10,   32,    0,  191,  194,  244,
	   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,
	   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,
	   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,
	   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,
	   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,
	   46,   48,   57,   48,   57,   46,   48,   57,   48,   57,   32,   48,
	   57,   32,   48,   57,   32,   46,   48,   57,   46,   46,   48,   57,
	   46,   46,   48,   57,   46,   46,   48,   57,   65,   82,   67,   47,
	   48,   57,   46,   48,   57,   48,   57,   13,   48,   57,   10,   13,
	   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,
	   90,   94,  122,   10,   33,   58,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,    9,   13,   32,  127,
	    0,   31,   10,    9,   13,   32,   33,  124,  126,   35,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,    9,   13,   32,
	  127,    0,   31,    9,   13,   32,  127,    0,   31,    9,   13,   32,
	  127,    0,   31,    9,   10,   13,   32,   58,   10,   13,   58,    9,
	   10,   13,   32,    9,   10,   13,   32,    9,   10,   13,   32,    9,
	   10,   13,   32,   58,    9,   10,   13,   32,    9,   10,   13,   32,
	    9,   10,   13,   32,   10,   13,   13,   33,  124,  126,   35,   39,
	   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,   33,
	   58,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,
	   90,   94,  122,    9,   13,   32,  127,    0,   31,   10,    9,   13,
	   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   13,   32,  127,    0,   31,    9,   13,
	   32,  127,    0,   31,    9,   13,   32,  127,    0,   31,   10,   87,
	   97,  122,   10,   97,  122,   10,   97,  122,   97,  122,   58,   97,
	  122,   10,   32,   48,   49,   57,   32,   46,   48,   57,   48,   57,
	   48,   57,   48,   57,   48,   57,   48,   57,   48,   57,   48,   57,
	   48,   57,   32,   48,   57,   10,   32,   33,  124,  126,   35,   39,
	   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,
	   48,   57,   10,   48,   57,   10,   48,   57,   10,   32,   48,   57,
	   10,   32,   10,   32,   10,   32,   10,   32,   48,   57,   32,   48,
	   57,   10,   32,   10,   32,   48,   57,   10,   48,   57,   10,   32,
	   33,   47,  124,  126,   35,   39,   42,   43,   45,   57,   65,   90,
	   94,  122,   10,   32,   33,  124,  126,   35,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,   33,   59,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,    9,   10,   32,   59,    9,   32,   59,   48,   57,    9,
	   32,   59,    9,   32,   33,  124,  126,   35,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,   33,   61,  124,  126,   35,
	   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   34,
	  124,  126,   33,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,    9,   32,   33,   59,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,    9,   34,   92,   32,
	  126,  128,  255,    9,   32,   59,    0,  191,  194,  244,    9,   10,
	   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,
	   65,   90,   94,  122,    9,   32,   33,  124,  126,   35,   39,   42,
	   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,   33,   61,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,   10,   33,   61,  124,  126,   35,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,  124,
	  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,
	  122,   10,   33,   61,  124,  126,   35,   39,   42,   43,   45,   46,
	   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,  124,  126,
	   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,
	   10,   32,   34,  124,  126,   33,   39,   42,   43,   45,   46,   48,
	   57,   65,   90,   94,  122,    9,   10,   32,   34,   92,   33,  126,
	  128,  255,    9,   34,   92,   32,   47,   48,   57,   58,  126,  128,
	  255,    9,   10,   34,   92,   32,   47,   48,   57,   58,  126,  128,
	  255,    9,   10,   34,   92,   32,   47,   48,   57,   58,  126,  128,
	  255,    9,   10,   32,   34,   92,   33,   47,   48,   57,   58,  126,
	  128,  255,    9,   10,   32,   34,   92,   33,  126,  128,  255,    9,
	   10,   32,   34,   92,   33,  126,  128,  255,    9,   10,   32,   34,
	   92,   33,  126,  128,  255,    9,   10,   32,   34,   92,   33,  126,
	  128,  255,    9,   34,   92,   32,   47,   48,   57,   58,  126,  128,
	  255,    9,   32,   34,   92,   33,   47,   48,   57,   58,  126,  128,
	  255,    9,   10,   32,   34,   92,   33,  126,  128,  255,    9,   10,
	   32,   34,   92,   33,  126,  128,  255,    9,   34,   92,   32,   47,
	   48,   57,   58,  126,  128,  255,    9,   10,   34,   92,   32,   47,
	   48,   57,   58,  126,  128,  255,    9,   10,   32,   59,    9,   10,
	   32,   59,    9,   32,   59,   48,   57,    9,   10,   32,   33,  124,
	  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,
	  122,    9,   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,
	   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,  124,  126,
	   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,
	   10,   32,   34,  124,  126,   33,   39,   42,   43,   45,   46,   48,
	   57,   65,   90,   94,  122,    9,   10,   32,   33,   59,  124,  126,
	   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,
	   10,   32,    0,  191,  194,  244,    9,   10,   32,   59,    9,   10,
	   32,   59,    9,   32,   59,   48,   57,    9,   10,   32,   33,  124,
	  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,
	  122,    9,   32,   33,  124,  126,   35,   39,   42,   43,   45,   46,
	   48,   57,   65,   90,   94,  122,   32,   33,   61,  124,  126,   35,
	   39,   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,   10,
	   32,   33,   61,  124,  126,   35,   39,   42,   43,   45,   46,   48,
	   57,   65,   90,   94,  122,   10,   32,   34,  124,  126,   33,   39,
	   42,   43,   45,   46,   48,   57,   65,   90,   94,  122,    9,   10,
	   32,   33,   59,  124,  126,   35,   39,   42,   43,   45,   46,   48,
	   57,   65,   90,   94,  122,    9,   32,   59,   48,   57,   10,   32,
	   48,   57,   10,   32,   48,   57,   10,   32,   48,   57,   10,   32,
	   10,   32,   10,   32,   48,   57,   10,   32,   48,   57,   10,   32,
	   48,   57,   10,   32,    0,  191,  194,  244,    9,   10,   32,   59,
	    9,   10,   32,   59,    9,   10,   32,   59,    9,   10,   32,   33,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,    9,   10,   32,   33,  124,  126,   35,   39,   42,   43,
	   45,   46,   48,   57,   65,   90,   94,  122,   10,   32,   33,   61,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,   10,   32,   34,  124,  126,   33,   39,   42,   43,   45,
	   46,   48,   57,   65,   90,   94,  122,    9,   10,   32,   33,   59,
	  124,  126,   35,   39,   42,   43,   45,   46,   48,   57,   65,   90,
	   94,  122,    9,   10,   32,   59,   48,   57,   10,   32,   48,   57,
	   10,   32,   48,   57,   10,   32,   48,   57,   10,   32,   48,   57,
	   10,   32,   48,   57,   10,   32,   10,   32,   48,   57,   10,   32,
	    0,  191,  194,  244,   10,   32,    0,  191,  194,  244,   32,   48,
	   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,
	   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,
	   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,
	   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,   32,   48,
	   57,   32,   48,   57,   32,   48,   57,   32,   48,   57,   46,   48,
	   57,   48,   57,   46,   48,   57,   48,   57,   32,   48,   57,   32,
	   48,   57,   32,   46,   48,   57,   46,   46,   48,   57,   46,   46,
	   48,   57,   46,   46,   48,   57,   65,   82,   67,   47,   48,   57,
	   46,   48,   57,   48,   57,   10,   13,   48,   57,    9,   10,   13,
	   32,   58,   10,   13,   58,    9,   10,   13,   32,    9,   10,   13,
	   32,    9,   10,   13,   32,    9,   10,   13,   32,   58,    9,   10,
	   13,   32,    9,   10,   13,   32,    9,   10,   13,   32,   10,   13,
	   10,   13,    0
	};
}

private static final char _warc_trans_keys[] = init__warc_trans_keys_0();


private static byte[] init__warc_single_lengths_0()
{
	return new byte [] {
	    0,    2,    1,    1,    0,    1,    2,    1,    2,    0,    0,    0,
	    0,    0,    0,    0,    0,    1,    5,    2,    0,    1,    1,    2,
	    2,    2,    2,    2,    0,    1,    2,    2,    0,    1,    6,    5,
	    7,    4,    3,    3,    5,    4,    3,    6,    3,    3,    0,    6,
	    5,    5,    5,    6,    5,    6,    5,    5,    3,    4,    4,    5,
	    5,    5,    5,    5,    3,    4,    5,    5,    3,    4,    4,    4,
	    3,    6,    5,    6,    5,    7,    2,    4,    4,    3,    6,    5,
	    5,    6,    5,    7,    3,    2,    2,    2,    2,    2,    2,    2,
	    2,    2,    4,    4,    4,    6,    6,    6,    5,    7,    4,    2,
	    2,    2,    2,    2,    2,    2,    0,    2,    2,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    0,    1,    0,    1,    0,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    0,    1,    0,    1,    1,    4,    1,    4,    4,    1,    6,    4,
	    4,    4,    5,    3,    4,    4,    4,    5,    4,    4,    4,    2,
	    4,    1,    4,    4,    1,    6,    4,    4,    4,    2,    1,    1,
	    0,    1,    2,    1,    2,    0,    0,    0,    0,    0,    0,    0,
	    0,    1,    5,    2,    0,    1,    1,    2,    2,    2,    2,    2,
	    0,    1,    2,    2,    0,    1,    6,    5,    7,    4,    3,    3,
	    5,    4,    3,    6,    3,    3,    0,    6,    5,    5,    5,    6,
	    5,    6,    5,    5,    3,    4,    4,    5,    5,    5,    5,    5,
	    3,    4,    5,    5,    3,    4,    4,    4,    3,    6,    5,    6,
	    5,    7,    2,    4,    4,    3,    6,    5,    5,    6,    5,    7,
	    3,    2,    2,    2,    2,    2,    2,    2,    2,    2,    4,    4,
	    4,    6,    6,    6,    5,    7,    4,    2,    2,    2,    2,    2,
	    2,    2,    0,    2,    2,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    0,    1,    0,    1,    0,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    0,    1,    0,    2,
	    5,    3,    4,    4,    4,    5,    4,    4,    4,    2,    2,    0,
	    0,    0,    0
	};
}

private static final byte _warc_single_lengths[] = init__warc_single_lengths_0();


private static byte[] init__warc_range_lengths_0()
{
	return new byte [] {
	    0,    1,    1,    1,    1,    1,    0,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    6,    0,    1,    1,    1,    1,
	    0,    0,    0,    0,    1,    1,    0,    0,    1,    1,    5,    6,
	    6,    0,    1,    0,    6,    6,    6,    6,    2,    0,    2,    6,
	    6,    6,    6,    6,    6,    6,    6,    2,    4,    4,    4,    4,
	    2,    2,    2,    2,    4,    4,    2,    2,    4,    4,    0,    0,
	    1,    6,    6,    6,    6,    6,    2,    0,    0,    1,    6,    6,
	    6,    6,    6,    6,    1,    1,    1,    1,    0,    0,    1,    1,
	    1,    2,    0,    0,    0,    6,    6,    6,    6,    6,    1,    1,
	    1,    1,    1,    1,    0,    0,    1,    2,    2,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    0,    1,    1,    1,    1,    1,    1,    1,
	    0,    1,    0,    1,    0,    1,    0,    1,    0,    0,    0,    0,
	    1,    1,    1,    1,    0,    6,    0,    6,    1,    0,    6,    1,
	    1,    1,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    6,    0,    6,    1,    0,    6,    1,    1,    1,    1,    1,    1,
	    1,    1,    0,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    6,    0,    1,    1,    1,    1,    0,    0,    0,    0,
	    1,    1,    0,    0,    1,    1,    5,    6,    6,    0,    1,    0,
	    6,    6,    6,    6,    2,    0,    2,    6,    6,    6,    6,    6,
	    6,    6,    6,    2,    4,    4,    4,    4,    2,    2,    2,    2,
	    4,    4,    2,    2,    4,    4,    0,    0,    1,    6,    6,    6,
	    6,    6,    2,    0,    0,    1,    6,    6,    6,    6,    6,    6,
	    1,    1,    1,    1,    0,    0,    1,    1,    1,    2,    0,    0,
	    0,    6,    6,    6,    6,    6,    1,    1,    1,    1,    1,    1,
	    0,    0,    1,    2,    2,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    0,    1,    1,    1,    1,    1,    1,    1,    0,    1,    0,    1,
	    0,    1,    0,    1,    0,    0,    0,    0,    1,    1,    1,    1,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    0
	};
}

private static final byte _warc_range_lengths[] = init__warc_range_lengths_0();


private static short[] init__warc_index_offsets_0()
{
	return new short [] {
	    0,    0,    4,    7,   10,   12,   15,   18,   21,   25,   27,   29,
	   31,   33,   35,   37,   39,   41,   44,   56,   59,   61,   64,   67,
	   71,   74,   77,   80,   83,   85,   88,   91,   94,   96,   99,  111,
	  123,  137,  142,  147,  151,  163,  174,  184,  197,  203,  207,  210,
	  223,  235,  247,  259,  272,  284,  297,  309,  317,  325,  334,  343,
	  353,  361,  369,  377,  385,  393,  402,  410,  418,  426,  435,  440,
	  445,  450,  463,  475,  488,  500,  514,  519,  524,  529,  534,  547,
	  559,  571,  584,  596,  610,  615,  619,  623,  627,  630,  633,  637,
	  641,  645,  650,  655,  660,  665,  678,  691,  704,  716,  730,  736,
	  740,  744,  748,  752,  756,  759,  762,  764,  769,  774,  777,  780,
	  783,  786,  789,  792,  795,  798,  801,  804,  807,  810,  813,  816,
	  819,  822,  825,  828,  831,  833,  835,  838,  840,  843,  845,  848,
	  851,  853,  856,  858,  861,  863,  866,  868,  871,  873,  875,  877,
	  879,  881,  884,  886,  889,  891,  902,  904,  915,  921,  923,  936,
	  942,  948,  954,  960,  964,  969,  974,  979,  985,  990,  995, 1000,
	 1003, 1014, 1016, 1027, 1033, 1035, 1048, 1054, 1060, 1066, 1070, 1073,
	 1076, 1078, 1081, 1084, 1087, 1091, 1093, 1095, 1097, 1099, 1101, 1103,
	 1105, 1107, 1110, 1122, 1125, 1127, 1130, 1133, 1137, 1140, 1143, 1146,
	 1149, 1151, 1154, 1157, 1160, 1162, 1165, 1177, 1189, 1203, 1208, 1213,
	 1217, 1229, 1240, 1250, 1263, 1269, 1273, 1276, 1289, 1301, 1313, 1325,
	 1338, 1350, 1363, 1375, 1383, 1391, 1400, 1409, 1419, 1427, 1435, 1443,
	 1451, 1459, 1468, 1476, 1484, 1492, 1501, 1506, 1511, 1516, 1529, 1541,
	 1554, 1566, 1580, 1585, 1590, 1595, 1600, 1613, 1625, 1637, 1650, 1662,
	 1676, 1681, 1685, 1689, 1693, 1696, 1699, 1703, 1707, 1711, 1716, 1721,
	 1726, 1731, 1744, 1757, 1770, 1782, 1796, 1802, 1806, 1810, 1814, 1818,
	 1822, 1825, 1828, 1830, 1835, 1840, 1843, 1846, 1849, 1852, 1855, 1858,
	 1861, 1864, 1867, 1870, 1873, 1876, 1879, 1882, 1885, 1888, 1891, 1894,
	 1897, 1899, 1901, 1904, 1906, 1909, 1911, 1914, 1917, 1919, 1922, 1924,
	 1927, 1929, 1932, 1934, 1937, 1939, 1941, 1943, 1945, 1947, 1950, 1952,
	 1956, 1962, 1966, 1971, 1976, 1981, 1987, 1992, 1997, 2002, 2005, 2008,
	 2009, 2010, 2011
	};
}

private static final short _warc_index_offsets[] = init__warc_index_offsets_0();


private static short[] init__warc_indicies_0()
{
	return new short [] {
	    0,    2,    3,    1,    4,    3,    1,    5,    3,    1,    3,    1,
	    6,    3,    1,    1,    7,    6,    8,    9,    1,   10,   11,   12,
	    1,   13,    1,   14,    1,   15,    1,   16,    1,   17,    1,   18,
	    1,   19,    1,   20,    1,   21,   22,    1,    1,   24,   25,   25,
	   25,   25,   25,   25,   25,   25,   25,   23,    1,   24,   23,   26,
	    1,   27,   28,    1,   27,   29,    1,   27,   30,   31,    1,    1,
	    1,   32,    1,   33,   32,    1,    1,   34,    1,   35,   34,   36,
	    1,   37,   36,    1,    1,    1,   38,    1,   39,   38,   31,    1,
	   27,   31,    1,    1,   24,   25,   40,   25,   25,   25,   25,   25,
	   25,   25,   23,    1,   24,   41,   41,   41,   41,   41,   41,   41,
	   41,   41,   23,   42,    1,   43,   41,   44,   41,   41,   41,   41,
	   41,   41,   41,   41,   23,   42,    1,   43,   44,   23,   45,   45,
	   46,   26,    1,   45,   45,   46,    1,   46,   46,   47,   47,   47,
	   47,   47,   47,   47,   47,   47,    1,   47,   48,   47,   47,   47,
	   47,   47,   47,   47,   47,    1,   50,   49,   49,   49,   49,   49,
	   49,   49,   49,    1,   45,   51,   49,   46,   49,   49,   49,   49,
	   49,   49,   49,   49,    1,   50,   52,   53,   50,   50,    1,   45,
	   51,   46,    1,   50,   50,    1,   44,    1,   54,   55,   55,   55,
	   55,   55,   55,   55,   55,   55,   23,   46,   46,   47,   47,   47,
	   47,   47,   47,   56,   47,   47,    1,   27,   47,   48,   47,   47,
	   47,   47,   47,   57,   47,   47,    1,   27,   47,   48,   47,   47,
	   47,   47,   47,   58,   47,   47,    1,   27,   30,   47,   48,   47,
	   47,   47,   47,   47,   59,   47,   47,    1,   27,   47,   48,   47,
	   47,   47,   47,   47,   59,   47,   47,    1,    1,   24,   55,   60,
	   55,   55,   55,   55,   55,   55,   55,   55,   23,    1,   24,   61,
	   41,   41,   41,   41,   41,   41,   41,   41,   23,   61,    1,   62,
	   42,   63,   61,   61,   23,   50,   52,   53,   50,   64,   50,   50,
	    1,   50,   27,   52,   53,   50,   65,   50,   50,    1,   50,   27,
	   52,   53,   50,   66,   50,   50,    1,   50,   27,   67,   52,   53,
	   50,   68,   50,   50,    1,   69,    1,   50,   70,   71,   69,   69,
	   32,   69,    1,   72,   70,   71,   69,   69,   32,   73,    1,   50,
	   74,   75,   73,   73,   34,   73,    1,   76,   74,   75,   73,   73,
	   34,   50,   52,   53,   50,   77,   50,   50,    1,   50,   78,   52,
	   53,   50,   77,   50,   50,    1,   79,    1,   50,   80,   81,   79,
	   79,   38,   79,    1,   82,   80,   81,   79,   79,   38,   50,   52,
	   53,   50,   68,   50,   50,    1,   50,   27,   52,   53,   50,   68,
	   50,   50,    1,   83,    1,   51,   84,   38,   83,    1,   85,   84,
	   38,   45,   45,   46,   31,    1,   84,    1,   86,   87,   87,   87,
	   87,   87,   87,   87,   87,   87,   38,   46,   46,   47,   47,   47,
	   47,   47,   47,   59,   47,   47,    1,    1,   39,   87,   88,   87,
	   87,   87,   87,   87,   87,   87,   87,   38,    1,   39,   79,   89,
	   89,   89,   89,   89,   89,   89,   89,   38,   83,    1,   51,   89,
	   84,   89,   89,   89,   89,   89,   89,   89,   89,   38,   50,   82,
	   79,   79,   38,   90,    1,   91,   92,   34,   90,    1,   93,   92,
	   34,   45,   45,   46,   36,    1,   92,    1,   94,   95,   95,   95,
	   95,   95,   95,   95,   95,   95,   34,   46,   46,   47,   47,   47,
	   47,   47,   47,   96,   47,   47,    1,   37,   47,   48,   47,   47,
	   47,   47,   47,   96,   47,   47,    1,    1,   35,   95,   97,   95,
	   95,   95,   95,   95,   95,   95,   95,   34,    1,   35,   73,   98,
	   98,   98,   98,   98,   98,   98,   98,   34,   90,    1,   91,   98,
	   92,   98,   98,   98,   98,   98,   98,   98,   98,   34,   45,   45,
	   46,   99,    1,   27,   37,  100,    1,   27,   37,  101,    1,   27,
	  102,  103,    1,    1,    1,  104,    1,  105,  104,    1,    1,  106,
	   34,   27,   35,  106,   34,   27,   37,  103,    1,   50,   76,   73,
	   73,   34,  107,    1,  108,  109,   32,  107,    1,  110,  109,   32,
	   90,    1,   45,   92,   34,  109,    1,  111,  112,  112,  112,  112,
	  112,  112,  112,  112,  112,   32,   92,    1,   46,   95,   95,   95,
	   95,   95,   95,   95,   95,   95,   34,    1,   33,  112,  113,  112,
	  112,  112,  112,  112,  112,  112,  112,   32,    1,   33,   69,  114,
	  114,  114,  114,  114,  114,  114,  114,   32,  107,    1,  108,  114,
	  109,  114,  114,  114,  114,  114,  114,  114,  114,   32,   90,    1,
	   45,   92,  115,   34,   27,   35,  116,   34,   27,   35,  117,   34,
	   27,  118,  106,   34,    1,    1,  119,   32,    1,  120,  119,   32,
	    1,    1,  121,    1,  122,  121,  103,    1,   50,   72,   69,   69,
	   32,   50,   62,   61,   61,   23,   21,  123,    1,   21,  124,    1,
	   21,  125,    1,   21,  126,    1,   21,  127,    1,   21,  128,    1,
	   21,  129,    1,   21,  130,    1,   21,  131,    1,   21,  132,    1,
	   21,  133,    1,   21,  134,    1,   21,  135,    1,   21,  136,    1,
	   21,  137,    1,   21,  138,    1,   21,  139,    1,   21,  140,    1,
	   21,  141,    1,   21,    1,  142,    1,  143,  144,    1,  145,    1,
	  146,  147,    1,  148,    1,   10,  149,    1,   10,  150,    1,   10,
	    1,  146,  151,    1,  146,    1,  143,  152,    1,  143,    1,   11,
	  153,    1,   11,    1,   11,   12,    1,  154,    1,  155,    1,  156,
	    1,  157,    1,  158,    1,  159,  158,    1,  160,    1,  161,  160,
	    1,  162,    1,  163,  164,  164,  164,  164,  164,  164,  164,  164,
	  164,    1,  165,    1,  164,  166,  164,  164,  164,  164,  164,  164,
	  164,  164,    1,  167,  168,  167,    1,    1,  169,  170,    1,  171,
	  172,  171,  173,  173,  173,  173,  173,  173,  173,  173,  173,    1,
	  171,  174,  171,    1,    1,  175,  176,  177,  176,    1,    1,  169,
	  178,  168,  178,    1,    1,  169,    1,  180,  181,    1,    1,  179,
	    1,    1,  182,  179,  184,  185,  186,  184,  183,  187,  188,  189,
	  187,  183,  190,  185,  186,  190,  183,  192,  193,  194,  192,    1,
	  191,  192,  196,  197,  192,  195,  198,  185,  186,  198,  183,  198,
	    1,  198,  198,  183,  180,  181,    1,  199,  200,  200,  200,  200,
	  200,  200,  200,  200,  200,    1,  201,    1,  200,  202,  200,  200,
	  200,  200,  200,  200,  200,  200,    1,  203,  204,  203,    1,    1,
	  205,  206,    1,  207,  208,  207,  209,  209,  209,  209,  209,  209,
	  209,  209,  209,    1,  207,  210,  207,    1,    1,  211,  212,  213,
	  212,    1,    1,  205,  214,  204,  214,    1,    1,  205,  215,  216,
	  217,    1,  218,  217,    1,  219,  217,    1,  217,    1,  220,  217,
	    1,    1,  221,  220,  222,  223,    1,  224,  225,  226,    1,  227,
	    1,  228,    1,  229,    1,  230,    1,  231,    1,  232,    1,  233,
	    1,  234,    1,  235,  236,    1,    1,  238,  239,  239,  239,  239,
	  239,  239,  239,  239,  239,  237,    1,  238,  237,  240,    1,  241,
	  242,    1,  241,  243,    1,  241,  244,  245,    1,    1,    1,  246,
	    1,  247,  246,    1,    1,  248,    1,  249,  248,  250,    1,  251,
	  250,    1,    1,    1,  252,    1,  253,  252,  245,    1,  241,  245,
	    1,    1,  238,  239,  254,  239,  239,  239,  239,  239,  239,  239,
	  237,    1,  238,  255,  255,  255,  255,  255,  255,  255,  255,  255,
	  237,  256,    1,  257,  255,  258,  255,  255,  255,  255,  255,  255,
	  255,  255,  237,  256,    1,  257,  258,  237,  259,  259,  260,  240,
	    1,  259,  259,  260,    1,  260,  260,  261,  261,  261,  261,  261,
	  261,  261,  261,  261,    1,  261,  262,  261,  261,  261,  261,  261,
	  261,  261,  261,    1,  264,  263,  263,  263,  263,  263,  263,  263,
	  263,    1,  259,  265,  263,  260,  263,  263,  263,  263,  263,  263,
	  263,  263,    1,  264,  266,  267,  264,  264,    1,  259,  265,  260,
	    1,  264,  264,    1,  258,    1,  268,  269,  269,  269,  269,  269,
	  269,  269,  269,  269,  237,  260,  260,  261,  261,  261,  261,  261,
	  261,  270,  261,  261,    1,  241,  261,  262,  261,  261,  261,  261,
	  261,  271,  261,  261,    1,  241,  261,  262,  261,  261,  261,  261,
	  261,  272,  261,  261,    1,  241,  244,  261,  262,  261,  261,  261,
	  261,  261,  273,  261,  261,    1,  241,  261,  262,  261,  261,  261,
	  261,  261,  273,  261,  261,    1,    1,  238,  269,  274,  269,  269,
	  269,  269,  269,  269,  269,  269,  237,    1,  238,  275,  255,  255,
	  255,  255,  255,  255,  255,  255,  237,  275,    1,  276,  256,  277,
	  275,  275,  237,  264,  266,  267,  264,  278,  264,  264,    1,  264,
	  241,  266,  267,  264,  279,  264,  264,    1,  264,  241,  266,  267,
	  264,  280,  264,  264,    1,  264,  241,  281,  266,  267,  264,  282,
	  264,  264,    1,  283,    1,  264,  284,  285,  283,  283,  246,  283,
	    1,  286,  284,  285,  283,  283,  246,  287,    1,  264,  288,  289,
	  287,  287,  248,  287,    1,  290,  288,  289,  287,  287,  248,  264,
	  266,  267,  264,  291,  264,  264,    1,  264,  292,  266,  267,  264,
	  291,  264,  264,    1,  293,    1,  264,  294,  295,  293,  293,  252,
	  293,    1,  296,  294,  295,  293,  293,  252,  264,  266,  267,  264,
	  282,  264,  264,    1,  264,  241,  266,  267,  264,  282,  264,  264,
	    1,  297,    1,  265,  298,  252,  297,    1,  299,  298,  252,  259,
	  259,  260,  245,    1,  298,    1,  300,  301,  301,  301,  301,  301,
	  301,  301,  301,  301,  252,  260,  260,  261,  261,  261,  261,  261,
	  261,  273,  261,  261,    1,    1,  253,  301,  302,  301,  301,  301,
	  301,  301,  301,  301,  301,  252,    1,  253,  293,  303,  303,  303,
	  303,  303,  303,  303,  303,  252,  297,    1,  265,  303,  298,  303,
	  303,  303,  303,  303,  303,  303,  303,  252,  264,  296,  293,  293,
	  252,  304,    1,  305,  306,  248,  304,    1,  307,  306,  248,  259,
	  259,  260,  250,    1,  306,    1,  308,  309,  309,  309,  309,  309,
	  309,  309,  309,  309,  248,  260,  260,  261,  261,  261,  261,  261,
	  261,  310,  261,  261,    1,  251,  261,  262,  261,  261,  261,  261,
	  261,  310,  261,  261,    1,    1,  249,  309,  311,  309,  309,  309,
	  309,  309,  309,  309,  309,  248,    1,  249,  287,  312,  312,  312,
	  312,  312,  312,  312,  312,  248,  304,    1,  305,  312,  306,  312,
	  312,  312,  312,  312,  312,  312,  312,  248,  259,  259,  260,  313,
	    1,  241,  251,  314,    1,  241,  251,  315,    1,  241,  316,  317,
	    1,    1,    1,  318,    1,  319,  318,    1,    1,  320,  248,  241,
	  249,  320,  248,  241,  251,  317,    1,  264,  290,  287,  287,  248,
	  321,    1,  322,  323,  246,  321,    1,  324,  323,  246,  304,    1,
	  259,  306,  248,  323,    1,  325,  326,  326,  326,  326,  326,  326,
	  326,  326,  326,  246,  306,    1,  260,  309,  309,  309,  309,  309,
	  309,  309,  309,  309,  248,    1,  247,  326,  327,  326,  326,  326,
	  326,  326,  326,  326,  326,  246,    1,  247,  283,  328,  328,  328,
	  328,  328,  328,  328,  328,  246,  321,    1,  322,  328,  323,  328,
	  328,  328,  328,  328,  328,  328,  328,  246,  304,    1,  259,  306,
	  329,  248,  241,  249,  330,  248,  241,  249,  331,  248,  241,  332,
	  320,  248,    1,    1,  333,  246,    1,  334,  333,  246,    1,    1,
	  335,    1,  336,  335,  317,    1,  264,  286,  283,  283,  246,  264,
	  276,  275,  275,  237,  235,  337,    1,  235,  338,    1,  235,  339,
	    1,  235,  340,    1,  235,  341,    1,  235,  342,    1,  235,  343,
	    1,  235,  344,    1,  235,  345,    1,  235,  346,    1,  235,  347,
	    1,  235,  348,    1,  235,  349,    1,  235,  350,    1,  235,  351,
	    1,  235,  352,    1,  235,  353,    1,  235,  354,    1,  235,  355,
	    1,  235,    1,  356,    1,  357,  358,    1,  359,    1,  360,  361,
	    1,  362,    1,  224,  363,    1,  224,  364,    1,  224,    1,  360,
	  365,    1,  360,    1,  357,  366,    1,  357,    1,  225,  367,    1,
	  225,    1,  225,  226,    1,  368,    1,  369,    1,  370,    1,  371,
	    1,  372,    1,  373,  372,    1,  374,    1,  375,  376,  374,    1,
	    1,  378,  379,    1,    1,  377,    1,    1,  380,  377,  382,  383,
	  384,  382,  381,  385,  386,  387,  385,  381,  388,  383,  384,  388,
	  381,  390,  391,  392,  390,    1,  389,  390,  394,  395,  390,  393,
	  396,  383,  384,  396,  381,  396,    1,  396,  396,  381,  378,  379,
	    1,  375,  376,    1,    1,    1,    1,    1,    0
	};
}

private static final short _warc_indicies[] = init__warc_indicies_0();


private static short[] init__warc_trans_targs_0()
{
	return new short [] {
	    2,    0,  152,    5,    3,    4,    6,    7,    8,  151,    9,  137,
	  149,   10,   11,   12,   13,   14,   15,   16,   17,   18,  117,   19,
	   20,   34,   21,  359,   22,   23,   24,   33,   25,   26,   27,   28,
	   29,   30,   31,   32,   35,   36,   37,   38,   47,   39,   40,   41,
	   42,   43,   44,   38,   45,   46,   48,   53,   49,   50,   51,   52,
	   54,   55,   56,  116,   57,   58,   59,   60,   69,   61,   98,  115,
	   62,   63,   79,   97,   64,   65,   66,   67,   70,   78,   68,   71,
	   73,   72,   74,   75,   76,   77,   80,   88,   82,   81,   83,   85,
	   84,   86,   87,   89,   90,   91,   92,   96,   93,   94,   95,   99,
	  106,  101,  100,  102,  103,  104,  105,  107,  108,  109,  110,  111,
	  112,  113,  114,  118,  119,  120,  121,  122,  123,  124,  125,  126,
	  127,  128,  129,  130,  131,  132,  133,  134,  135,  136,  138,  139,
	  147,  140,  141,  145,  142,  143,  144,  146,  148,  150,  153,  154,
	  155,  156,  157,  158,  159,  160,  161,  162,  163,  359,  164,  164,
	  165,  168,  166,  167,  162,  163,  165,  168,  169,  165,  169,  171,
	  360,  179,  172,  173,  172,  175,  177,  174,  175,  177,  174,  171,
	  176,  360,  179,  173,  175,  177,  178,  181,  182,  361,  183,  183,
	  184,  187,  185,  186,  181,  182,  184,  187,  188,  184,  188,  190,
	  340,  193,  191,  192,  194,  195,  196,  339,  197,  325,  337,  198,
	  199,  200,  201,  202,  203,  204,  205,  206,  305,  207,  208,  222,
	  209,  362,  210,  211,  212,  221,  213,  214,  215,  216,  217,  218,
	  219,  220,  223,  224,  225,  226,  235,  227,  228,  229,  230,  231,
	  232,  226,  233,  234,  236,  241,  237,  238,  239,  240,  242,  243,
	  244,  304,  245,  246,  247,  248,  257,  249,  286,  303,  250,  251,
	  267,  285,  252,  253,  254,  255,  258,  266,  256,  259,  261,  260,
	  262,  263,  264,  265,  268,  276,  270,  269,  271,  273,  272,  274,
	  275,  277,  278,  279,  280,  284,  281,  282,  283,  287,  294,  289,
	  288,  290,  291,  292,  293,  295,  296,  297,  298,  299,  300,  301,
	  302,  306,  307,  308,  309,  310,  311,  312,  313,  314,  315,  316,
	  317,  318,  319,  320,  321,  322,  323,  324,  326,  327,  335,  328,
	  329,  333,  330,  331,  332,  334,  336,  338,  341,  342,  343,  344,
	  345,  346,  347,  348,  358,  349,  362,  357,  350,  351,  350,  353,
	  355,  352,  353,  355,  352,  349,  354,  362,  357,  351,  353,  355,
	  356
	};
}

private static final short _warc_trans_targs[] = init__warc_trans_targs_0();


private static byte[] init__warc_trans_actions_0()
{
	return new byte [] {
	    0,    0,    0,    1,    0,    0,    1,   15,    1,    1,   17,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,   19,    1,    1,
	   21,    1,    1,   45,    1,    1,   23,    1,    0,    0,    0,    0,
	    0,    0,    0,    0,    1,    1,    1,   21,    1,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,   21,    1,    1,    1,    1,    1,
	    1,    1,   21,    1,    1,    1,    1,   23,    1,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    0,    1,    1,    1,   23,    1,    0,    0,    1,    0,
	    0,    0,    0,    0,    0,    0,    0,    1,    1,    1,   23,    0,
	    0,    0,    0,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    0,    0,
	    0,    0,    3,    0,    5,    0,    0,    0,    1,   27,   11,    0,
	    0,    1,    0,    0,   13,   35,    9,   32,   29,    7,    1,    1,
	    0,    0,   11,    1,    0,    0,    1,   29,    7,   29,    1,   35,
	    0,   13,   13,   32,    9,   32,    1,    0,    1,    0,   11,    0,
	    0,    1,    0,    0,   13,   35,    9,   32,   29,    7,    1,    0,
	    0,    1,    0,    0,    1,   15,    1,    1,   17,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,   19,    1,    1,   21,    1,
	    1,   41,    1,    1,   23,    1,    0,    0,    0,    0,    0,    0,
	    0,    0,    1,    1,    1,   21,    1,    0,    0,    0,    0,    0,
	    0,    0,    0,    0,   21,    1,    1,    1,    1,    1,    1,    1,
	   21,    1,    1,    1,    1,   23,    1,    0,    0,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,
	    0,    1,    1,    1,   23,    1,    0,    0,    1,    0,    0,    0,
	    0,    0,    0,    0,    0,    1,    1,    1,   23,    0,    0,    0,
	    0,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,    1,
	    1,    1,    1,    1,    1,    1,    1,    1,    0,    0,    0,    0,
	    3,    0,    5,    0,    0,    1,   25,    0,   11,    1,    0,    0,
	    1,   29,    7,   29,    1,   35,    0,   38,   13,   32,    9,   32,
	    1
	};
}

private static final byte _warc_trans_actions[] = init__warc_trans_actions_0();


static final int warc_start = 1;
static final int warc_first_final = 359;
static final int warc_error = 0;

static final int warc_en_warc_fields_lenient = 170;
static final int warc_en_warc_fields = 180;
static final int warc_en_any_header_lenient = 189;
static final int warc_en_any_header = 1;


// line 335 "WarcParser.rl"
}