/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.nosceon.datolite.db;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.Statement;
import java.util.LinkedList;
import java.util.Queue;

/**
 * @author Johan Siebens
 */
class AutoCleanConnectionHandler implements InvocationHandler {

    private final Connection target;

    private final Queue<Statement> statements;

    public AutoCleanConnectionHandler(Connection target) {
        this.target = target;
        this.statements = new LinkedList<>();
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        try {
            switch (method.getName()) {
                case "close":
                    return handleClose(method, args);
                case "createStatement":
                    return statementMethod(method, args);
                case "prepareStatement":
                    return statementMethod(method, args);
                case "prepareCall":
                    return statementMethod(method, args);
                default:
                    return method.invoke(target, args);
            }
        }
        catch (InvocationTargetException e) {
            throw e.getTargetException();
        }
    }

    private Object handleClose(Method method, Object[] args) throws Throwable {
        while (!statements.isEmpty()) {
            statements.poll().close();
        }
        return method.invoke(target, args);
    }

    private Object statementMethod(Method method, Object[] args) throws Throwable {
        Statement statement = (Statement) method.invoke(target, args);
        statements.add(statement);
        return statement;
    }

}
