/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.nosceon.datolite.db;

import com.jolbox.bonecp.BoneCPDataSource;
import com.typesafe.config.Config;
import org.nosceon.datolite.Context;
import org.nosceon.datolite.db.exception.UnableToGetConnectionException;
import org.nosceon.datolite.db.exception.UncategorizedDatabaseException;
import org.nosceon.datolite.util.Memoizer;

import javax.sql.DataSource;
import java.lang.reflect.Proxy;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.concurrent.TimeUnit;

/**
 * @author Johan Siebens
 */
final class DbPluginImpl implements DbPlugin {

    public static final String DEFAULT_CONFIG_PATH = "db";

    private final Memoizer<String, DataSource> cache = new Memoizer<>(this::createDataSource);

    private final Context context;

    DbPluginImpl(Context context) {
        this.context = context;
    }

    @Override
    public DataSource dataSource() {
        return dataSource(DEFAULT_CONFIG_PATH);
    }

    @Override
    public DataSource dataSource(String configPath) {
        return cache.get(configPath);
    }

    @Override
    public Connection connection() {
        return connection(true);
    }

    @Override
    public Connection connection(boolean autoCommit) {
        return connection(DEFAULT_CONFIG_PATH, autoCommit);
    }

    @Override
    public Connection connection(String configPath) {
        return connection(configPath, true);
    }

    @Override
    public Connection connection(String configPath, boolean autoCommit) {
        try {
            Connection connection = dataSource(configPath).getConnection();
            connection.setAutoCommit(autoCommit);
            return connection;
        }
        catch (SQLException e) {
            throw new UnableToGetConnectionException(e);
        }
    }

    @Override
    public <A> A withConnection(ConnectionCallback<A> block) {
        return withConnection(DEFAULT_CONFIG_PATH, block);
    }

    @Override
    public <A> A withConnection(String configPath, ConnectionCallback<A> block) {
        try (Connection con = autoClean(connection(configPath))) {
            return block.withConnection(con);
        }
        catch (SQLException e) {
            throw new UncategorizedDatabaseException(e);
        }
    }

    @Override
    public <A> A withTransaction(ConnectionCallback<A> block) {
        return withTransaction(DEFAULT_CONFIG_PATH, block);
    }

    @Override
    public <A> A withTransaction(String configPath, final ConnectionCallback<A> block) {
        return withConnection(configPath, new ConnectionCallback<A>() {

            @Override
            public A withConnection(Connection connection) throws SQLException {
                boolean commit = false;
                try {
                    connection.setAutoCommit(false);
                    A a = block.withConnection(connection);
                    commit = true;
                    return a;
                }
                finally {
                    if (commit) {
                        connection.commit();
                    }
                    else {
                        connection.rollback();
                    }
                }
            }

        });
    }

    private Connection autoClean(Connection connection) {
        return (Connection) Proxy.newProxyInstance(getClass().getClassLoader(), new Class<?>[]{Connection.class}, new AutoCleanConnectionHandler(connection));
    }

    private DataSource createDataSource(String path) {
        Config conf = lookupConfig(path);

        BoneCPDataSource dataSource = new BoneCPDataSource();

        dataSource.setDriverClass(conf.getString("driver"));
        dataSource.setJdbcUrl(conf.getString("url"));

        dataSource.setUsername(conf.getString("user"));
        dataSource.setPassword(conf.getString("password"));

        dataSource.setPartitionCount(conf.getInt("partition-count"));
        dataSource.setMaxConnectionsPerPartition(conf.getInt("max-connections-per-partition"));
        dataSource.setMinConnectionsPerPartition(conf.getInt("min-connections-per-partition"));
        dataSource.setAcquireIncrement(conf.getInt("acquire-increment"));
        dataSource.setAcquireRetryAttempts(conf.getInt("acquire-retry-attempts"));
        dataSource.setAcquireRetryDelayInMs(conf.getLong("acquire-retry-delay"));
        dataSource.setConnectionTimeoutInMs(conf.getLong("connection-timeout"));
        dataSource.setIdleMaxAge(conf.getDuration("idle-max-age", TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
        dataSource.setMaxConnectionAge(conf.getDuration("max-connection-age", TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
        dataSource.setDisableJMX(conf.getBoolean("disable-jmx"));
        dataSource.setStatisticsEnabled(conf.getBoolean("statistics-enabled"));
        dataSource.setIdleConnectionTestPeriod(conf.getDuration("idle-connection-test-period", TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
        dataSource.setDisableConnectionTracking(conf.getBoolean("disable-connection-tracking"));

        context.onShutdown(dataSource::close);

        return dataSource;
    }

    private Config lookupConfig(String path) {
        Config config = context.config();
        Config defaults = config.getConfig("datolite.defaults.bonecp");
        return config.hasPath(path) ? config.getConfig(path).withFallback(defaults) : defaults;
    }

}
