/**
 * Copyright (C) 2011 Brian Ferris <bdferris@onebusaway.org> Copyright (C) 2012 Google, Inc.
 *
 * <p>Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a copy of the License at
 *
 * <p>http://www.apache.org/licenses/LICENSE-2.0
 *
 * <p>Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.onebusaway.gtfs.serialization;

import java.io.*;
import java.util.*;
import org.onebusaway.csv_entities.CsvEntityContext;
import org.onebusaway.csv_entities.CsvEntityReader;
import org.onebusaway.csv_entities.CsvInputSource;
import org.onebusaway.csv_entities.CsvTokenizerStrategy;
import org.onebusaway.csv_entities.EntityHandler;
import org.onebusaway.csv_entities.exceptions.CsvEntityIOException;
import org.onebusaway.csv_entities.schema.DefaultEntitySchemaFactory;
import org.onebusaway.gtfs.impl.GtfsDaoImpl;
import org.onebusaway.gtfs.model.*;
import org.onebusaway.gtfs.services.GenericMutableDao;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GtfsReader extends CsvEntityReader {

  private final Logger _log = LoggerFactory.getLogger(GtfsReader.class);

  public static final String KEY_CONTEXT = GtfsReader.class.getName() + ".context";

  private List<Class<?>> _entityClasses = new ArrayList<>();

  private GtfsReaderContextImpl _context = new GtfsReaderContextImpl();

  private GenericMutableDao _entityStore = new GtfsDaoImpl();

  private List<Agency> _agencies = new ArrayList<>();

  private Map<Class<?>, Map<String, String>> _agencyIdsByEntityClassAndId = new HashMap<>();

  private String _defaultAgencyId;

  private final Map<String, String> _agencyIdMapping = new HashMap<>();

  private boolean _overwriteDuplicates = false;

  public GtfsReader() {

    _entityClasses.add(Agency.class);
    _entityClasses.add(Block.class);
    _entityClasses.add(ShapePoint.class);
    _entityClasses.add(Icon.class);
    _entityClasses.add(Area.class);
    _entityClasses.add(BookingRule.class);
    _entityClasses.add(Route.class);
    _entityClasses.add(RouteNetworkAssignment.class);
    _entityClasses.add(Level.class);
    _entityClasses.add(Stop.class);
    _entityClasses.add(Location.class);
    _entityClasses.add(LocationGroup.class);
    _entityClasses.add(LocationGroupElement.class);
    _entityClasses.add(Trip.class);
    _entityClasses.add(StopAreaElement.class);
    _entityClasses.add(StopTime.class);
    _entityClasses.add(ServiceCalendar.class);
    _entityClasses.add(ServiceCalendarDate.class);
    _entityClasses.add(RiderCategory.class);
    _entityClasses.add(FareMedium.class);
    _entityClasses.add(FareProduct.class);
    _entityClasses.add(FareLegRule.class);
    _entityClasses.add(FareAttribute.class);
    _entityClasses.add(FareRule.class);
    _entityClasses.add(FareTransferRule.class);
    _entityClasses.add(Frequency.class);
    _entityClasses.add(Pathway.class);
    _entityClasses.add(Transfer.class);
    _entityClasses.add(FeedInfo.class);
    _entityClasses.add(Ridership.class);
    _entityClasses.add(Translation.class);
    _entityClasses.add(Vehicle.class);
    _entityClasses.add(Facility.class);
    _entityClasses.add(FacilityPropertyDefinition.class);
    _entityClasses.add(FacilityProperty.class);
    _entityClasses.add(DirectionEntry.class);
    _entityClasses.add(Network.class);
    _entityClasses.add(Timeframe.class);

    CsvTokenizerStrategy tokenizerStrategy = new CsvTokenizerStrategy();
    setTokenizerStrategy(tokenizerStrategy);

    setTrimValues(true);

    /** Prep the Entity Schema Factories */
    DefaultEntitySchemaFactory schemaFactory = createEntitySchemaFactory();
    setEntitySchemaFactory(schemaFactory);

    CsvEntityContext ctx = getContext();
    ctx.put(KEY_CONTEXT, _context);

    addEntityHandler(new EntityHandlerImpl());
  }

  public void setInputLocation(File path) throws IOException {
    super.setInputLocation(path);
  }

  public void setLastModifiedTime(Long lastModifiedTime) {
    if (lastModifiedTime != null) getContext().put("lastModifiedTime", lastModifiedTime);
  }

  public Long getLastModfiedTime() {
    return (Long) getContext().get("lastModifiedTime");
  }

  public List<Agency> getAgencies() {
    return _agencies;
  }

  public void setAgencies(List<Agency> agencies) {
    _agencies = new ArrayList<>(agencies);
  }

  public void setDefaultAgencyId(String feedId) {
    _defaultAgencyId = feedId;
  }

  public String getDefaultAgencyId() {
    if (_defaultAgencyId != null) return _defaultAgencyId;
    if (_agencies.size() > 0) return _agencies.getFirst().getId();
    throw new NoDefaultAgencyIdException();
  }

  public void addAgencyIdMapping(String fromAgencyId, String toAgencyId) {
    _agencyIdMapping.put(fromAgencyId, toAgencyId);
  }

  public GtfsReaderContext getGtfsReaderContext() {
    return _context;
  }

  public GenericMutableDao getEntityStore() {
    return _entityStore;
  }

  public void setEntityStore(GenericMutableDao entityStore) {
    _entityStore = entityStore;
  }

  public List<Class<?>> getEntityClasses() {
    return _entityClasses;
  }

  public void setEntityClasses(List<Class<?>> entityClasses) {
    _entityClasses = entityClasses;
  }

  public void setOverwriteDuplicates(boolean overwriteDuplicates) {
    _overwriteDuplicates = overwriteDuplicates;
  }

  public void readEntities(Class<?> entityClass, Reader reader)
      throws IOException, CsvEntityIOException {
    if (entityClass == Location.class) {
      for (Location location : new LocationsGeoJSONReader(reader, getDefaultAgencyId()).read()) {
        injectEntity(location);
      }
    } else {
      super.readEntities(entityClass, reader);
    }
  }

  public void run() throws IOException {
    run(getInputSource());
  }

  public void run(CsvInputSource source) throws IOException {

    List<Class<?>> classes = getEntityClasses();

    _entityStore.open();

    for (Class<?> entityClass : classes) {
      _log.info("reading entities: " + entityClass.getName());

      readEntities(entityClass, source);
      _entityStore.flush();
    }

    _entityStore.close();
  }

  /****
   * Protected Methods
   ****/

  protected DefaultEntitySchemaFactory createEntitySchemaFactory() {
    return GtfsEntitySchemaFactory.createEntitySchemaFactory();
  }

  protected Object getEntity(Class<?> entityClass, Serializable id) {
    if (entityClass == null) throw new IllegalArgumentException("entity class must not be null");
    if (id == null) throw new IllegalArgumentException("entity id must not be null");
    return _entityStore.getEntityForId(entityClass, id);
  }

  protected String getTranslatedAgencyId(String agencyId) {
    String id = _agencyIdMapping.get(agencyId);
    if (id != null) return id;
    return agencyId;
  }

  protected String getAgencyForEntity(Class<?> entityType, String entityId) {

    Map<String, String> agencyIdsByEntityId = _agencyIdsByEntityClassAndId.get(entityType);

    if (agencyIdsByEntityId != null) {
      String id = agencyIdsByEntityId.get(entityId);
      if (id != null) return id;
    }

    throw new EntityReferenceNotFoundException(entityType, entityId);
  }

  /****
   * Private Internal Classes
   ****/

  private class EntityHandlerImpl implements EntityHandler {

    public void handleEntity(Object entity) {

      if (entity instanceof final Agency agency) {
        if (agency.getId() == null) {
          if (_defaultAgencyId == null) agency.setId(agency.getName());
          else agency.setId(_defaultAgencyId);
        }

        // If we already have this agency from a previous load, then we don't
        // add it or save it to the entity store
        if (_agencies.contains(agency)) return;

        _agencies.add((Agency) entity);
      } else if (entity instanceof final BookingRule bookingRule) {
        registerAgencyId(BookingRule.class, bookingRule.getId());
      } else if (entity instanceof final Pathway pathway) {
        registerAgencyId(Pathway.class, pathway.getId());
      } else if (entity instanceof final Level level) {
        registerAgencyId(Level.class, level.getId());
      } else if (entity instanceof final Route route) {
        registerAgencyId(Route.class, route.getId());
      } else if (entity instanceof final Trip trip) {
        registerAgencyId(Trip.class, trip.getId());
      } else if (entity instanceof final Stop stop) {
        registerAgencyId(Stop.class, stop.getId());
      } else if (entity instanceof final FareProduct product) {
        registerAgencyId(FareProduct.class, product.getId());
      } else if (entity instanceof final FareMedium medium) {
        registerAgencyId(FareMedium.class, medium.getId());
      } else if (entity instanceof final RiderCategory category) {
        registerAgencyId(RiderCategory.class, category.getId());
      } else if (entity instanceof final FareAttribute fare) {
        registerAgencyId(FareAttribute.class, fare.getId());
      } else if (entity instanceof final Area area) {
        registerAgencyId(Area.class, area.getId());
      } else if (entity instanceof final Location location) {
        registerAgencyId(Location.class, location.getId());
      } else if (entity instanceof final LocationGroup group) {
        registerAgencyId(LocationGroup.class, group.getId());
      } else if (entity instanceof final LocationGroupElement locationGroupElement) {
        var locationGroup =
            _entityStore.getEntityForId(
                LocationGroup.class, locationGroupElement.getLocationGroupId());
        Objects.requireNonNull(
            locationGroup,
            "Cannot find location group for id: %s"
                .formatted(locationGroupElement.getLocationGroupId()));
        locationGroup.addLocation(locationGroupElement.getStop());
      } else if (entity instanceof final StopAreaElement stopAreaElement) {
        var area = _entityStore.getEntityForId(Area.class, stopAreaElement.getArea().getId());
        area.addStop(stopAreaElement.getStop());
      } else if (entity instanceof final Vehicle vehicle) {
        registerAgencyId(Vehicle.class, vehicle.getId());
      } else if (entity instanceof final Facility facility) {
        registerAgencyId(Facility.class, facility.getId());
      } else if (entity instanceof final FacilityPropertyDefinition facilityPropertyDefinition) {
        registerAgencyId(FacilityPropertyDefinition.class, facilityPropertyDefinition.getId());
      } else if (entity instanceof final Icon icon) {
        registerAgencyId(Icon.class, icon.getId());
      }

      if (entity instanceof IdentityBean<?>) {
        _entityStore.saveEntity(entity);
      }
    }

    private void registerAgencyId(Class<?> entityType, AgencyAndId id) {

      Map<String, String> agencyIdsByEntityId = _agencyIdsByEntityClassAndId.get(entityType);

      if (agencyIdsByEntityId == null) {
        agencyIdsByEntityId = new HashMap<>();
        _agencyIdsByEntityClassAndId.put(entityType, agencyIdsByEntityId);
      }

      if (agencyIdsByEntityId.containsKey(id.getId()) && !_overwriteDuplicates) {
        throw new DuplicateEntityException(entityType, id);
      }

      agencyIdsByEntityId.put(id.getId(), id.getAgencyId());
    }
  }

  private class GtfsReaderContextImpl implements GtfsReaderContext {

    public Object getEntity(Class<?> entityClass, Serializable id) {
      return GtfsReader.this.getEntity(entityClass, id);
    }

    public String getDefaultAgencyId() {
      return GtfsReader.this.getDefaultAgencyId();
    }

    public List<Agency> getAgencies() {
      return GtfsReader.this.getAgencies();
    }

    public String getAgencyForEntity(Class<?> entityType, String entityId) {
      return GtfsReader.this.getAgencyForEntity(entityType, entityId);
    }

    public String getTranslatedAgencyId(String agencyId) {
      return GtfsReader.this.getTranslatedAgencyId(agencyId);
    }
  }
}
