/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.aaa.cert.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Base64;
import java.util.Date;
import java.util.concurrent.TimeUnit;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequestBuilder;
import org.opendaylight.aaa.cert.impl.KeyStoreConstant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ODLKeyTool {
    private static final Logger LOG = LoggerFactory.getLogger(ODLKeyTool.class);
    private static final SecureRandom RANDOM = new SecureRandom();
    private final String workingDir;

    protected ODLKeyTool() {
        this(KeyStoreConstant.KEY_STORE_PATH);
    }

    public ODLKeyTool(String workingDirectory) {
        this.workingDir = workingDirectory;
        KeyStoreConstant.createDir(this.workingDir);
    }

    public KeyStore addCertificate(KeyStore keyStore, String certificate, String alias, boolean deleteOld) {
        try {
            X509Certificate newCert = ODLKeyTool.getCertificate(certificate);
            if (keyStore.isCertificateEntry(alias) && deleteOld) {
                keyStore.deleteEntry(alias);
            }
            if (newCert == null) {
                LOG.warn("{} Not a valid certificate {}", (Object)alias, (Object)certificate);
                return null;
            }
            keyStore.setCertificateEntry(alias, newCert);
            return keyStore;
        }
        catch (KeyStoreException e) {
            LOG.error("failed to add certificate", (Throwable)e);
            return null;
        }
    }

    public byte[] convertKeystoreToBytes(KeyStore keyStore, String keystorePassword) {
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        try {
            keyStore.store(byteArrayOutputStream, keystorePassword.toCharArray());
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            LOG.error("Fatal error convert keystore to bytes", (Throwable)e);
        }
        return byteArrayOutputStream.toByteArray();
    }

    public KeyStore createKeyStoreWithSelfSignCert(String keyStoreName, String keystorePassword, String distinguishedName, String keyAlias, int validity) {
        return this.createKeyStoreWithSelfSignCert(keyStoreName, keystorePassword, distinguishedName, keyAlias, validity, "RSA", 2048, "SHA1WithRSAEncryption");
    }

    public KeyStore createKeyStoreWithSelfSignCert(String keyStoreName, String keystorePassword, String distinguishedName, String keyAlias, int validity, String keyAlg, int keySize, String signAlg) {
        try {
            KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(keyAlg);
            keyPairGenerator.initialize(keySize);
            KeyPair keyPair = keyPairGenerator.generateKeyPair();
            long currTime = System.currentTimeMillis();
            SubjectPublicKeyInfo keyInfo = SubjectPublicKeyInfo.getInstance((Object)keyPair.getPublic().getEncoded());
            X509v3CertificateBuilder x509V3CertBuilder = new X509v3CertificateBuilder(new X500Name(distinguishedName), ODLKeyTool.getSecureRandomeInt(), new Date(currTime), new Date(currTime + TimeUnit.DAYS.toMillis(validity)), new X500Name(distinguishedName), keyInfo);
            X509CertificateHolder x509Cert = x509V3CertBuilder.build(new JcaContentSignerBuilder(signAlg).build(keyPair.getPrivate()));
            KeyStore ctlKeyStore = KeyStore.getInstance("JKS");
            ctlKeyStore.load(null, keystorePassword.toCharArray());
            Certificate[] chain = new Certificate[]{new JcaX509CertificateConverter().getCertificate(x509Cert)};
            ctlKeyStore.setKeyEntry(keyAlias, keyPair.getPrivate(), keystorePassword.toCharArray(), chain);
            LOG.info("{} is created", (Object)keyStoreName);
            return ctlKeyStore;
        }
        catch (IOException | SecurityException | KeyStoreException | NoSuchAlgorithmException | CertificateException | OperatorCreationException e) {
            LOG.error("Fatal error creating keystore", e);
            return null;
        }
    }

    public KeyStore createEmptyKeyStore(String keystorePassword) {
        try {
            KeyStore trustKeyStore = KeyStore.getInstance("JKS");
            trustKeyStore.load(null, keystorePassword.toCharArray());
            return trustKeyStore;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            LOG.error("Failed to create trust keystore", (Throwable)e);
            return null;
        }
    }

    public boolean exportKeystore(KeyStore keystore, String keystorePassword, String fileName) {
        boolean bl;
        if (keystore == null) {
            return false;
        }
        File realPath = KeyStoreConstant.toAbsoluteFile(fileName, this.workingDir);
        FileOutputStream fOutputStream = new FileOutputStream(realPath);
        try {
            keystore.store(fOutputStream, keystorePassword.toCharArray());
            bl = true;
        }
        catch (Throwable throwable) {
            try {
                try {
                    fOutputStream.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
                LOG.error("Fatal error export keystore", (Throwable)e);
                return false;
            }
        }
        fOutputStream.close();
        return bl;
    }

    public String generateCertificateReq(KeyStore keyStore, String keystorePassword, String keyAlias, String signAlg, boolean withTag) {
        try {
            if (keyStore.containsAlias(keyAlias)) {
                X509Certificate odlCert = (X509Certificate)keyStore.getCertificate(keyAlias);
                PublicKey pubKey = odlCert.getPublicKey();
                PrivateKey privKey = (PrivateKey)keyStore.getKey(keyAlias, keystorePassword.toCharArray());
                String subject = odlCert.getSubjectDN().getName();
                X500Name xName = new X500Name(subject);
                SubjectPublicKeyInfo subPubKeyInfo = SubjectPublicKeyInfo.getInstance((Object)pubKey.getEncoded());
                PKCS10CertificationRequestBuilder csrb = new PKCS10CertificationRequestBuilder(xName, subPubKeyInfo);
                ContentSigner contSigner = new JcaContentSignerBuilder(signAlg).build(privKey);
                String certReq = Base64.getEncoder().encodeToString(csrb.build(contSigner).getEncoded());
                return !withTag ? certReq : "-----BEGIN CERTIFICATE REQUEST-----" + '\n' + certReq + '\n' + "-----END CERTIFICATE REQUEST-----";
            }
            LOG.info("KeyStore does not contain alias {}", (Object)keyAlias);
            return "";
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | OperatorCreationException e) {
            LOG.error("Failed to generate certificate request", e);
            return "";
        }
    }

    public String getCertificate(KeyStore keyStore, String certAlias, boolean withTag) {
        try {
            if (keyStore.containsAlias(certAlias)) {
                X509Certificate odlCert = (X509Certificate)keyStore.getCertificate(certAlias);
                String cert = Base64.getEncoder().encodeToString(odlCert.getEncoded());
                return !withTag ? cert : "-----BEGIN CERTIFICATE-----" + '\n' + cert + '\n' + "-----END CERTIFICATE-----";
            }
            LOG.info("KeyStore does not contain alias {}", (Object)certAlias);
            return "";
        }
        catch (KeyStoreException | CertificateException e) {
            LOG.error("Failed to get Certificate", (Throwable)e);
            return "";
        }
    }

    private static X509Certificate getCertificate(String certificate) {
        if (certificate.isEmpty()) {
            return null;
        }
        if (certificate.contains("-----BEGIN CERTIFICATE-----")) {
            int fIdx = certificate.indexOf("-----BEGIN CERTIFICATE-----") + "-----BEGIN CERTIFICATE-----".length();
            int sIdx = certificate.indexOf("-----END CERTIFICATE-----");
            certificate = certificate.substring(fIdx, sIdx);
        }
        byte[] byteCert = Base64.getDecoder().decode(certificate);
        ByteArrayInputStream inputStreamCert = new ByteArrayInputStream(byteCert);
        try {
            CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
            X509Certificate newCert = (X509Certificate)certFactory.generateCertificate(inputStreamCert);
            newCert.checkValidity();
            return newCert;
        }
        catch (CertificateException e) {
            LOG.error("Failed to get certificate", (Throwable)e);
            return null;
        }
    }

    private static BigInteger getSecureRandomeInt() {
        BigInteger bigInt = BigInteger.valueOf(RANDOM.nextInt());
        return new BigInteger(1, bigInt.toByteArray());
    }

    public KeyStore loadKeyStore(byte[] keyStoreBytes, String keystorePassword) {
        try {
            KeyStore keyStore = KeyStore.getInstance("JKS");
            keyStore.load(new ByteArrayInputStream(keyStoreBytes), keystorePassword.toCharArray());
            return keyStore;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            LOG.error("Fatal error load keystore", (Throwable)e);
            return null;
        }
    }

    public KeyStore loadKeyStore(String keyStoreName, String keystorePassword) {
        KeyStore keyStore;
        File realPath = KeyStoreConstant.toAbsoluteFile(keyStoreName, this.workingDir);
        FileInputStream fInputStream = new FileInputStream(realPath);
        try {
            KeyStore keyStore2 = KeyStore.getInstance("JKS");
            keyStore2.load(fInputStream, keystorePassword.toCharArray());
            keyStore = keyStore2;
        }
        catch (Throwable throwable) {
            try {
                try {
                    fInputStream.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
                LOG.error("failed to get keystore {}", (Object)e.getMessage());
                return null;
            }
        }
        fInputStream.close();
        return keyStore;
    }
}

