package org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.errors.errors;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;
import java.lang.Class;
import java.lang.Integer;
import java.lang.Override;
import java.lang.String;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.Nullable;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.$YangModuleInfoImpl;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.DataResourceIdentifier;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.errors.Errors;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.errors.errors.error.ErrorInfo;
import org.opendaylight.yangtools.yang.binding.Augmentable;
import org.opendaylight.yangtools.yang.binding.ChildOf;
import org.opendaylight.yangtools.yang.binding.Enumeration;
import org.opendaylight.yangtools.yang.common.QName;

/**
 * An entry containing information about one specific error that occurred while 
 * processing a RESTCONF request.
 *
 * <p>
 * This class represents the following YANG schema fragment defined in module <b>ietf-restconf</b>
 * <pre>
 * list error {
 *   leaf error-type {
 *     type enumeration {
 *       enum transport {
 *       }
 *       enum rpc {
 *       }
 *       enum protocol {
 *       }
 *       enum application {
 *       }
 *     }
 *   }
 *   leaf error-tag {
 *     type string;
 *   }
 *   leaf error-app-tag {
 *     type string;
 *   }
 *   leaf error-path {
 *     type data-resource-identifier;
 *   }
 *   leaf error-message {
 *     type string;
 *   }
 *   container error-info {
 *   }
 * }
 * </pre>The schema path to identify an instance is
 * <i>ietf-restconf/errors/errors/error</i>
 *
 * <p>To create instances of this class use {@link ErrorBuilder}.
 * @see ErrorBuilder
 *
 *
 */
public interface Error
    extends
    ChildOf<Errors>,
    Augmentable<Error>
{


    public enum ErrorType implements Enumeration {
        /**
         * The transport layer
         *
         */
        Transport(0, "transport"),
        
        /**
         * The rpc or notification layer
         *
         */
        Rpc(1, "rpc"),
        
        /**
         * The protocol operation layer
         *
         */
        Protocol(2, "protocol"),
        
        /**
         * The server application layer
         *
         */
        Application(3, "application")
        ;
    
        private static final Map<String, ErrorType> NAME_MAP;
        private static final Map<Integer, ErrorType> VALUE_MAP;
    
        static {
            final Builder<String, ErrorType> nb = ImmutableMap.builder();
            final Builder<Integer, ErrorType> vb = ImmutableMap.builder();
            for (ErrorType enumItem : ErrorType.values()) {
                vb.put(enumItem.value, enumItem);
                nb.put(enumItem.name, enumItem);
            }
    
            NAME_MAP = nb.build();
            VALUE_MAP = vb.build();
        }
    
        private final String name;
        private final int value;
    
        private ErrorType(int value, String name) {
            this.value = value;
            this.name = name;
        }
    
        @Override
        public String getName() {
            return name;
        }
    
        @Override
        public int getIntValue() {
            return value;
        }
    
        /**
         * Return the enumeration member whose {@link #getName()} matches specified value.
         *
         * @param name YANG assigned name
         * @return corresponding ErrorType item, if present
         * @throws NullPointerException if name is null
         */
        public static Optional<ErrorType> forName(String name) {
            return Optional.ofNullable(NAME_MAP.get(Objects.requireNonNull(name)));
        }
    
        /**
         * Return the enumeration member whose {@link #getIntValue()} matches specified value.
         *
         * @param intValue integer value
         * @return corresponding ErrorType item, or null if no such item exists
         */
        public static ErrorType forValue(int intValue) {
            return VALUE_MAP.get(intValue);
        }
    }

    public static final @NonNull QName QNAME = $YangModuleInfoImpl.qnameOf("error");

    @Override
    default Class<org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.errors.errors.Error> implementedInterface() {
        return org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.errors.errors.Error.class;
    }
    
    /**
     * The protocol layer where the error occurred.
     *
     *
     *
     * @return <code>org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.errors.errors.Error.ErrorType</code> <code>errorType</code>, or <code>null</code> if not present
     */
    @Nullable ErrorType getErrorType();
    
    /**
     * The enumerated error tag.
     *
     *
     *
     * @return <code>java.lang.String</code> <code>errorTag</code>, or <code>null</code> if not present
     */
    @Nullable String getErrorTag();
    
    /**
     * The application-specific error tag.
     *
     *
     *
     * @return <code>java.lang.String</code> <code>errorAppTag</code>, or <code>null</code> if not present
     */
    @Nullable String getErrorAppTag();
    
    /**
     * The target data resource identifier associated with the error, if any.
     *
     *
     *
     * @return <code>org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.DataResourceIdentifier</code> <code>errorPath</code>, or <code>null</code> if not present
     */
    @Nullable DataResourceIdentifier getErrorPath();
    
    /**
     * A message describing the error.
     *
     *
     *
     * @return <code>java.lang.String</code> <code>errorMessage</code>, or <code>null</code> if not present
     */
    @Nullable String getErrorMessage();
    
    /**
     * A container allowing additional information to be included in the error report.
     *
     *
     *
     * @return <code>org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.errors.errors.error.ErrorInfo</code> <code>errorInfo</code>, or <code>null</code> if not present
     */
    @Nullable ErrorInfo getErrorInfo();

}

