package org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.errors.errors;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.DataResourceIdentifier;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.errors.errors.error.ErrorInfo;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link ErrorBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     ErrorBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new ErrorBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of ErrorBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see ErrorBuilder
 * @see Builder
 *
 */
public class ErrorBuilder implements Builder<Error> {

    private String _errorAppTag;
    private ErrorInfo _errorInfo;
    private String _errorMessage;
    private DataResourceIdentifier _errorPath;
    private String _errorTag;
    private Error.ErrorType _errorType;


    Map<Class<? extends Augmentation<Error>>, Augmentation<Error>> augmentation = Collections.emptyMap();

    public ErrorBuilder() {
    }

    public ErrorBuilder(Error base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<Error>>, Augmentation<Error>> aug =((AugmentationHolder<Error>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._errorAppTag = base.getErrorAppTag();
        this._errorInfo = base.getErrorInfo();
        this._errorMessage = base.getErrorMessage();
        this._errorPath = base.getErrorPath();
        this._errorTag = base.getErrorTag();
        this._errorType = base.getErrorType();
    }


    public String getErrorAppTag() {
        return _errorAppTag;
    }
    
    public ErrorInfo getErrorInfo() {
        return _errorInfo;
    }
    
    public String getErrorMessage() {
        return _errorMessage;
    }
    
    public DataResourceIdentifier getErrorPath() {
        return _errorPath;
    }
    
    public String getErrorTag() {
        return _errorTag;
    }
    
    public Error.ErrorType getErrorType() {
        return _errorType;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<Error>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public ErrorBuilder setErrorAppTag(final String value) {
        this._errorAppTag = value;
        return this;
    }
    
    public ErrorBuilder setErrorInfo(final ErrorInfo value) {
        this._errorInfo = value;
        return this;
    }
    
    public ErrorBuilder setErrorMessage(final String value) {
        this._errorMessage = value;
        return this;
    }
    
    public ErrorBuilder setErrorPath(final DataResourceIdentifier value) {
        this._errorPath = value;
        return this;
    }
    
    public ErrorBuilder setErrorTag(final String value) {
        this._errorTag = value;
        return this;
    }
    
    public ErrorBuilder setErrorType(final Error.ErrorType value) {
        this._errorType = value;
        return this;
    }
    
    public ErrorBuilder addAugmentation(Class<? extends Augmentation<Error>> augmentationType, Augmentation<Error> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public ErrorBuilder removeAugmentation(Class<? extends Augmentation<Error>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public Error build() {
        return new ErrorImpl(this);
    }

    private static final class ErrorImpl
        extends AbstractAugmentable<Error>
        implements Error {
    
        private final String _errorAppTag;
        private final ErrorInfo _errorInfo;
        private final String _errorMessage;
        private final DataResourceIdentifier _errorPath;
        private final String _errorTag;
        private final Error.ErrorType _errorType;
    
        ErrorImpl(ErrorBuilder base) {
            super(base.augmentation);
            this._errorAppTag = base.getErrorAppTag();
            this._errorInfo = base.getErrorInfo();
            this._errorMessage = base.getErrorMessage();
            this._errorPath = base.getErrorPath();
            this._errorTag = base.getErrorTag();
            this._errorType = base.getErrorType();
        }
    
        @Override
        public String getErrorAppTag() {
            return _errorAppTag;
        }
        
        @Override
        public ErrorInfo getErrorInfo() {
            return _errorInfo;
        }
        
        @Override
        public String getErrorMessage() {
            return _errorMessage;
        }
        
        @Override
        public DataResourceIdentifier getErrorPath() {
            return _errorPath;
        }
        
        @Override
        public String getErrorTag() {
            return _errorTag;
        }
        
        @Override
        public Error.ErrorType getErrorType() {
            return _errorType;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_errorAppTag);
            result = prime * result + Objects.hashCode(_errorInfo);
            result = prime * result + Objects.hashCode(_errorMessage);
            result = prime * result + Objects.hashCode(_errorPath);
            result = prime * result + Objects.hashCode(_errorTag);
            result = prime * result + Objects.hashCode(_errorType);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!Error.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            Error other = (Error)obj;
            if (!Objects.equals(_errorAppTag, other.getErrorAppTag())) {
                return false;
            }
            if (!Objects.equals(_errorInfo, other.getErrorInfo())) {
                return false;
            }
            if (!Objects.equals(_errorMessage, other.getErrorMessage())) {
                return false;
            }
            if (!Objects.equals(_errorPath, other.getErrorPath())) {
                return false;
            }
            if (!Objects.equals(_errorTag, other.getErrorTag())) {
                return false;
            }
            if (!Objects.equals(_errorType, other.getErrorType())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                ErrorImpl otherImpl = (ErrorImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<Error>>, Augmentation<Error>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("Error");
            CodeHelpers.appendValue(helper, "_errorAppTag", _errorAppTag);
            CodeHelpers.appendValue(helper, "_errorInfo", _errorInfo);
            CodeHelpers.appendValue(helper, "_errorMessage", _errorMessage);
            CodeHelpers.appendValue(helper, "_errorPath", _errorPath);
            CodeHelpers.appendValue(helper, "_errorTag", _errorTag);
            CodeHelpers.appendValue(helper, "_errorType", _errorType);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
