package org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.restconf;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.restconf.restconf.Config;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.restconf.restconf.Modules;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.restconf.restconf.Operational;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.restconf.restconf.Operations;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.restconf.restconf.Streams;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link RestconfBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     RestconfBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new RestconfBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of RestconfBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see RestconfBuilder
 * @see Builder
 *
 */
public class RestconfBuilder implements Builder<Restconf> {

    private Config _config;
    private Modules _modules;
    private Operational _operational;
    private Operations _operations;
    private Streams _streams;
    private Restconf.Version _version;


    Map<Class<? extends Augmentation<Restconf>>, Augmentation<Restconf>> augmentation = Collections.emptyMap();

    public RestconfBuilder() {
    }

    public RestconfBuilder(Restconf base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<Restconf>>, Augmentation<Restconf>> aug =((AugmentationHolder<Restconf>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._config = base.getConfig();
        this._modules = base.getModules();
        this._operational = base.getOperational();
        this._operations = base.getOperations();
        this._streams = base.getStreams();
        this._version = base.getVersion();
    }


    public Config getConfig() {
        return _config;
    }
    
    public Modules getModules() {
        return _modules;
    }
    
    public Operational getOperational() {
        return _operational;
    }
    
    public Operations getOperations() {
        return _operations;
    }
    
    public Streams getStreams() {
        return _streams;
    }
    
    public Restconf.Version getVersion() {
        return _version;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<Restconf>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public RestconfBuilder setConfig(final Config value) {
        this._config = value;
        return this;
    }
    
    public RestconfBuilder setModules(final Modules value) {
        this._modules = value;
        return this;
    }
    
    public RestconfBuilder setOperational(final Operational value) {
        this._operational = value;
        return this;
    }
    
    public RestconfBuilder setOperations(final Operations value) {
        this._operations = value;
        return this;
    }
    
    public RestconfBuilder setStreams(final Streams value) {
        this._streams = value;
        return this;
    }
    
    public RestconfBuilder setVersion(final Restconf.Version value) {
        this._version = value;
        return this;
    }
    
    public RestconfBuilder addAugmentation(Class<? extends Augmentation<Restconf>> augmentationType, Augmentation<Restconf> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public RestconfBuilder removeAugmentation(Class<? extends Augmentation<Restconf>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public Restconf build() {
        return new RestconfImpl(this);
    }

    private static final class RestconfImpl
        extends AbstractAugmentable<Restconf>
        implements Restconf {
    
        private final Config _config;
        private final Modules _modules;
        private final Operational _operational;
        private final Operations _operations;
        private final Streams _streams;
        private final Restconf.Version _version;
    
        RestconfImpl(RestconfBuilder base) {
            super(base.augmentation);
            this._config = base.getConfig();
            this._modules = base.getModules();
            this._operational = base.getOperational();
            this._operations = base.getOperations();
            this._streams = base.getStreams();
            this._version = base.getVersion();
        }
    
        @Override
        public Config getConfig() {
            return _config;
        }
        
        @Override
        public Modules getModules() {
            return _modules;
        }
        
        @Override
        public Operational getOperational() {
            return _operational;
        }
        
        @Override
        public Operations getOperations() {
            return _operations;
        }
        
        @Override
        public Streams getStreams() {
            return _streams;
        }
        
        @Override
        public Restconf.Version getVersion() {
            return _version;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_config);
            result = prime * result + Objects.hashCode(_modules);
            result = prime * result + Objects.hashCode(_operational);
            result = prime * result + Objects.hashCode(_operations);
            result = prime * result + Objects.hashCode(_streams);
            result = prime * result + Objects.hashCode(_version);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!Restconf.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            Restconf other = (Restconf)obj;
            if (!Objects.equals(_config, other.getConfig())) {
                return false;
            }
            if (!Objects.equals(_modules, other.getModules())) {
                return false;
            }
            if (!Objects.equals(_operational, other.getOperational())) {
                return false;
            }
            if (!Objects.equals(_operations, other.getOperations())) {
                return false;
            }
            if (!Objects.equals(_streams, other.getStreams())) {
                return false;
            }
            if (!Objects.equals(_version, other.getVersion())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                RestconfImpl otherImpl = (RestconfImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<Restconf>>, Augmentation<Restconf>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("Restconf");
            CodeHelpers.appendValue(helper, "_config", _config);
            CodeHelpers.appendValue(helper, "_modules", _modules);
            CodeHelpers.appendValue(helper, "_operational", _operational);
            CodeHelpers.appendValue(helper, "_operations", _operations);
            CodeHelpers.appendValue(helper, "_streams", _streams);
            CodeHelpers.appendValue(helper, "_version", _version);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
