package org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.restconf.restconf.modules;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.Uri;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.yang.types.rev130715.YangIdentifier;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link ModuleBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     ModuleBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new ModuleBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of ModuleBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see ModuleBuilder
 * @see Builder
 *
 */
public class ModuleBuilder implements Builder<Module> {

    private List<YangIdentifier> _deviation;
    private List<YangIdentifier> _feature;
    private YangIdentifier _name;
    private Uri _namespace;
    private Module.Revision _revision;
    private ModuleKey key;


    Map<Class<? extends Augmentation<Module>>, Augmentation<Module>> augmentation = Collections.emptyMap();

    public ModuleBuilder() {
    }

    public ModuleBuilder(Module base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<Module>>, Augmentation<Module>> aug =((AugmentationHolder<Module>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._name = base.getName();
        this._revision = base.getRevision();
        this._deviation = base.getDeviation();
        this._feature = base.getFeature();
        this._namespace = base.getNamespace();
    }


    public ModuleKey key() {
        return key;
    }
    
    public List<YangIdentifier> getDeviation() {
        return _deviation;
    }
    
    public List<YangIdentifier> getFeature() {
        return _feature;
    }
    
    public YangIdentifier getName() {
        return _name;
    }
    
    public Uri getNamespace() {
        return _namespace;
    }
    
    public Module.Revision getRevision() {
        return _revision;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<Module>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public ModuleBuilder withKey(final ModuleKey key) {
        this.key = key;
        return this;
    }
    public ModuleBuilder setDeviation(final List<YangIdentifier> values) {
        this._deviation = values;
        return this;
    }
    
    public ModuleBuilder setFeature(final List<YangIdentifier> values) {
        this._feature = values;
        return this;
    }
    
    
    public ModuleBuilder setName(final YangIdentifier value) {
        this._name = value;
        return this;
    }
    
    public ModuleBuilder setNamespace(final Uri value) {
        this._namespace = value;
        return this;
    }
    
    public ModuleBuilder setRevision(final Module.Revision value) {
        this._revision = value;
        return this;
    }
    
    public ModuleBuilder addAugmentation(Class<? extends Augmentation<Module>> augmentationType, Augmentation<Module> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public ModuleBuilder removeAugmentation(Class<? extends Augmentation<Module>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public Module build() {
        return new ModuleImpl(this);
    }

    private static final class ModuleImpl
        extends AbstractAugmentable<Module>
        implements Module {
    
        private final List<YangIdentifier> _deviation;
        private final List<YangIdentifier> _feature;
        private final YangIdentifier _name;
        private final Uri _namespace;
        private final Module.Revision _revision;
        private final ModuleKey key;
    
        ModuleImpl(ModuleBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new ModuleKey(base.getName(), base.getRevision());
            }
            this._name = key.getName();
            this._revision = key.getRevision();
            this._deviation = base.getDeviation();
            this._feature = base.getFeature();
            this._namespace = base.getNamespace();
        }
    
        @Override
        public ModuleKey key() {
            return key;
        }
        
        @Override
        public List<YangIdentifier> getDeviation() {
            return _deviation;
        }
        
        @Override
        public List<YangIdentifier> getFeature() {
            return _feature;
        }
        
        @Override
        public YangIdentifier getName() {
            return _name;
        }
        
        @Override
        public Uri getNamespace() {
            return _namespace;
        }
        
        @Override
        public Module.Revision getRevision() {
            return _revision;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_deviation);
            result = prime * result + Objects.hashCode(_feature);
            result = prime * result + Objects.hashCode(_name);
            result = prime * result + Objects.hashCode(_namespace);
            result = prime * result + Objects.hashCode(_revision);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!Module.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            Module other = (Module)obj;
            if (!Objects.equals(_deviation, other.getDeviation())) {
                return false;
            }
            if (!Objects.equals(_feature, other.getFeature())) {
                return false;
            }
            if (!Objects.equals(_name, other.getName())) {
                return false;
            }
            if (!Objects.equals(_namespace, other.getNamespace())) {
                return false;
            }
            if (!Objects.equals(_revision, other.getRevision())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                ModuleImpl otherImpl = (ModuleImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<Module>>, Augmentation<Module>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("Module");
            CodeHelpers.appendValue(helper, "_deviation", _deviation);
            CodeHelpers.appendValue(helper, "_feature", _feature);
            CodeHelpers.appendValue(helper, "_name", _name);
            CodeHelpers.appendValue(helper, "_namespace", _namespace);
            CodeHelpers.appendValue(helper, "_revision", _revision);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
