package org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.yang.patch;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.restconf.rev131019.yang.patch.yang.patch.Edit;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link YangPatchBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     YangPatchBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new YangPatchBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of YangPatchBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see YangPatchBuilder
 * @see Builder
 *
 */
public class YangPatchBuilder implements Builder<YangPatch> {

    private String _comment;
    private List<Edit> _edit;
    private String _patchId;


    Map<Class<? extends Augmentation<YangPatch>>, Augmentation<YangPatch>> augmentation = Collections.emptyMap();

    public YangPatchBuilder() {
    }

    public YangPatchBuilder(YangPatch base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<YangPatch>>, Augmentation<YangPatch>> aug =((AugmentationHolder<YangPatch>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._comment = base.getComment();
        this._edit = base.getEdit();
        this._patchId = base.getPatchId();
    }


    public String getComment() {
        return _comment;
    }
    
    public List<Edit> getEdit() {
        return _edit;
    }
    
    public String getPatchId() {
        return _patchId;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<YangPatch>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    private static void check_commentLength(final String value) {
        final int length = value.length();
        if (length <= 1024) {
            return;
        }
        CodeHelpers.throwInvalidLength("[[0..1024]]", value);
    }
    
    public YangPatchBuilder setComment(final String value) {
    if (value != null) {
        check_commentLength(value);
        
    }
        this._comment = value;
        return this;
    }
    public YangPatchBuilder setEdit(final List<Edit> values) {
        this._edit = values;
        return this;
    }
    
    
    public YangPatchBuilder setPatchId(final String value) {
        this._patchId = value;
        return this;
    }
    
    public YangPatchBuilder addAugmentation(Class<? extends Augmentation<YangPatch>> augmentationType, Augmentation<YangPatch> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public YangPatchBuilder removeAugmentation(Class<? extends Augmentation<YangPatch>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public YangPatch build() {
        return new YangPatchImpl(this);
    }

    private static final class YangPatchImpl
        extends AbstractAugmentable<YangPatch>
        implements YangPatch {
    
        private final String _comment;
        private final List<Edit> _edit;
        private final String _patchId;
    
        YangPatchImpl(YangPatchBuilder base) {
            super(base.augmentation);
            this._comment = base.getComment();
            this._edit = base.getEdit();
            this._patchId = base.getPatchId();
        }
    
        @Override
        public String getComment() {
            return _comment;
        }
        
        @Override
        public List<Edit> getEdit() {
            return _edit;
        }
        
        @Override
        public String getPatchId() {
            return _patchId;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_comment);
            result = prime * result + Objects.hashCode(_edit);
            result = prime * result + Objects.hashCode(_patchId);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!YangPatch.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            YangPatch other = (YangPatch)obj;
            if (!Objects.equals(_comment, other.getComment())) {
                return false;
            }
            if (!Objects.equals(_edit, other.getEdit())) {
                return false;
            }
            if (!Objects.equals(_patchId, other.getPatchId())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                YangPatchImpl otherImpl = (YangPatchImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<YangPatch>>, Augmentation<YangPatch>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("YangPatch");
            CodeHelpers.appendValue(helper, "_comment", _comment);
            CodeHelpers.appendValue(helper, "_edit", _edit);
            CodeHelpers.appendValue(helper, "_patchId", _patchId);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
