/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.netconf.codec;

import com.google.common.base.MoreObjects;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.buffer.Unpooled;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Objects;
import javax.xml.transform.TransformerException;
import org.eclipse.jdt.annotation.NonNull;
import org.opendaylight.netconf.api.messages.FramingMechanism;
import org.opendaylight.netconf.api.messages.NetconfMessage;
import org.opendaylight.netconf.codec.FramingParts;
import org.opendaylight.netconf.codec.MessageWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract sealed class FramingSupport {
    private static final Logger LOG = LoggerFactory.getLogger(FramingSupport.class);
    public static final int DEFAULT_CHUNK_SIZE = 8192;
    public static final int MIN_CHUNK_SIZE = 128;
    public static final int MAX_CHUNK_SIZE = 0x1000000;

    private FramingSupport() {
    }

    public static @NonNull FramingSupport eom() {
        return EOM.INSTANCE;
    }

    public static @NonNull FramingSupport chunk() {
        return Chunk.DEFAULT;
    }

    public static @NonNull FramingSupport chunk(int chunkSize) {
        return chunkSize == 8192 ? Chunk.DEFAULT : new Chunk(chunkSize);
    }

    public abstract @NonNull FramingMechanism mechanism();

    public final String toString() {
        return this.addToStringAttributes(MoreObjects.toStringHelper(FramingSupport.class)).toString();
    }

    MoreObjects.ToStringHelper addToStringAttributes(MoreObjects.ToStringHelper helper) {
        return helper.add("mechanism", (Object)this.mechanism());
    }

    abstract void writeMessage(ByteBufAllocator var1, NetconfMessage var2, MessageWriter var3, ByteBuf var4) throws IOException, TransformerException;

    private static final class EOM
    extends FramingSupport {
        static final @NonNull EOM INSTANCE = new EOM();

        private EOM() {
        }

        @Override
        public FramingMechanism mechanism() {
            return FramingMechanism.EOM;
        }

        @Override
        void writeMessage(ByteBufAllocator alloc, NetconfMessage message, MessageWriter writer, ByteBuf out) throws IOException, TransformerException {
            boolean needEom;
            try (ByteBufOutputStream os = new ByteBufOutputStream(out);){
                writer.writeMessage(message, (OutputStream)os);
                needEom = os.writtenBytes() != 0;
            }
            if (needEom) {
                out.writeBytes(FramingParts.END_OF_MESSAGE);
            }
        }
    }

    private static final class Chunk
    extends FramingSupport {
        static final @NonNull Chunk DEFAULT = new Chunk(8192);
        private final int chunkSize;

        Chunk(int chunkSize) {
            if (chunkSize < 128) {
                throw new IllegalArgumentException(chunkSize + " is lower than minimum supported 128");
            }
            if (chunkSize > 0x1000000) {
                throw new IllegalArgumentException(chunkSize + " is lower than maximum supported 16777216");
            }
            this.chunkSize = chunkSize;
        }

        @Override
        public FramingMechanism mechanism() {
            return FramingMechanism.CHUNK;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        void writeMessage(ByteBufAllocator alloc, NetconfMessage message, MessageWriter writer, ByteBuf out) throws IOException, TransformerException {
            boolean needEnd;
            ByteBuf allocated = alloc.ioBuffer();
            try {
                int size;
                ByteBuf buffer;
                int maxCapacity = allocated.maxCapacity();
                if (maxCapacity < this.chunkSize) {
                    LOG.debug("Allocated buffer cannot support chunk size {}", (Object)this.chunkSize);
                    if (maxCapacity >= 128) {
                        LOG.debug("Using chunk size {}", (Object)maxCapacity);
                        buffer = allocated;
                        size = maxCapacity;
                    } else {
                        LOG.debug("Using chunk size {} with unpooled on-heap buffer", (Object)128);
                        buffer = Unpooled.buffer((int)128);
                        size = 128;
                    }
                } else {
                    buffer = allocated;
                    size = this.chunkSize;
                }
                try (ChunkOutputStream os = new ChunkOutputStream(out, buffer, size);){
                    writer.writeMessage(message, os);
                    needEnd = os.finishMessage();
                }
            }
            finally {
                allocated.release();
            }
            if (needEnd) {
                out.writeBytes(FramingParts.END_OF_CHUNKS);
            }
        }

        @Override
        MoreObjects.ToStringHelper addToStringAttributes(MoreObjects.ToStringHelper helper) {
            return super.addToStringAttributes(helper).add("chunkSize", this.chunkSize);
        }
    }

    private static final class ChunkOutputStream
    extends OutputStream {
        private final ByteBuf buffer;
        private final ByteBuf out;
        private final int chunkSize;
        private boolean needEnd = false;

        ChunkOutputStream(ByteBuf out, ByteBuf buffer, int chunkSize) {
            this.out = Objects.requireNonNull(out);
            this.buffer = Objects.requireNonNull(buffer);
            this.chunkSize = chunkSize;
        }

        @Override
        public void write(int value) throws IOException {
            int size = this.size();
            if (size == this.chunkSize) {
                this.sendChunk(size);
            }
            this.buffer.writeByte(value);
        }

        @Override
        public void write(byte[] bytes, int off, int len) throws IOException {
            int xfer;
            Objects.checkFromIndexSize(off, len, bytes.length);
            if (len == 0) {
                return;
            }
            int from = off;
            int remaining = len;
            int available = this.chunkSize - this.size();
            do {
                if (available == 0) {
                    this.sendChunk(this.chunkSize);
                    available = this.chunkSize;
                }
                xfer = Math.min(remaining, available);
                this.buffer.writeBytes(bytes, from, xfer);
                from += xfer;
                available -= xfer;
            } while ((remaining -= xfer) != 0);
        }

        boolean finishMessage() {
            int size = this.size();
            if (size != 0) {
                this.sendChunk(size);
            }
            return this.needEnd;
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("chunkSize", this.chunkSize).add("pending", this.size()).toString();
        }

        private void sendChunk(int size) {
            this.out.writeBytes(FramingParts.START_OF_CHUNK).writeBytes(Integer.toString(size).getBytes(StandardCharsets.US_ASCII)).writeByte(10).writeBytes(this.buffer);
            this.buffer.clear();
            this.needEnd = true;
        }

        private int size() {
            return this.buffer.writerIndex();
        }
    }
}

