/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.yangtools.yang.model.repo.fs;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.FluentFuture;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.opendaylight.yangtools.util.concurrent.FluentFutures;
import org.opendaylight.yangtools.yang.common.Revision;
import org.opendaylight.yangtools.yang.model.api.source.SourceIdentifier;
import org.opendaylight.yangtools.yang.model.api.source.SourceRepresentation;
import org.opendaylight.yangtools.yang.model.api.source.YangTextSource;
import org.opendaylight.yangtools.yang.model.repo.api.MissingSchemaSourceException;
import org.opendaylight.yangtools.yang.model.repo.spi.AbstractSchemaSourceCache;
import org.opendaylight.yangtools.yang.model.repo.spi.PotentialSchemaSource;
import org.opendaylight.yangtools.yang.model.repo.spi.SchemaSourceRegistry;
import org.opendaylight.yangtools.yang.model.spi.source.FileYangTextSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FilesystemSchemaSourceCache<T extends SourceRepresentation>
extends AbstractSchemaSourceCache<T> {
    private static final Logger LOG = LoggerFactory.getLogger(FilesystemSchemaSourceCache.class);
    private static final Map<Class<? extends SourceRepresentation>, StorageAdapter<? extends SourceRepresentation>> STORAGE_ADAPTERS = Collections.singletonMap(YangTextSource.class, new YangTextStorageAdapter());
    private static final Pattern CACHED_FILE_PATTERN = Pattern.compile("(?<moduleName>[^@]+)(@(?<revision>" + String.valueOf(Revision.STRING_FORMAT_PATTERN) + "))?");
    private final Class<T> representation;
    private final Path storageDirectory;

    @Deprecated(since="14.0.7")
    public FilesystemSchemaSourceCache(SchemaSourceRegistry consumer, Class<T> representation, File storageDirectory) {
        this(consumer, representation, storageDirectory.toPath());
    }

    public FilesystemSchemaSourceCache(SchemaSourceRegistry consumer, Class<T> representation, Path storageDirectory) {
        super(consumer, representation, PotentialSchemaSource.Costs.LOCAL_IO);
        this.representation = representation;
        this.storageDirectory = Objects.requireNonNull(storageDirectory);
        FilesystemSchemaSourceCache.checkSupportedRepresentation(representation);
        try {
            Files.createDirectories(storageDirectory, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IllegalArgumentException("Cannot establist storage at " + String.valueOf(storageDirectory), e);
        }
        Preconditions.checkArgument((boolean)Files.isReadable(storageDirectory));
        Preconditions.checkArgument((boolean)Files.isWritable(storageDirectory));
        this.init();
    }

    private static void checkSupportedRepresentation(Class<? extends SourceRepresentation> representation) {
        for (Class<? extends SourceRepresentation> supportedRepresentation : STORAGE_ADAPTERS.keySet()) {
            if (!supportedRepresentation.isAssignableFrom(representation)) continue;
            return;
        }
        throw new IllegalArgumentException(String.format("This cache does not support representation: %s, supported representations are: %s", representation, STORAGE_ADAPTERS.keySet()));
    }

    private void init() {
        CachedModulesFileVisitor fileVisitor = new CachedModulesFileVisitor();
        try {
            Files.walkFileTree(this.storageDirectory, fileVisitor);
        }
        catch (IOException e) {
            LOG.warn("Unable to restore cache from {}. Starting with an empty cache", (Object)this.storageDirectory, (Object)e);
            return;
        }
        fileVisitor.getCachedSchemas().stream().forEach(x$0 -> this.register((SourceIdentifier)x$0));
    }

    public synchronized FluentFuture<? extends T> getSource(SourceIdentifier sourceIdentifier) {
        Path file = FilesystemSchemaSourceCache.sourceIdToFile(sourceIdentifier, this.storageDirectory);
        if (Files.exists(file, new LinkOption[0]) && Files.isReadable(file)) {
            LOG.trace("Source {} found in cache as {}", (Object)sourceIdentifier, (Object)file);
            SourceRepresentation restored = STORAGE_ADAPTERS.get(this.representation).restore(sourceIdentifier, file);
            return FluentFutures.immediateFluentFuture((Object)((SourceRepresentation)this.representation.cast(restored)));
        }
        LOG.debug("Source {} not found in cache as {}", (Object)sourceIdentifier, (Object)file);
        return FluentFutures.immediateFailedFluentFuture((Throwable)new MissingSchemaSourceException(sourceIdentifier, "Source not found"));
    }

    protected synchronized void offer(T source) {
        LOG.trace("Source {} offered to cache", (Object)source.sourceId());
        Path file = this.sourceIdToFile(source);
        if (Files.exists(file, new LinkOption[0])) {
            LOG.debug("Source {} already in cache as {}", (Object)source.sourceId(), (Object)file);
            return;
        }
        this.storeSource(file, source);
        this.register(source.sourceId());
        LOG.trace("Source {} stored in cache as {}", (Object)source.sourceId(), (Object)file);
    }

    private Path sourceIdToFile(T source) {
        return FilesystemSchemaSourceCache.sourceIdToFile(source.sourceId(), this.storageDirectory);
    }

    static Path sourceIdToFile(SourceIdentifier identifier, Path storageDirectory) {
        Revision rev = identifier.revision();
        return rev != null ? storageDirectory.resolve(identifier.toYangFilename()) : FilesystemSchemaSourceCache.findFileWithNewestRev(identifier, storageDirectory);
    }

    private static Path findFileWithNewestRev(final SourceIdentifier identifier, Path storageDirectory) {
        Path[] files = (Path[])Arrays.stream(storageDirectory.toFile().listFiles(new FilenameFilter(){
            final Pattern pat;
            {
                this.pat = Pattern.compile(Pattern.quote(identifier.name().getLocalName()) + "(\\.yang|@\\d\\d\\d\\d-\\d\\d-\\d\\d.yang)");
            }

            @Override
            public boolean accept(File dir, String name) {
                return this.pat.matcher(name).matches();
            }
        })).map(File::toPath).toArray(Path[]::new);
        if (files.length == 0) {
            return storageDirectory.resolve(identifier.toYangFilename());
        }
        if (files.length == 1) {
            return files[0];
        }
        Path file = null;
        TreeMap map = new TreeMap(Revision::compare);
        for (Path sorted : files) {
            String fileName = sorted.getFileName().toString();
            Matcher match = Revision.STRING_FORMAT_PATTERN.matcher(fileName);
            if (match.find()) {
                Revision rev;
                String revStr = match.group();
                try {
                    rev = Revision.of((String)revStr);
                }
                catch (DateTimeParseException e) {
                    LOG.info("Unable to parse date from yang file name {}, falling back to not-present", (Object)fileName, (Object)e);
                    rev = null;
                }
                map.put(Optional.ofNullable(rev), sorted);
                continue;
            }
            map.put(Optional.empty(), sorted);
        }
        file = (Path)map.lastEntry().getValue();
        return file;
    }

    private void storeSource(Path file, T schemaRepresentation) {
        STORAGE_ADAPTERS.get(this.representation).store(file, (SourceRepresentation)schemaRepresentation);
    }

    private static final class CachedModulesFileVisitor
    extends SimpleFileVisitor<Path> {
        private final ArrayList<SourceIdentifier> cachedSchemas = new ArrayList();

        private CachedModulesFileVisitor() {
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            FileVisitResult fileVisitResult = super.visitFile(file, attrs);
            String fileName = file.toFile().getName();
            Optional<SourceIdentifier> si = CachedModulesFileVisitor.getSourceIdentifier(fileName = com.google.common.io.Files.getNameWithoutExtension((String)fileName));
            if (si.isPresent()) {
                LOG.trace("Restoring cached file {} as {}", (Object)file, (Object)si.orElseThrow());
                this.cachedSchemas.add(si.orElseThrow());
            } else {
                LOG.debug("Skipping cached file {}, cannot restore source identifier from filename: {}, does not match {}", new Object[]{file, fileName, CACHED_FILE_PATTERN});
            }
            return fileVisitResult;
        }

        private static Optional<SourceIdentifier> getSourceIdentifier(String fileName) {
            Matcher matcher = CACHED_FILE_PATTERN.matcher(fileName);
            if (matcher.matches()) {
                String moduleName = matcher.group("moduleName");
                String revision = matcher.group("revision");
                return Optional.of(new SourceIdentifier(moduleName, revision));
            }
            return Optional.empty();
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            LOG.warn("Unable to restore cached file {}. Ignoring", (Object)file, (Object)exc);
            return FileVisitResult.CONTINUE;
        }

        public List<SourceIdentifier> getCachedSchemas() {
            return this.cachedSchemas;
        }
    }

    private static abstract class StorageAdapter<T extends SourceRepresentation> {
        private final Class<T> supportedType;

        protected StorageAdapter(Class<T> supportedType) {
            this.supportedType = supportedType;
        }

        void store(Path file, SourceRepresentation schemaSourceRepresentation) {
            Preconditions.checkArgument((boolean)this.supportedType.isAssignableFrom(schemaSourceRepresentation.getClass()), (String)"Cannot store schema source %s, this adapter only supports %s", (Object)schemaSourceRepresentation, this.supportedType);
            this.storeAsType(file, (SourceRepresentation)this.supportedType.cast(schemaSourceRepresentation));
        }

        protected abstract void storeAsType(Path var1, T var2);

        T restore(SourceIdentifier sourceIdentifier, Path cachedSource) {
            Preconditions.checkArgument((boolean)Files.isRegularFile(cachedSource, new LinkOption[0]));
            Preconditions.checkArgument((boolean)Files.isReadable(cachedSource));
            return this.restoreAsType(sourceIdentifier, cachedSource);
        }

        abstract T restoreAsType(SourceIdentifier var1, Path var2);
    }

    private static final class YangTextStorageAdapter
    extends StorageAdapter<YangTextSource> {
        protected YangTextStorageAdapter() {
            super(YangTextSource.class);
        }

        @Override
        protected void storeAsType(Path file, YangTextSource cast) {
            try (InputStream castStream = cast.asByteSource(StandardCharsets.UTF_8).openStream();){
                Files.copy(castStream, file, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                throw new IllegalStateException("Cannot store schema source " + String.valueOf(cast.sourceId()) + " to " + String.valueOf(file), e);
            }
        }

        @Override
        YangTextSource restoreAsType(SourceIdentifier sourceIdentifier, Path cachedSource) {
            return new FileYangTextSource(sourceIdentifier, cachedSource, StandardCharsets.UTF_8);
        }
    }
}

