package org.openl.ie.constrainer.impl;

import org.openl.ie.constrainer.*;
import org.openl.ie.tools.Reusable;
import org.openl.ie.tools.ReusableFactory;

//
//: IntVarImpl.java
//
/**
 * A generic implementation of the IntVar interface.
 */
public class IntVarImpl extends IntExpImpl implements IntVar {
    /**
     * Undo Class for IntVar.
     */
    static final class UndoIntVarImpl extends UndoSubject {

        static final ReusableFactory _factory = new ReusableFactory() {
            @Override
            protected Reusable createNewElement() {
                return new UndoIntVarImpl();
            }

        };

        int _history_index;

        static UndoIntVarImpl getIntVarUndo() {
            return (UndoIntVarImpl) _factory.getElement();
        }

        /**
         * Returns a String representation of this object.
         *
         * @return a String representation of this object.
         */
        @Override
        public String toString() {
            return "UndoIntVar " + undoable();
        }

        @Override
        public void undo() {
            IntVarImpl intvar = (IntVarImpl) undoable();
            // System.out.println("++ Undo: " + intvar);
            intvar.history().restore(_history_index);
            super.undo();
            // System.out.println("-- Undo: " + intvar);
        }

        @Override
        public void undoable(Undoable u) {
            super.undoable(u);
            IntVarImpl intvar = (IntVarImpl) u;
            _history_index = intvar.history().currentIndex();
            // System.out.println("++ SAVE: " + intvar + "index:" +
            // _history_index);

        }

    } // ~UndoIntVarImpl

    private Domain _domain;

    private final IntDomainHistory _history;

    public IntVarImpl(Constrainer constrainer, int min, int max, String name, int domain_type) {
        super(constrainer, name);

        int size = max - min + 1;

        switch (domain_type) {
            case DOMAIN_PLAIN:
                _domain = new DomainImpl(this, min, max);
                break;
            case DOMAIN_BIT_FAST:
                _domain = new DomainBits(this, min, max);
                break;
            case DOMAIN_BIT_SMALL:
                _domain = new DomainBits2(this, min, max);
                break;
            case DOMAIN_DEFAULT:
                if (size < 16) {
                    _domain = new DomainBits(this, min, max);
                } else if (size < 128) {
                    _domain = new DomainBits2(this, min, max);
                } else {
                    _domain = new DomainImpl(this, min, max);
                }
                break;
        }

        _history = new IntDomainHistory(this);
    }

    @Override
    public boolean contains(int value) {
        return _domain.contains(value);
    }

    @Override
    public Undo createUndo() {
        _history.saveUndo();
        return UndoIntVarImpl.getIntVarUndo();
    }

    @Override
    public String domainToString() {
        return _domain.toString();
    }

    @Override
    public int domainType() {
        return _domain.type();
    }

    @Override
    public void forceInsert(int val) {
        _domain.forceInsert(val);
    }

    @Override
    public void forceMax(int val) {
        _domain.forceMax(val);
    }

    @Override
    public void forceMin(int val) {
        _domain.forceMin(val);
    }

    @Override
    public void forceSize(int val) {
        _domain.forceSize(val);
    }

    public IntDomainHistory history() {
        return _history;
    }

    @Override
    public boolean isLinear() {
        return true;
    }

    @Override
    public void iterateDomain(IntExp.IntDomainIterator it) throws Failure {
        _domain.iterateDomain(it);
    }

    @Override
    public int max() {
        return _domain.max();
    }

    @Override
    public int min() {
        return _domain.min();
    }

    @Override
    public void propagate() throws Failure {
        _history.propagate();
    }

    @Override
    protected void removeRangeInternal(int min, int max) throws Failure {
        if (_domain.removeRange(min, max)) {
            _history.setMin(_domain.min());
            _history.setMax(_domain.max());
            _history.remove(min, max);
            addToPropagationQueue();
        }
    }

    // removes only min and max value (leaves holes)
    @Override
    public void removeValue(int value) throws Failure {
        if (_domain.removeValue(value)) {
            _history.setMin(_domain.min());
            _history.setMax(_domain.max());
            _history.remove(value);
            addToPropagationQueue();
        }
    }

    @Override
    public void setMax(int max) throws Failure {
        if (_domain.setMax(max)) {
            _history.setMax(max());
            addToPropagationQueue();
        }
    }

    @Override
    public void setMin(int min) throws Failure {
        if (_domain.setMin(min)) {
            _history.setMin(min());
            addToPropagationQueue();
        }
    }

    @Override
    public void setValue(int value) throws Failure {
        setMin(value);
        setMax(value);
    }

    @Override
    public int size() {
        return _domain.size();
    }

    @Override
    public int value() throws Failure {
        if (!bound()) {
            constrainer().fail("Attempt to get value of the unbound variable " + this);
        }
        return _domain.min();
    }

} // ~IntVarImpl
