package org.openl.excel.parser.sax;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.poi.openxml4j.exceptions.InvalidFormatException;
import org.apache.poi.openxml4j.exceptions.OpenXML4JException;
import org.apache.poi.openxml4j.opc.OPCPackage;
import org.apache.poi.openxml4j.opc.PackageAccess;
import org.apache.poi.openxml4j.opc.PackagePart;
import org.apache.poi.openxml4j.opc.PackageRelationship;
import org.apache.poi.openxml4j.opc.PackageRelationshipCollection;
import org.apache.poi.openxml4j.opc.PackageRelationshipTypes;
import org.apache.poi.openxml4j.opc.PackagingURIHelper;
import org.apache.poi.ss.util.CellAddress;
import org.apache.poi.util.XMLHelper;
import org.apache.poi.xssf.eventusermodel.XSSFReader;
import org.apache.poi.xssf.model.CommentsTable;
import org.apache.poi.xssf.usermodel.XSSFRelation;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import org.openl.excel.parser.ExcelParseException;
import org.openl.excel.parser.ExcelReader;
import org.openl.excel.parser.ExcelUtils;
import org.openl.excel.parser.ParserDateUtil;
import org.openl.excel.parser.SheetDescriptor;
import org.openl.excel.parser.TableStyles;
import org.openl.rules.table.IGridRegion;
import org.openl.util.FileTool;
import org.openl.util.FileUtils;

public class SAXReader implements ExcelReader {

    private final ParserDateUtil parserDateUtil = new ParserDateUtil();

    private final String fileName;
    private File tempFile;

    private boolean use1904Windowing;
    private List<SAXSheetDescriptor> sheets;
    private MinimalStyleTable styleTable;

    public SAXReader(String fileName) {
        this.fileName = fileName;
        ExcelUtils.configureZipBombDetection();
    }

    public SAXReader(InputStream is) {
        // Save to temp file because using an InputStream has a higher memory footprint than using a File. See POI
        // javadocs.
        tempFile = FileTool.toTempFile(is, "stream.xlsx");
        this.fileName = tempFile.getAbsolutePath();
        ExcelUtils.configureZipBombDetection();
    }

    @Override
    public List<SAXSheetDescriptor> getSheets() {
        if (sheets == null) {
            try (ReadOnlyOPCPackage pkg = ReadOnlyOPCPackage.open(fileName)) {

                XMLReader parser = XMLHelper.newXMLReader();
                WorkbookHandler handler = new WorkbookHandler();
                parser.setContentHandler(handler);

                // process the first sheet
                XSSFReader r = new XSSFReader(pkg.pck);
                try (InputStream workbookData = r.getWorkbookData()) {
                    parser.parse(new InputSource(workbookData));
                }

                use1904Windowing = handler.isUse1904Windowing();

                sheets = handler.getSheetDescriptors();
            } catch (IOException | OpenXML4JException | SAXException | ParserConfigurationException e) {
                throw new ExcelParseException(e);
            }
        }

        return sheets;
    }

    @Override
    public Object[][] getCells(SheetDescriptor sheet) {
        SAXSheetDescriptor saxSheet = (SAXSheetDescriptor) sheet;
        try (ReadOnlyOPCPackage pkg = ReadOnlyOPCPackage.open(fileName)) {
            XSSFReader r = new XSSFReader(pkg.pck);

            initializeNeededData(r, pkg.pck);

            XMLReader parser = XMLHelper.newXMLReader();
            SheetHandler handler = new SheetHandler(r.getSharedStringsTable(),
                    use1904Windowing,
                    styleTable,
                    parserDateUtil);
            parser.setContentHandler(handler);

            try (InputStream sheetData = r.getSheet(saxSheet.getRelationId())) {
                parser.parse(new InputSource(sheetData));
            }

            CellAddress start = handler.getStart();
            saxSheet.setFirstRowNum(start.getRow());
            saxSheet.setFirstColNum(start.getColumn());

            return handler.getCells();
        } catch (IOException | OpenXML4JException | SAXException | ParserConfigurationException e) {
            throw new ExcelParseException(e);
        }
    }

    @Override
    public boolean isUse1904Windowing() {
        // Initialize use1904Windowing property if it's not initialized yet
        if (sheets == null) {
            getSheets();
        }

        return use1904Windowing;
    }

    @Override
    public TableStyles getTableStyles(SheetDescriptor sheet, IGridRegion tableRegion) {
        SAXSheetDescriptor saxSheet = (SAXSheetDescriptor) sheet;
        try (ReadOnlyOPCPackage pkg = ReadOnlyOPCPackage.open(fileName)) {

            XSSFReader r = new XSSFReader(pkg.pck);

            initializeNeededData(r, pkg.pck);

            XMLReader parser = XMLHelper.newXMLReader();
            StyleIndexHandler styleIndexHandler = new StyleIndexHandler(tableRegion, saxSheet.getIndex());
            parser.setContentHandler(styleIndexHandler);

            try (InputStream sheetData = r.getSheet(saxSheet.getRelationId())) {
                parser.parse(new InputSource(sheetData));
            }

            return new SAXTableStyles(tableRegion,
                    styleIndexHandler.getCellIndexes(),
                    r.getStylesTable(),
                    getSheetComments(pkg.pck, saxSheet),
                    styleIndexHandler.getFormulas());
        } catch (IOException | OpenXML4JException | SAXException | ParserConfigurationException e) {
            throw new ExcelParseException(e);
        }
    }

    @Override
    public void close() {
        styleTable = null;
        sheets = null;
        use1904Windowing = false;

        FileUtils.deleteQuietly(tempFile);
        tempFile = null;
        parserDateUtil.reset();
    }

    private void initializeNeededData(XSSFReader r, OPCPackage pkg) {
        // Ensure that needed settings were read from workbook and styles files
        if (sheets == null) {
            getSheets();
        }

        if (styleTable == null) {
            parseStyles(r, pkg);
        }
    }

    private void parseStyles(XSSFReader r, OPCPackage pkg) {
        List<PackagePart> parts = pkg.getPartsByContentType(XSSFRelation.STYLES.getContentType());
        if (parts.isEmpty()) {
            return;
        }

        try (InputStream stylesData = r.getStylesData()) {
            XMLReader styleParser = XMLHelper.newXMLReader();
            StyleHandler styleHandler = new StyleHandler();
            styleParser.setContentHandler(styleHandler);
            styleParser.parse(new InputSource(stylesData));
            styleTable = styleHandler.getStyleTable();
        } catch (IOException | OpenXML4JException | SAXException | ParserConfigurationException e) {
            throw new ExcelParseException(e);
        }
    }

    private CommentsTable getSheetComments(OPCPackage pkg, SAXSheetDescriptor sheet) {
        try {
            // Get workbook part
            PackageRelationship workbookRel = pkg.getRelationshipsByType(PackageRelationshipTypes.CORE_DOCUMENT)
                    .getRelationship(0);
            PackagePart workbookPart = pkg.getPart(workbookRel);

            // Find sheet part by relation id
            PackageRelationship sheetRel = workbookPart.getRelationship(sheet.getRelationId());
            PackagePart sheetPart = pkg.getPart(PackagingURIHelper.createPartName(sheetRel.getTargetURI()));

            PackageRelationshipCollection commentRelList = sheetPart
                    .getRelationshipsByType(XSSFRelation.SHEET_COMMENTS.getRelation());
            if (commentRelList.size() > 0) {
                // Comments have only one relationship
                PackageRelationship commentRel = commentRelList.getRelationship(0);
                PackagePart commentPart = pkg.getPart(PackagingURIHelper.createPartName(commentRel.getTargetURI()));

                return new CommentsTable(commentPart);
            }

            return null;
        } catch (InvalidFormatException | IOException e) {
            return null;
        }
    }

    private static class ReadOnlyOPCPackage implements AutoCloseable {
        final OPCPackage pck;

        static ReadOnlyOPCPackage open(String fileName) throws InvalidFormatException {
            OPCPackage pck = OPCPackage.open(fileName, PackageAccess.READ);
            return new ReadOnlyOPCPackage(pck);
        }

        private ReadOnlyOPCPackage(OPCPackage pck) {
            this.pck = pck;
        }

        @Override
        public void close() {
            // OPCPackage implementation makes SAVE on close() method. It is unacceptable for the READ only package.
            // Instead, it is required to call revert() for the READ only package.
            // On the other side it is easy to use try-with-resource to close a stream.
            // So to achieve it we wrap OPCPackage to call revert() on close().
            pck.revert();
        }
    }
}
