package org.openl.rules.lang.xls.types.meta;

import static org.openl.rules.datatype.binding.DatatypeTableBoundNode.getCellSource;

import java.util.Collections;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.openl.base.INamedThing;
import org.openl.binding.impl.NodeType;
import org.openl.binding.impl.SimpleNodeUsage;
import org.openl.exception.OpenLCompilationException;
import org.openl.meta.IMetaInfo;
import org.openl.rules.datatype.binding.DatatypeTableBoundNode;
import org.openl.rules.lang.xls.syntax.TableSyntaxNode;
import org.openl.rules.lang.xls.types.CellMetaInfo;
import org.openl.rules.lang.xls.types.DatatypeOpenClass;
import org.openl.rules.table.ICell;
import org.openl.rules.table.ILogicalTable;
import org.openl.rules.table.openl.GridCellSourceCodeModule;
import org.openl.syntax.impl.IdentifierNode;
import org.openl.syntax.impl.Tokenizer;
import org.openl.types.IOpenClass;
import org.openl.types.IOpenField;
import org.openl.types.java.JavaOpenClass;
import org.openl.util.ParserUtils;

public class DatatypeTableMetaInfoReader extends BaseMetaInfoReader<DatatypeTableBoundNode> {
    private static final Logger LOG = LoggerFactory.getLogger(DatatypeTableMetaInfoReader.class);

    public DatatypeTableMetaInfoReader(DatatypeTableBoundNode boundNode) {
        super(boundNode);
    }

    @Override
    protected TableSyntaxNode getTableSyntaxNode() {
        return getBoundNode().getTableSyntaxNode();
    }

    @Override
    protected CellMetaInfo getHeaderMetaInfo() {
        DatatypeOpenClass dataType = getBoundNode().getDataType();
        IdentifierNode identifier = getBoundNode().getParentClassIdentifier();
        if (identifier != null && dataType.getSuperClass() != null) {
            return createMetaInfo(identifier, dataType.getSuperClass().getMetaInfo());
        }
        return null;
    }

    @Override
    public CellMetaInfo getBodyMetaInfo(int row, int col) {
        ILogicalTable logicalTable = getBoundNode().getTable();

        ICell firstCell = logicalTable.getCell(0, 0);
        int r = row - firstCell.getAbsoluteRow();
        int c = col - firstCell.getAbsoluteColumn();
        if (r < 0 || c < 0) {
            return getHeaderMetaInfo();
        }
        if (!logicalTable.isNormalOrientation()) {
            int temp = r;
            r = c;
            c = temp;
        }
        if (c > 0) {
            if (c == 2) {
                // Default Values
                try {
                    ILogicalTable logicalRow = logicalTable.getRow(r);
                    IOpenField field = getField(logicalRow);
                    if (field == null) {
                        return null;
                    }
                    IOpenClass type = field.getType();
                    boolean multiValue = false;
                    if (type.getAggregateInfo().isAggregate(type)) {
                        type = type.getAggregateInfo().getComponentType(type);
                        multiValue = true;
                    }

                    return new CellMetaInfo(type, multiValue);
                } catch (OpenLCompilationException e) {
                    LOG.error(e.getMessage(), e);
                    return null;
                }
            }

            // Field names
            return null;
        }

        ILogicalTable logicalRow = logicalTable.getRow(r);
        GridCellSourceCodeModule typeCellSource = getCellSource(logicalRow, null, 0);
        if (!ParserUtils.isBlankOrCommented(typeCellSource.getCode())) {
            try {
                IOpenField field = getField(logicalRow);
                if (field == null) {
                    return null;
                }
                IMetaInfo fieldMetaInfo = field.getType().getMetaInfo();
                IdentifierNode[] idn = Tokenizer.tokenize(typeCellSource, "[]\n\r");
                return createMetaInfo(idn[0], fieldMetaInfo);
            } catch (OpenLCompilationException e) {
                LOG.error(e.getMessage(), e);
                return null;
            }
        }

        return null;
    }

    private IOpenField getField(ILogicalTable logicalRow) throws OpenLCompilationException {
        String fieldName = getName(logicalRow);
        if (fieldName == null) {
            return null;
        }

        DatatypeOpenClass dataType = getBoundNode().getDataType();
        IOpenField field = dataType.getField(fieldName);
        if (field == null) {
            return null;
        }
        return field;
    }

    private static CellMetaInfo createMetaInfo(IdentifierNode identifier, IMetaInfo typeMeta) {
        if (typeMeta == null) {
            return null;
        }
        SimpleNodeUsage nodeUsage = new SimpleNodeUsage(identifier,
                typeMeta.getDisplayName(INamedThing.SHORT),
                typeMeta.getSourceUrl(),
                NodeType.DATATYPE);

        return new CellMetaInfo(JavaOpenClass.STRING, false, Collections.singletonList(nodeUsage));
    }

    private static String getName(ILogicalTable row) throws OpenLCompilationException {
        GridCellSourceCodeModule nameCellSource = getCellSource(row, null, 1);
        IdentifierNode[] idn = Tokenizer.tokenize(nameCellSource, " \r\n");
        if (idn.length != 1) {
            // Table with error. Skip it
            return null;
        } else {
            String name = idn[0].getIdentifier();
            if (name.endsWith(DatatypeTableBoundNode.TRANSIENT_FIELD_SUFFIX) || name
                    .endsWith(DatatypeTableBoundNode.NON_TRANSIENT_FIELD_SUFFIX)) {
                return name.substring(0, name.length() - 1);
            }
            return name;
        }
    }
}
