package org.openl.rules.calc;

import java.util.Objects;

import org.openl.rules.binding.RecursiveSpreadsheetMethodPreBindingException;
import org.openl.types.IOpenClass;
import org.openl.types.IOpenField;
import org.openl.types.java.JavaOpenClass;
import org.openl.vm.IRuntimeEnv;

public class CustomSpreadsheetResultField extends ASpreadsheetField implements IOriginalDeclaredClassesOpenField {

    protected IOpenField field;
    private final IOpenClass[] declaringClasses;

    public CustomSpreadsheetResultField(CustomSpreadsheetResultOpenClass declaringClass, IOpenField field) {
        super(declaringClass, field.getName(), null);
        this.field = Objects.requireNonNull(field, "field cannot be null");
        this.declaringClasses = new IOpenClass[]{declaringClass};
    }

    public CustomSpreadsheetResultField(IOpenClass declaringClass, String name, IOpenClass type) {
        super(declaringClass, name, type);
        this.declaringClasses = new IOpenClass[]{declaringClass};
    }

    @Override
    public Object get(Object target, IRuntimeEnv env) {
        if (field != null) {
            throw new IllegalStateException("Spreadsheet cell type is not resolved at compile time");
        }
        if (target == null) {
            return getType().nullObject();
        }
        Object res = ((SpreadsheetResult) target).getFieldValue(getName());
        return processResult(res);
    }

    @Override
    public IOpenClass getType() {
        if (field != null) {
            // Lazy initialization for cells level recursive compilation
            try {
                setType(field.getType());
                field = null;
            } catch (RecursiveSpreadsheetMethodPreBindingException e) {
                return JavaOpenClass.OBJECT;
            }
        }
        return super.getType();
    }

    protected Object processResult(Object res) {
        return  ((CustomSpreadsheetResultOpenClass) getDeclaringClass()).getModule()
                .getObjectToDataOpenCastConvertor()
                .convert(res, getType());
    }

    @Override
    public boolean isWritable() {
        return true;
    }

    @Override
    public void set(Object target, Object value, IRuntimeEnv env) {
        if (target != null) {
            ((SpreadsheetResult) target).setFieldValue(getName(), value);
        }
    }

    @Override
    public IOpenClass[] getDeclaringClasses() {
        return declaringClasses;
    }

}