/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.model;

/**
 * A Factory for creating resources, bNodes and literals.
 * The created object should all properly implement the
 * equals(java.lang.Object) method.
 **/
public interface ValueFactory {

	/**
	 * Creates a new URI from the supplied string-representation.
	 *
	 * @param uri A string-representation of a URI.
	 * @return An object representing the URI.
	 * 
	 * @throws IlllegalArgumentException If the supplied string does not resolve
	 * to a legal (absolute) URI.
	 **/
	public URI createURI(String uri);
	
	/**
	 * Creates a new URI that will get the supplied namespace and local name.
	 *
	 * @param namespace A namespace.
	 * @param localName A legal local name. A legal local name adheres to the
	 * definition of an NCName as specified at
	 * <a href="http://www.w3.org/TR/REC-xml-names/#NT-NCName">
	 * http://www.w3.org/TR/REC-xml-names/#NT-NCName</a>.
	 * @throws IllegalArgumentException If the supplied namespace and localname
	 * do not resolve to a legal (absolute) URI.
	 **/
	public URI createURI(String namespace, String localName);

	/**
	 * Creates a new bNode.
	 *
	 * @return An object representing the bNode.
	 **/
	public BNode createBNode();
	
	/**
	 * creates a new bNode with the given node identifier.
	 * @param nodeId the bnode identifier
	 * @return an object representing the bnode.
	 */
	public BNode createBNode(String nodeId);

	/**
	 * Creates a new literal with the supplied value.
	 *
	 * @param value The literal's value.
	 **/
	public Literal createLiteral(String value);

	/**
	 * Creates a new literal with the supplied value and language attribute.
	 *
	 * @param value The literal's value.
	 * @param language The literal's language attribute, or <tt>null</tt> if the
	 * literal doesn't have a language.
	 **/
	public Literal createLiteral(String value, String language);

	/**
	 * Creates a new literal with the supplied value and datatype.
	 *
	 * @param value The literal's value.
	 * @param datatype The literal's datatype, or <tt>null</tt> if the literal
	 * doesn't have a datatype.
	 **/
	public Literal createLiteral(String value, URI datatype);
	
	/**
	 * Creates a new statement with the supplied subject, predicate and object.
	 * 
	 * @param subject The statement's subject.
	 * @param predicate The statement's predicate.
	 * @param object The statement's object.
	 * @return The created statement.
	 **/
	public Statement createStatement(Resource subject, URI predicate, Value object);
}
