/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.model.impl;

import java.io.Serializable;

import org.openrdf.model.GraphException;
import org.openrdf.model.Literal;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.sesame.sail.StatementIterator;

/**
 * An implementation of the Literal interface.
 **/
public class LiteralImpl implements Literal, Serializable {

/*-------------------------------------------+
| Variables                                  |
+-------------------------------------------*/

	/**
	 * The label.
	 **/
	private String _label;

	/**
	 * The language tag (null if not set).
	 **/
	private String _language;

	/**
	 * The datatype (null if not set).
	 **/
	private URI _datatype;

/*-------------------------------------------+
| Constructors                               |
+-------------------------------------------*/

	/**
	 * Creates a new Literal which will get the supplied label,
	 * but no language tag or datatype (=<tt>null</tt>).
	 *
	 * @param label The label for this literal.
	 **/
	public LiteralImpl(String label) {
		if (label == null) {
			throw new IllegalArgumentException("label must not be null");
		}
		_label = label;
	}

	/**
	 * Creates a new Literal which will get the supplied label and language tag.
	 *
	 * @param label The label for this literal.
	 * @param language The language tag of the supplied label, or null
	 * if not specified.
	 **/
	public LiteralImpl(String label, String language) {
		this(label);
		_language = (language == null) ? null : language.toLowerCase();
	}

	/**
	 * Creates a new Literal which will get the supplied label and datatype.
	 *
	 * @param label The label for this literal.
	 * @param datatype The datatype of the supplied label, or null
	 * if not specified.
	 **/
	public LiteralImpl(String label, URI datatype) {
		this(label);
		_datatype = datatype;
	}

/*-------------------------------------------+
| Methods                                    |
+-------------------------------------------*/

	// implements Literal.getLabel()
	public String getLabel() {
		return _label;
	}

	// implements Literal.getLanguage()
	public String getLanguage() {
		return _language;
	}

	// implements Literal.getDatatype()
	public URI getDatatype() {
		return _datatype;
	}

	// implements Literal.equals(Object)
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (o instanceof Literal) {
			Literal other = (Literal)o;

			// Compare labels
			if (!_label.equals(other.getLabel())) {
				return false;
			}

			// Compare datatypes
			if (_datatype == null) {
				if (other.getDatatype() != null) {
					return false;
				}
			}
			else {
				if (!_datatype.equals(other.getDatatype())) {
					return false;
				}
			}

			// Compare language tags
			if (_language == null) {
				if (other.getLanguage() != null) {
					return false;
				}
			}
			else {
				if (!_language.equals(other.getLanguage())) {
					return false;
				}
			}

			return true;
		}

		return false;
	}

	// implements Literal.hashCode()
	public int hashCode() {
		return _label.hashCode();
	}

	// Implements Comparable.compareTo(Object)
	public int compareTo(Object o) {
		if (this == o) {
			return 0;
		}

		if (o instanceof Literal) {
			Literal other = (Literal)o;

			int result = _label.compareTo(other.getLabel());

			if (result == 0) {
				// labels are equal, compare datatypes. Literals without
				// datatypes are less than literals that do have one.

				if (_datatype == null) {
					if (other.getDatatype() != null) {
						result = -1;
					}
				}
				else if (other.getDatatype() == null) {
					result = 1;
				}
				else {
					result = _datatype.compareTo(other.getDatatype());
				}
			}

			if (result == 0) {
				// labels and datatypes are equal, compare languages. Literals
				// without languages are less than literals that do have one.

				if (_language == null) {
					if (other.getLanguage() != null) {
						result = -1;
					}
				}
				else if (other.getLanguage() == null) {
					result = 1;
				}
				else {
					result = _language.compareTo(other.getLanguage());
				}
			}

			return result;
		}

		// Force a ClassCastException if o is not a Value
		Value other = (Value)o;

		// Literal > URI and Literal > BNode
		return 1;
	}

	/**
	 * Returns the label of the literal.
	 **/
	public String toString() {
		return _label;
	}

	// Implements Value.getObjectStatements()
	public StatementIterator getObjectStatements()
		throws GraphException
	{
		throw new GraphException("no backing store associated");
	}
}
