/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.rio;

/**
 * An interface defining methods for receiving warning and error messages
 * from an RDF parser.
 **/
public interface ParseErrorListener {

	/**
	 * Reports a warning from the parser. Warning messages are generated
	 * by the parser when it encounters data that is syntactically correct
	 * but which is likely to be a typo. Examples are the use of unknown
	 * or deprecated RDF URIs, e.g. <tt>rdfs:Property</tt> instead of
	 * <tt>rdf:Property</tt>.
	 *
	 * @param msg A warning message.
	 * @param lineNo A line number related to the warning, or -1 if not
	 * available or applicable.
	 * @param colNo A column number related to the warning, or -1 if not
	 * available or applicable.
	 **/
	public void warning(String msg, int lineNo, int colNo);

	/**
	 * Reports an error from the parser. Error messages are generated by
	 * the parser when it encounters an error in the RDF document. The
	 * parser will try its best to recover from the error and continue
	 * parsing when <tt>stopAtFirstError</tt> has been set to
	 * <tt>false</tt>.
	 *
	 * @param msg A error message.
	 * @param lineNo A line number related to the error, or -1 if not
	 * available or applicable.
	 * @param colNo A column number related to the error, or -1 if not
	 * available or applicable.
	 * @see org.openrdf.rio.Parser#setStopAtFirstError
	 **/
	public void error(String msg, int lineNo, int colNo);

	/**
	 * Reports a fatal error from the parser. A fatal error is an error
	 * of which the RDF parser cannot recover. The parser will stop parsing
	 * directly after it reported the fatal error. Example fatal errors are
	 * unbalanced start- and end-tags in an XML-encoded RDF document.
	 *
	 * @param msg A error message.
	 * @param lineNo A line number related to the error, or -1 if not
	 * available or applicable.
	 * @param colNo A column number related to the error, or -1 if not
	 * available or applicable.
	 **/
	public void fatalError(String msg, int lineNo, int colNo);
}
