/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.rio.ntriples;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.rio.RdfDocumentWriter;

/**
 * An implementation of the RdfDocumentWriter interface that writes RDF
 * documents in N-Triples format. The N-Triples format is defined in
 * <a href="http://www.w3.org/TR/rdf-testcases/#ntriples">this
 * section</a> of the RDF Test Cases document.
 **/
public class NTriplesWriter implements RdfDocumentWriter {

/*---------------------------------+
| Variables                        |
+---------------------------------*/

	private Writer _out;

	private boolean _writingStarted;

/*---------------------------------+
| Constructors                     |
+---------------------------------*/

	/**
	 * Creates a new NTriplesWriter that will write to the
	 * supplied OutputStream.
	 *
	 * @param out The OutputStream to write the N-Triples document to.
	 **/
	public NTriplesWriter(OutputStream out) {
		this( new OutputStreamWriter(out) );
	}

	/**
	 * Creates a new NTriplesWriter that will write to the
	 * supplied Writer.
	 *
	 * @param out The Writer to write the N-Triples document to.
	 **/
	public NTriplesWriter(Writer out) {
		_out = out;
		_writingStarted = false;
	}

/*---------------------------------+
| Methods from interface RdfWriter |
+---------------------------------*/

	public void setNamespace(String prefix, String name) {
		// N-Triples does not support namespace prefixes (yet).
	}

	public void startDocument()
		throws IOException
	{
		if (_writingStarted) {
			throw new RuntimeException("Document writing has already started");
		}

		_writingStarted = true;
	}

	public void endDocument()
		throws IOException
	{
		if (!_writingStarted) {
			throw new RuntimeException("Document writing has not yet started");
		}

		try {
			_out.flush();
		}
		finally {
			_writingStarted = false;
		}
	}

	public void writeStatement(Resource subj, URI pred, Value obj)
		throws IOException
	{
		if (!_writingStarted) {
			throw new RuntimeException("Document writing has not yet been started");
		}

		// SUBJECT
		_writeResource(subj);
		_out.write(" ");

		// PREDICATE
		_writeURI(pred);
		_out.write(" ");

		// OBJECT
		if (obj instanceof Resource) {
			_writeResource( (Resource)obj );
		}
		else if (obj instanceof Literal) {
			_writeLiteral( (Literal)obj );
		}

		_out.write(" .");
		_writeNewLine();
	}

	public void writeComment(String comment)
		throws IOException
	{
		_out.write("# ");
		_out.write(comment);
		_writeNewLine();
	}

/*---------------------------------+
| Other methods                    |
+---------------------------------*/

	private void _writeResource(Resource res)
		throws IOException
	{
		if (res instanceof BNode) {
			_writeBNode((BNode)res);
		}
		else {
			_writeURI((URI)res);
		}
	}

	private void _writeURI(URI uri)
		throws IOException
	{
		_out.write(NTriplesUtil.toNTriplesString(uri));
	}

	private void _writeBNode(BNode bNode)
		throws IOException
	{
		_out.write(NTriplesUtil.toNTriplesString(bNode));
	}

	private void _writeLiteral(Literal lit)
		throws IOException
	{
		_out.write(NTriplesUtil.toNTriplesString(lit));
	}

	private void _writeNewLine()
		throws IOException
	{
		_out.write("\n");
	}
}
