/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact:
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.rio.rdfxml;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;

import org.openrdf.util.xml.XMLReaderFactory;
import org.openrdf.util.xml.XmlDatatypeUtil;
import org.openrdf.util.xml.XmlUtil;
import org.openrdf.vocabulary.RDF;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.ValueFactoryImpl;

import org.openrdf.rio.NamespaceListener;
import org.openrdf.rio.ParseErrorListener;
import org.openrdf.rio.ParseException;
import org.openrdf.rio.ParseLocationListener;
import org.openrdf.rio.StatementHandler;
import org.openrdf.rio.StatementHandlerException;

/**
 * A parser for XML-serialized RDF. This parser operates directly
 * on the SAX events generated by a SAX-enabled XML parser. The XML parser
 * should be compliant with SAX2. You should specify which SAX parser should
 * be used by setting the <code>org.xml.sax.driver</code> property.
 * This parser is not thread-safe, therefore it's public methods are
 * synchronized.
 * <p>
 * To parse a document using this parser:
 * <ul>
 *   <li>Create an instance of RdfXmlParser, optionally supplying it with your
 *   own ValueFactory.</li>
 *   <li>Set the StatementHandler.</li>
 *   <li>Optionally, set the ParseErrorListener, ParseLocationListener and/or
 *   NamespaceListener.</li>
 *   <li>Optionally, specify whether the parser should verify the data it
 *   parses and whether it should stop immediately when it finds an error in
 *   the data (both default to <tt>true</tt>).
 *   <li>Call the parse method.</li>
 * </ul>
 * Example code:
 * <pre>
 * // Use the SAX2-compliant Xerces parser:
 * System.setProperty(
 *     "org.xml.sax.driver",
 *     "org.apache.xerces.parsers.SAXParser");
 *
 * Parser parser = new RdfXmlParser();
 * parser.setStatementHandler(myStatementHandler);
 * parser.setParseErrorListener(myParseErrorListener);
 * parser.setVerifyData(true);
 * parser.stopAtFirstError(false);
 *
 * // Parse the data from inputStream, resolving any
 * // relative URIs against http://foo/bar:
 * parser.parse(inputStream, "http://foo/bar");
 * </pre>
 *
 * @see org.openrdf.model.ValueFactory
 * @see org.openrdf.rio.StatementHandler
 * @see org.openrdf.rio.ParseErrorListener
 * @see org.openrdf.rio.ParseLocationListener
 * @see org.openrdf.rio.NamespaceListener
 **/
public class RdfXmlParser implements org.openrdf.rio.Parser {

/*------------------------------------------------------+
| Frequently used resources                             |
+------------------------------------------------------*/

	/** The rdf:type resource. **/
	private URI RDF_TYPE;

	/** The rdf:subject resource. **/
	private URI RDF_SUBJECT;

	/** The rdf:predicate resource. **/
	private URI RDF_PREDICATE;

	/** The rdf:object resource. **/
	private URI RDF_OBJECT;

	/** The rdf:Statement resource. **/
	private URI RDF_STATEMENT;

	/** The rdf:li resource. **/
	private URI RDF_LI;

	/** The rdf:first resource. **/
	private URI RDF_FIRST;

	/** The rdf:rest resource. **/
	private URI RDF_REST;

	/** The rdf:nil resource. **/
	private URI RDF_NIL;

/*------------------------------------------------------+
| Variables                                             |
+------------------------------------------------------*/

	/**
	 * A filter filtering calls to SAX methods specifically for this parser.
	 **/
	private SAXFilter _saxFilter;

	/**
	 * A factory for creating resources, bNodes and literals.
	 **/
	private ValueFactory _valueFactory;

	/**
	 * Mapping from bNode ID's as used in the RDF document to the
	 * object created for it by the ValueFactory.
	 **/
	private Map _bNodeIdMap;

	/**
	 * The object to report statements to.
	 **/
	private StatementHandler _statementHandler;

	/**
	 * The object to report parse errors to.
	 **/
	private ParseErrorListener _errorListener;

	/**
	 * The base URI for resolving relative URIs. This variable is set/modified
	 * by the SAXFilter during parsing such that it always represents the URI
	 * of the context in which elements are reported.
	 **/
	private org.openrdf.util.uri.URI _baseURI;

	/**
	 * The base URI of the document. This variable is set when
	 * <tt>parse(inputStream, baseURI)</tt> is called and will not be changed
	 * during parsing.
	 **/
	private String _documentURI;

	/**
	 * The language of literal values as can be specified using xml:lang
	 * attributes. This variable is set/modified by the SAXFilter during
	 * parsing such that it always represents the language of the context
	 * in which elements are reported.
	 **/
	private String _xmlLang;

	/**
	 * A stack of node- and property elements.
	 **/
	private Stack _elementStack = new Stack();

	/**
	 * A set containing URIs that have been generated as a result of rdf:ID
	 * attributes. These URIs should be unique within a single document.
	 **/
	private Set _usedIDs = new HashSet();

	/**
	 * Flag indicating whether the parser should check the data it parses.
	 **/
	boolean _verifyData = true;

	/**
	 * Flag indicating whether the parser should preserve bnode identifiers specified
	 * in the source.
	 */
	boolean _preserveBNodeIds = false;

	/**
	 * Indicates how datatyped literals should be handled. Legal
	 * values are <tt>DT_IGNORE</tt>, <tt>DT_VERIFY</tt> and
	 * <tt>DT_NORMALIZE</tt>.
	 **/
	private int _datatypeHandling;

	/**
	 * Flag indicating whether the parser should stop parsing when it finds
	 * an error in the data.
	 **/
	boolean _stopAtFirstError = true;

/*------------------------------------------------------+
| Constructors                                          |
+------------------------------------------------------*/

	/**
	 * Creates a new RdfXmlParser that will use a <tt>ValueFactoryImpl</tt> to
	 * create objects for resources, bNodes and literals.
	 * @see org.openrdf.model.impl.ValueFactoryImpl
	 **/
	public RdfXmlParser() {
		this(new ValueFactoryImpl());
	}

	/**
	 * Creates a new RdfXmlParser that will use the supplied ValueFactory to
	 * create objects for resources, bNodes and literals.
	 *
	 * @param valueFactory A ValueFactory.
	 **/
	public RdfXmlParser(ValueFactory valueFactory) {
		_valueFactory = valueFactory;

		_bNodeIdMap = new HashMap();
		_datatypeHandling = DT_VERIFY;

		RDF_TYPE = _valueFactory.createURI(RDF.TYPE);
		RDF_SUBJECT = _valueFactory.createURI(RDF.SUBJECT);
		RDF_PREDICATE = _valueFactory.createURI(RDF.PREDICATE);
		RDF_OBJECT = _valueFactory.createURI(RDF.OBJECT);
		RDF_STATEMENT = _valueFactory.createURI(RDF.STATEMENT);
		RDF_LI = _valueFactory.createURI(RDF.LI);
		RDF_FIRST = _valueFactory.createURI(RDF.FIRST);
		RDF_REST = _valueFactory.createURI(RDF.REST);
		RDF_NIL = _valueFactory.createURI(RDF.NIL);

		// SAXFilter does some filtering and verifying of SAX events
		_saxFilter = new SAXFilter(this);
	}

/*------------------------------------------------------+
| Methods from interface Parser                         |
+------------------------------------------------------*/

	// implements Parser.setStatementHandler(StatementHandler)
	public synchronized void setStatementHandler(StatementHandler sh) {
		_statementHandler = sh;
	}

	// implements Parser.setParseErrorListener(ParseErrorListener)
	public synchronized void setParseErrorListener(ParseErrorListener el) {
		_errorListener = el;
	}

	// implements Parser.setParseLocationListener(ParseLocationListener)
	public synchronized void setParseLocationListener(ParseLocationListener ll) {
		_saxFilter.setParseLocationListener(ll);
	}

	// implements Parser.setNamespaceListener(NamespaceListener)
	public synchronized void setNamespaceListener(NamespaceListener nl) {
		_saxFilter.setNamespaceListener(nl);
	}

	// implements Parser.setVerifyData(boolean)
	public synchronized void setVerifyData(boolean verifyData) {
		_verifyData = verifyData;
	}

	// implements Parser.setPreserveBNodeIds(boolean)
	public synchronized void setPreserveBNodeIds(boolean preserveBNodeIds) {
		_preserveBNodeIds = preserveBNodeIds;
	}

	// implements Parser.setStopAtFirstError(boolean)
	public synchronized void setStopAtFirstError(boolean stopAtFirstError) {
		_stopAtFirstError = stopAtFirstError;
	}

	// implements Parser.setDatatypeHandling(int)
	public void setDatatypeHandling(int datatypeHandling) {
		_datatypeHandling = datatypeHandling;
	}

	/**
	 * Sets the parser in a mode to parse stand-alone RDF documents. In
	 * stand-alone RDF documents, the enclosing <tt>rdf:RDF</tt> root element is
	 * optional if this root element contains just one element (e.g.
	 * <tt>rdf:Description</tt>.
	 **/
	public void setParseStandAloneDocuments(boolean standAloneDocs) {
		_saxFilter.setParseStandAloneDocuments(standAloneDocs);
	}

	/**
	 * Returns whether the parser is currently in a mode to parse stand-alone
	 * RDF documents.
	 *
	 * @see #setParseStandAloneDocuments
	 **/
	public boolean getParseStandAloneDocuments() {
		return _saxFilter.getParseStandAloneDocuments();
	}

	/**
	 * Parses the data from the supplied InputStream, using the supplied
	 * baseURI to resolve any relative URI references.
	 *
	 * @param in The InputStream from which to read the data.
	 * @param baseURI The URI associated with the data in the InputStream.
	 * @exception IOException If an I/O error occurred while data was read
	 * from the InputStream.
	 * @exception ParseException If the parser has found an unrecoverable
	 * parse error.
	 * @exception StatementHandler If the configured statement handler
	 * encountered an unrecoverable error.
	 * @exception IllegalArgumentException If the supplied input stream or
	 * base URI is <tt>null</tt>.
	 **/
	public synchronized void parse(InputStream in, String baseURI)
		throws IOException, ParseException, StatementHandlerException
	{
		if (in == null) {
			throw new IllegalArgumentException("Input stream cannot be 'null'");
		}
		if (baseURI == null) {
			throw new IllegalArgumentException("Base URI cannot be 'null'");
		}

		InputSource inputSource = new InputSource(in);
		inputSource.setSystemId(baseURI);

		_parse(inputSource);
	}

	/**
	 * Parses the data from the supplied Reader, using the supplied baseURI
	 * to resolve any relative URI references.
	 *
	 * @param reader The Reader from which to read the data.
	 * @param baseURI The URI associated with the data in the InputStream.
	 * @exception IOException If an I/O error occurred while data was read
	 * from the InputStream.
	 * @exception ParseException If the parser has found an unrecoverable
	 * parse error.
	 * @exception StatementHandlerException If the configured statement handler
	 * has encountered an unrecoverable error.
	 * @exception IllegalArgumentException If the supplied reader or base URI
	 * is <tt>null</tt>.
	 **/
	public synchronized void parse(Reader reader, String baseURI)
		throws IOException, ParseException, StatementHandlerException
	{
		if (reader == null) {
			throw new IllegalArgumentException("Reader cannot be 'null'");
		}
		if (baseURI == null) {
			throw new IllegalArgumentException("Base URI cannot be 'null'");
		}

		InputSource inputSource = new InputSource(reader);
		inputSource.setSystemId(baseURI);

		_parse(inputSource);
	}

	private void _parse(InputSource inputSource)
		throws IOException, ParseException, StatementHandlerException
	{
		try {
			_documentURI = inputSource.getSystemId();

			//_saxFilter.clear();
			_saxFilter.setDocumentURI(_documentURI);

			XMLReader xmlReader = XMLReaderFactory.createXMLReader();
			xmlReader.setContentHandler(_saxFilter);

			xmlReader.parse(inputSource);
		}
		catch (SAXParseException e) {
			Exception wrappedExc = e.getException();
			if (wrappedExc == null) {
				wrappedExc = e;
			}
			throw new ParseException(wrappedExc, e.getLineNumber(), e.getColumnNumber());
		}
		catch (SAXException e) {
			Exception wrappedExc = e.getException();
			if (wrappedExc == null) {
				wrappedExc = e;
			}
			if (wrappedExc instanceof StatementHandlerException) {
				throw (StatementHandlerException)wrappedExc;
			}
			else {
				throw new ParseException(wrappedExc, -1, -1);
			}
		}
		finally {
			// Clean up
			_saxFilter.clear();
			_baseURI = null;
			_xmlLang = null;
			_elementStack.clear();
			_usedIDs.clear();
			_bNodeIdMap.clear();
		}
	}

/*------------------------------------------------------+
| Methods called by SAXFilter                           |
+------------------------------------------------------*/

	void setBaseURI(org.openrdf.util.uri.URI baseURI) {
		_baseURI = baseURI;
	}

	void setXmlLang(String xmlLang) {
		if ("".equals(xmlLang)) {
			_xmlLang = null;
		}
		else {
			_xmlLang = xmlLang;
		}
	}

	void startElement(String namespaceURI, String localName, String qName, Atts atts)
		throws SAXException
	{
		if (_topIsProperty()) {
			// this element represents the subject and/or object of a statement
			_processNodeElt(namespaceURI, localName, qName, atts, false);
		}
		else {
			// this element represents a property
			_processPropertyElt(namespaceURI, localName, qName, atts, false);
		}
	}

	void endElement(String namespaceURI, String localName, String qName)
		throws SAXException
	{
		Object topElement = _peekStack(0);

		if (topElement instanceof NodeElement) {
			// Check if top node is 'volatile', meaning that it doesn't have a
			// start- and end element associated with it.
			if ( ((NodeElement)topElement).isVolatile() ) {
				_elementStack.pop();
			}
		}
		else {
			// topElement instanceof PropertyElement
			PropertyElement predicate = (PropertyElement)topElement;

			if (predicate.parseCollection()) {
				Resource lastListResource = predicate.getLastListResource();

				if (lastListResource == null) {
					// no last list resource, list must have been empty.
					NodeElement subject = (NodeElement)_peekStack(1);

					_reportStatement(
							subject.getResource(), predicate.getURI(), RDF_NIL);

					_handleReification(RDF_NIL);
				}
				else {
					// Generate the final tail of the list.
					_reportStatement(lastListResource, RDF_REST, RDF_NIL);
				}
			}

		}

		_elementStack.pop();
	}

	void emptyElement(String namespaceURI, String localName, String qName, Atts atts)
		throws SAXException
	{
		if (_topIsProperty()) {
			// this element represents the subject and/or object of a statement
			_processNodeElt(namespaceURI, localName, qName, atts, true);
		}
		else {
			// this element represents a property
			_processPropertyElt(namespaceURI, localName, qName, atts, true);
		}
	}

	void text(String text)
		throws SAXException
	{
		if (!_topIsProperty()) {
			throw new SAXException("unexpected literal");
		}

		PropertyElement propEl = (PropertyElement)_peekStack(0);
		String datatype = propEl.getDatatype();

		Literal lit = _createLiteral(text, _xmlLang, datatype);

		NodeElement subject = (NodeElement)_peekStack(1);
		PropertyElement predicate = (PropertyElement)_peekStack(0);

		_reportStatement(subject.getResource(), predicate.getURI(), lit);

		_handleReification(lit);
	}

/*------------------------------------------------------+
| RDF processing methods                                |
+------------------------------------------------------*/

	/* Process a node element (can be both subject and object) */
	private void _processNodeElt(String namespaceURI, String localName, String qName, Atts atts, boolean isEmptyElt)
		throws SAXException
	{
		if (_verifyData) {
			// Check the element name
			_checkNodeEltName(namespaceURI, localName, qName);
		}

		Resource nodeResource = _getNodeResource(atts);
		NodeElement nodeElement = new NodeElement(nodeResource);

		if (!_elementStack.isEmpty()) {
			// node can be object of a statement, or part of an rdf:List
			NodeElement subject = (NodeElement)_peekStack(1);
			PropertyElement predicate = (PropertyElement)_peekStack(0);

			if (predicate.parseCollection()) {
				Resource lastListRes = predicate.getLastListResource();
				BNode newListRes = _createBNode();

				if (lastListRes == null) {
					// first element in the list
					_reportStatement(subject.getResource(),
							predicate.getURI(), newListRes);

					_handleReification(newListRes);
				}
				else {
					// not the first element in the list
					_reportStatement(lastListRes, RDF_REST, newListRes);
				}

				_reportStatement(newListRes, RDF_FIRST, nodeResource);

				predicate.setLastListResource(newListRes);
			}
			else {
				_reportStatement( subject.getResource(),
						predicate.getURI(), nodeResource);

				_handleReification(nodeResource);
			}
		}

		if (!localName.equals("Description") ||
			!namespaceURI.equals(RDF.NAMESPACE))
		{
			// element name is uri's type
			URI className = null;
			if ("".equals(namespaceURI)) {
				// No namespace, use base URI
				className = _buildResourceFromLocalName(localName);
			}
			else {
				className = _createURI( namespaceURI + localName );
			}
			_reportStatement(nodeResource, RDF_TYPE, className);
		}

		Att type = atts.removeAtt(RDF.NAMESPACE, "type");
		if (type != null) {
			// rdf:type attribute, value is a URI-reference
			URI className = _buildURIFromReference(type.getValue());

			_reportStatement(nodeResource, RDF_TYPE, className);
		}

		if (_verifyData) {
			_checkRdfAtts(atts);
		}

		_processSubjectAtts(nodeElement, atts);

		if (!isEmptyElt) {
			_elementStack.push(nodeElement);
		}
	}

	/**
	 * Retrieves the resource of a node element (subject or object) using
	 * relevant attributes (rdf:ID, rdf:about and rdf:nodeID) from its
	 * attributes list.
	 *
	 * @return a resource or a bNode.
	 **/
	private Resource _getNodeResource(Atts atts)
		throws SAXException
	{
		Att id = atts.removeAtt(RDF.NAMESPACE, "ID");
		Att about = atts.removeAtt(RDF.NAMESPACE, "about");
		Att nodeID = atts.removeAtt(RDF.NAMESPACE, "nodeID");

		if (_verifyData) {
			int definedAttsCount = 0;

			if (id     != null) { definedAttsCount++; }
			if (about  != null) { definedAttsCount++; }
			if (nodeID != null) { definedAttsCount++; }

			if (definedAttsCount > 1) {
				sendError("Only one of the attributes rdf:ID, rdf:about or rdf:nodeID can be used here");
			}
		}

		Resource result = null;

		if (id != null) {
			result = _buildURIFromID(id.getValue());
		}
		else if (about != null) {
			result = _buildURIFromReference(about.getValue());
		}
		else if (nodeID != null) {
			result = _createBNode(nodeID.getValue());
		}
		else {
			// No resource specified, generate a bNode
			result = _createBNode();
		}

		return result;
	}

	/** processes subject attributes. **/
	private void _processSubjectAtts(NodeElement nodeElt, Atts atts)
		throws SAXException
	{
		Resource subject = nodeElt.getResource();

		Iterator iter = atts.iterator();

		while (iter.hasNext()) {
			Att att = (Att)iter.next();

			URI predicate = _createURI( att.getURI() );
			Literal lit = _createLiteral(att.getValue(), _xmlLang, null);

			_reportStatement(subject, predicate, lit);
		}
	}

	private void _processPropertyElt(String namespaceURI, String localName, String qName, Atts atts, boolean isEmptyElt)
		throws SAXException
	{
		if (_verifyData) {
			_checkPropertyEltName(namespaceURI, localName, qName);
		}

		// Get the URI of the property
		URI propURI = null;
		if (namespaceURI.equals("")) {
			// no namespace URI
			sendError("unqualified property element <" + qName + "> not allowed");
			// Use base URI as namespace:
			propURI = _buildResourceFromLocalName(localName);
		}
		else {
			propURI = _createURI( namespaceURI + localName );
		}

		// List expansion rule
		if (propURI.equals(RDF_LI)) {
			NodeElement subject = (NodeElement)_peekStack(0);
			propURI = _createURI(
					RDF.NAMESPACE + "_" + subject.getNextLiCounter() );
		}

		// Push the property on the stack.
		PropertyElement predicate = new PropertyElement(propURI);
		_elementStack.push(predicate);

		// Check if property has a reification ID
		Att id = atts.removeAtt(RDF.NAMESPACE, "ID");
		if (id != null) {
			URI reifURI = _buildURIFromID(id.getValue());
			predicate.setReificationURI(reifURI);
		}

		// Check for presence of rdf:parseType attribute
		Att parseType = atts.removeAtt(RDF.NAMESPACE, "parseType");

		if (parseType != null) {
			if (_verifyData) {
				_checkNoMoreAtts(atts);
			}

			String parseTypeValue = parseType.getValue();

			if (parseTypeValue.equals("Resource")) {
				BNode objectResource = _createBNode();
				NodeElement subject = (NodeElement)_peekStack(1);

				_reportStatement(
						subject.getResource(), propURI, objectResource);

				if (isEmptyElt) {
					_handleReification(objectResource);
				}
				else {
					NodeElement object = new NodeElement(objectResource);
					object.setIsVolatile(true);
					_elementStack.push(object);
				}
			}
			else if (parseTypeValue.equals("Collection")) {
				if (isEmptyElt) {
					NodeElement subject = (NodeElement)_peekStack(1);
					_reportStatement(subject.getResource(), propURI, RDF_NIL);
					_handleReification(RDF_NIL);
				}
				else {
					predicate.setParseCollection(true);
				}
			}
			else {
				// other parseType
				if (!parseTypeValue.equals("Literal")) {
					sendWarning("unknown parseType: " + parseType.getValue());
				}

				if (isEmptyElt) {
					NodeElement subject = (NodeElement)_peekStack(1);

					Literal lit = _createLiteral("", null, RDF.XMLLITERAL);

					_reportStatement(subject.getResource(), propURI, lit);

					_handleReification(lit);
				}
				else {
					// The next string is an rdf:XMLLiteral
					predicate.setDatatype(RDF.XMLLITERAL);

					_saxFilter.setParseLiteralMode();
				}
			}
		}
		// parseType == null
		else if (isEmptyElt) {
			// empty element without an rdf:parseType attribute

			// Note: we handle rdf:datatype attributes here to allow datatyped
			// empty strings in documents. The current spec does have a
			// production rule that matches this, which is likely to be an
			// omission on its part.
			Att datatype = atts.getAtt(RDF.NAMESPACE, "datatype");

			if (atts.size() == 0 || atts.size() == 1 && datatype != null) {
				// element had no attributes, or only the optional
				// rdf:ID and/or rdf:datatype attributes.
				NodeElement subject = (NodeElement)_peekStack(1);

				Literal lit = (datatype == null) ?
					_createLiteral("", _xmlLang, null) :
					_createLiteral("", null, datatype.getValue());

				_reportStatement(subject.getResource(), propURI, lit);
				_handleReification(lit);
			}
			else {
				// Create resource for the statement's object.
				Resource resourceRes = _getPropertyResource(atts);

				// All special rdf attributes have been checked/removed.
				if (_verifyData) {
					_checkRdfAtts(atts);
				}

				NodeElement resourceElt = new NodeElement(resourceRes);
				NodeElement subject = (NodeElement)_peekStack(1);

				_reportStatement(subject.getResource(), propURI, resourceRes);
				_handleReification(resourceRes);

				Att type = atts.removeAtt(RDF.NAMESPACE, "type");
				if (type != null) {
					// rdf:type attribute, value is a URI-reference
					URI className = _buildURIFromReference(type.getValue());

					_reportStatement(resourceRes, RDF_TYPE, className);
				}

				_processSubjectAtts(resourceElt, atts);
			}
		}
		else {
			// Not an empty element, sub elements will follow.

			// Check for rdf:datatype attribute
			Att datatype = atts.removeAtt(RDF.NAMESPACE, "datatype");
			if (datatype != null) {
				predicate.setDatatype(datatype.getValue());
			}

			// No more attributes are expected.
			if (_verifyData) {
				_checkNoMoreAtts(atts);
			}
		}
		
		if (isEmptyElt) {
			// Empty element has been pushed on the stack
			// at the start of this method, remove it.
			_elementStack.pop();
		}
	}

	/**
	 * Retrieves the object resource of a property element using relevant attributes
	 * (rdf:resource and rdf:nodeID) from its attributes list.
	 *
	 * @return a resource or a bNode.
	 **/
	private Resource _getPropertyResource(Atts atts)
		throws SAXException
	{
		Att resource = atts.removeAtt(RDF.NAMESPACE, "resource");
		Att nodeID = atts.removeAtt(RDF.NAMESPACE, "nodeID");

		if (_verifyData) {
			int definedAttsCount = 0;

			if (resource != null) { definedAttsCount++; }
			if (nodeID   != null) { definedAttsCount++; }

			if (definedAttsCount > 1) {
				sendError("Only one of the attributes rdf:resource or rdf:nodeID can be used here");
			}
		}

		Resource result = null;

		if (resource != null) {
			result = _buildURIFromReference(resource.getValue());
		}
		else if (nodeID != null) {
			result = _createBNode(nodeID.getValue());
		}
		else {
			// No resource specified, generate a bNode
			result = _createBNode();
		}

		return result;
	}

	/*
	 * Processes any rdf:ID attributes that generate reified statements. This
	 * method assumes that a PropertyElement (which can have an rdf:ID
	 * attribute) is on top of the stack, and a NodeElement is below that.
	 */
	private void _handleReification(Value value)
		throws SAXException
	{
		PropertyElement predicate = (PropertyElement)_peekStack(0);

		if (predicate.isReified()) {
			NodeElement subject = (NodeElement)_peekStack(1);
			URI reifRes = predicate.getReificationURI();
			_reifyStatement(reifRes, subject.getResource(), predicate.getURI(), value);
		}
	}

	private void _reifyStatement(Resource reifNode, Resource subj, URI pred, Value obj)
		throws SAXException
	{
		_reportStatement(reifNode, RDF_TYPE, RDF_STATEMENT);
		_reportStatement(reifNode, RDF_SUBJECT, subj);
		_reportStatement(reifNode, RDF_PREDICATE, pred);
		_reportStatement(reifNode, RDF_OBJECT, obj);
	}

	/**
	 * Builds a Resource from a non-qualified localname.
	 **/
	private URI _buildResourceFromLocalName(String localName)
		throws SAXException
	{
		// Resolve the relative URI against the base URI
		String uriString = _baseURI.resolve("#" + localName).toString();
		return _createURI(uriString);
	}

	/**
	 * Builds a Resource from the value of an rdf:ID attribute.
	 **/
	private URI _buildURIFromID(String id)
		throws SAXException
	{
		if (_verifyData) {
			// Check if 'id' is a legal NCName
			if (!XmlUtil.isNCName(id)) {
				sendError("Not an XML Name: " + id);
			}
		}

		// Resolve the relative URI against the base URI
		String uriString = _baseURI.resolve("#" + id).toString();

		if (_verifyData) {
			// uriString should be unique in the current document

			if (!_usedIDs.add(uriString)) {
				// uriString was not added because the set already contained
				// an equal string.
				sendError("ID '" + id + "' has already been defined");
			}
		}

		return _createURI(uriString);
	}

	private URI _buildURIFromReference(String uriReference)
		throws SAXException
	{
		org.openrdf.util.uri.URI relUri = new org.openrdf.util.uri.URI(uriReference);

		if (_verifyData) {
			if (relUri.isRelative() && !relUri.isSelfReference() && // Relative URI that is not a self-reference
				_baseURI.isOpaque())
			{
				sendError("Relative URI '" + uriReference +
						"' cannot be resolved using the opaque base URI '" + _baseURI + "'");
			}
		}

		String uriString = _baseURI.resolve(relUri).toString();
		return _createURI(uriString);
	}

	private URI _createURI(String uri)
		throws SAXException
	{
		try {
			return _valueFactory.createURI(uri);
		}
		catch (Exception e) {
			throw new SAXException(e);
		}
	}

	private BNode _createBNode()
		throws SAXException
	{
		try {
			return _valueFactory.createBNode();
		}
		catch (Exception e) {
			throw new SAXException(e);
		}
	}

	private BNode _createBNode(String nodeID)
		throws SAXException
	{
		if (_verifyData) {
			// Check if 'nodeID' is a legal NCName
			if (!XmlUtil.isNCName(nodeID)) {
				sendError("Not an XML Name: " + nodeID);
			}
		}

		// Maybe the node ID has been used before:
		BNode result = (BNode)_bNodeIdMap.get(nodeID);

		if (result == null) {
			// This is a new node ID, create a new BNode object for it
			try {
				if (_preserveBNodeIds) {
					result = _valueFactory.createBNode(nodeID);
				}
				else {
					result = _valueFactory.createBNode();
				}
			}
			catch (Exception e) {
				throw new SAXException(e);
			}

			// Remember it, the nodeID might occur again.
			_bNodeIdMap.put(nodeID, result);
		}

		return result;
	}

	private Literal _createLiteral(String label, String lang, String datatype)
		throws SAXException
	{
		URI dtURI = null;
		if (datatype != null) {
			if (_datatypeHandling == DT_VERIFY) {
				if (!XmlDatatypeUtil.isValidValue(label, datatype)) {
					sendError("'" + label + "' is not a valid value for datatype " + datatype);
				}
			}
			else if (_datatypeHandling == DT_NORMALIZE) {
				try {
					label = XmlDatatypeUtil.normalize(label, datatype);
				}
				catch (IllegalArgumentException e) {
					sendError("'" + label + "' is not a valid value for datatype " + datatype + ": " + e.getMessage());
				}
			}

			dtURI = _createURI(datatype);
		}

		try {
			if (dtURI != null) {
				return _valueFactory.createLiteral(label, dtURI);
			}
			else if (lang != null) {
				return _valueFactory.createLiteral(label, lang);
			}
			else {
				return _valueFactory.createLiteral(label);
			}
		}
		catch (Exception e) {
			throw new SAXException(e);
		}
	}

	private Object _peekStack(int distFromTop) {
		return _elementStack.get(_elementStack.size() - 1 - distFromTop);
	}

	private boolean _topIsProperty() {
		return
			_elementStack.isEmpty() ||
			_peekStack(0) instanceof PropertyElement;
	}

	/**
	 * Checks whether the node element name is from the RDF namespace and, if so, if it is
	 * allowed to be used in a node element. If the name is equal to one of the disallowed
	 * names (RDF, ID, about, parseType, resource, nodeID, datatype and li), an error is
	 * generated. If the name is not defined in the RDF namespace, but it claims that it
	 * is from this namespace, a warning is generated.
	 **/
	private void _checkNodeEltName(String namespaceURI, String localName, String qName)
		throws SAXException
	{
		if (RDF.NAMESPACE.equals(namespaceURI)) {

			if (localName.equals("Description") ||
				localName.equals("Seq") ||
				localName.equals("Bag") ||
				localName.equals("Alt") ||
				localName.equals("Statement") ||
				localName.equals("Property") ||
				localName.equals("List") ||
				localName.equals("subject") ||
				localName.equals("predicate") ||
				localName.equals("object") ||
				localName.equals("type") ||
				localName.equals("value") ||
				localName.equals("first") ||
				localName.equals("rest") ||
				localName.equals("nil") ||
				localName.startsWith("_"))
			{
				// These are OK
			}
			else if (
				localName.equals("li") ||
				localName.equals("RDF") ||
				localName.equals("ID") ||
				localName.equals("about") ||
				localName.equals("parseType") ||
				localName.equals("resource") ||
				localName.equals("nodeID") ||
				localName.equals("datatype"))
			{
				sendError("<" + qName + "> not allowed as node element");
			}
			else if (
				localName.equals("bagID") ||
				localName.equals("aboutEach") ||
				localName.equals("aboutEachPrefix"))
			{
				sendError(qName + " is no longer a valid RDF name");
			}
			else {
				sendWarning("unknown rdf element <" + qName + ">");
			}
		}
	}

	/**
	 * Checks whether the property element name is from the RDF namespace and, if so,
	 * if it is allowed to be used in a property element. If the name is equal to one of
	 * the disallowed names (RDF, ID, about, parseType, resource and li), an error is
	 * generated. If the name is not defined in the RDF namespace, but it claims that it
	 * is from this namespace, a warning is generated.
	 **/
	private void _checkPropertyEltName(String namespaceURI, String localName, String qName)
		throws SAXException
	{
		if (RDF.NAMESPACE.equals(namespaceURI)) {

			if (localName.equals("li") ||
				localName.equals("Seq") ||
				localName.equals("Bag") ||
				localName.equals("Alt") ||
				localName.equals("Statement") ||
				localName.equals("Property") ||
				localName.equals("List") ||
				localName.equals("subject") ||
				localName.equals("predicate") ||
				localName.equals("object") ||
				localName.equals("type") ||
				localName.equals("value") ||
				localName.equals("first") ||
				localName.equals("rest") ||
				localName.equals("nil") ||
				localName.startsWith("_"))
			{
				// These are OK
			}
			else if (
				localName.equals("Description") ||
				localName.equals("RDF") ||
				localName.equals("ID") ||
				localName.equals("about") ||
				localName.equals("parseType") ||
				localName.equals("resource") ||
				localName.equals("nodeID") ||
				localName.equals("datatype"))
			{
				sendError("<" + qName + "> not allowed as property element");
			}
			else if (
				localName.equals("bagID") ||
				localName.equals("aboutEach") ||
				localName.equals("aboutEachPrefix"))
			{
				sendError(qName + " is no longer a valid RDF name");
			}
			else {
				sendWarning("unknown rdf element <" + qName + ">");
			}
		}
	}

	/**
	 * Checks whether 'atts' contains attributes from the RDF namespace that are not
	 * allowed as attributes. If such an attribute is found, an error is generated and
	 * the attribute is removed from 'atts'. If the attribute is not defined in the RDF
	 * namespace, but it claims that it is from this namespace, a warning is generated.
	 **/
	private void _checkRdfAtts(Atts atts)
		throws SAXException
	{
		Iterator iter = atts.iterator();

		while (iter.hasNext()) {
			Att att = (Att)iter.next();

			if (RDF.NAMESPACE.equals(att.getNamespace())) {
				String localName = att.getLocalName();

				if (localName.equals("Seq") ||
					localName.equals("Bag") ||
					localName.equals("Alt") ||
					localName.equals("Statement") ||
					localName.equals("Property") ||
					localName.equals("List") ||
					localName.equals("subject") ||
					localName.equals("predicate") ||
					localName.equals("object") ||
					localName.equals("type") ||
					localName.equals("value") ||
					localName.equals("first") ||
					localName.equals("rest") ||
					localName.equals("nil") ||
					localName.startsWith("_"))
				{
					// These are OK
				}
				else if (
					localName.equals("Description") ||
					localName.equals("li") ||
					localName.equals("RDF") ||
					localName.equals("ID") ||
					localName.equals("about") ||
					localName.equals("parseType") ||
					localName.equals("resource") ||
					localName.equals("nodeID") ||
					localName.equals("datatype"))
				{
					sendError("'" + att.getQName() + "' not allowed as attribute name");
					iter.remove();
				}
				else if (
					localName.equals("bagID") ||
					localName.equals("aboutEach") ||
					localName.equals("aboutEachPrefix"))
				{
					sendError(att.getQName() + " is no longer a valid RDF name");
				}
				else {
					sendWarning("unknown rdf attribute '" + att.getQName() + "'");
				}
			}
		}
	}

	/**
	 * Checks whether 'atts' is empty. If this is not the case, a warning is generated
	 * for each attribute that is still present.
	 **/
	private void _checkNoMoreAtts(Atts atts)
		throws SAXException
	{
		if (atts.size() > 0) {
			Iterator iter = atts.iterator();

			while (iter.hasNext()) {
				Att att = (Att)iter.next();

				sendError("unexpected attribute '" + att.getQName() + "'");
				iter.remove();
			}
		}
	}

	/**
	 * Reports a stament to the configured StatementHandler.
	 *
	 * @param subject The statement's subject.
	 * @param predicate The statement's predicate.
	 * @param object The statement's object.
	 * @exception SAXException If the configured StatementHandler throws a
	 * StatementHandlerException, which will be wrapped in a SAXException.
	 **/
	private void _reportStatement(Resource subject, URI predicate, Value object)
		throws SAXException
	{
		try {
			_statementHandler.handleStatement(subject, predicate, object);
		}
		catch (StatementHandlerException e) {
			// Wrap exception in a SAXException, it will be unwrapped in the
			// parse() method
			throw new SAXException(e);
		}
	}

	void sendWarning(String msg) {
		if (_errorListener != null) {
			Locator loc = _saxFilter.getLocator();
			if (loc == null) {
				_errorListener.warning(msg, -1, -1);
			}
			else {
				_errorListener.warning(msg, loc.getLineNumber(), loc.getColumnNumber());
			}
		}
	}

	void sendError(String msg)
		throws SAXException
	{
		if (_errorListener != null) {
			Locator loc = _saxFilter.getLocator();
			if (loc == null) {
				_errorListener.error(msg, -1, -1);
			}
			else {
				_errorListener.error(msg, loc.getLineNumber(), loc.getColumnNumber());
			}
		}

		if (_stopAtFirstError) {
			throw new SAXException(msg);
		}
	}

	void sendFatalError(String msg)
		throws SAXException
	{
		if (_errorListener != null) {
			Locator loc = _saxFilter.getLocator();
			if (loc == null) {
				_errorListener.fatalError(msg, -1, -1);
			}
			else {
				_errorListener.fatalError(msg, loc.getLineNumber(), loc.getColumnNumber());
			}
		}

		throw new SAXException(msg);
	}

/*------------------------------------------------+
| Inner classes NodeElement and PropertyElement   |
+------------------------------------------------*/

	static class NodeElement {
		private Resource _resource;
		private boolean _isVolatile = false;;
		private int _liCounter = 1;

		public NodeElement(Resource resource) {
			_resource = resource;
		}

		public Resource getResource() {
			return _resource;
		}

		public void setIsVolatile(boolean isVolatile) {
			_isVolatile = isVolatile;
		}

		public boolean isVolatile() {
			return _isVolatile;
		}

		public int getNextLiCounter() {
			return _liCounter++;
		}
	}

	static class PropertyElement {
		/** The property URI. **/
		private URI _uri;

		/** An optional reification identifier. **/
		private URI _reificationURI;

		/** An optional datatype. **/
		private String _datatype;

		/** Flag indicating whether this PropertyElement has an
		 * attribute <tt>rdf:parseType="Collection"</tt>.  **/
		private boolean _parseCollection = false;

		/** The resource that was used to append the last part
		 * of an rdf:List. **/
		private Resource _lastListResource;

		public PropertyElement(URI uri) {
			_uri = uri;
		}

		public URI getURI() {
			return _uri;
		}

		public boolean isReified() {
			return _reificationURI != null;
		}

		public void setReificationURI(URI reifURI) {
			_reificationURI = reifURI;
		}

		public URI getReificationURI() {
			return _reificationURI;
		}

		public void setDatatype(String datatype) {
			_datatype = datatype;
		}

		public String getDatatype() {
			return _datatype;
		}

		public boolean parseCollection() {
			return _parseCollection;
		}

		public void setParseCollection(boolean parseCollection) {
			_parseCollection = parseCollection;
		}

		public Resource getLastListResource() {
			return _lastListResource;
		}

		public void setLastListResource(Resource resource) {
			_lastListResource = resource;
		}
	}
}
