/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.rio.rdfxml;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.openrdf.util.xml.XmlUtil;
import org.openrdf.vocabulary.RDF;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.rio.RdfDocumentWriter;

/**
 * An implementation of the RdfDocumentWriter interface that writes RDF
 * documents in XML-serialized RDF format.
 **/
public class RdfXmlWriter implements RdfDocumentWriter {

/*---------------------------------+
| Variables                        |
+---------------------------------*/

	protected Writer _out;

	protected Map _namespaceTable = new HashMap();
	
	protected Set _stylesheets = new HashSet();

	protected boolean _writingStarted = false;

	private Resource _lastWrittenSubject = null;

/*---------------------------------+
| Constructors                     |
+---------------------------------*/

	/**
	 * Creates a new RdfXmlWriter that will write to the supplied OutputStream.
	 *
	 * @param out The OutputStream to write the RDF/XML document to.
	 **/
	public RdfXmlWriter(OutputStream out) {
		try {
			_out = new OutputStreamWriter(out, "UTF-8");
		}
		catch (UnsupportedEncodingException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Creates a new RdfXmlWriter that will write to the supplied Writer.
	 *
	 * @param out The Writer to write the RDF/XML document to.
	 **/
	public RdfXmlWriter(Writer out) {
		_out = out;
	}

/*---------------------------------+
| Methods from interface RdfWriter |
+---------------------------------*/

	public void setNamespace(String prefix, String name) {
		_setNamespace(prefix, name, true);
	}

	protected void _setNamespace(String prefix, String name, boolean fixedPrefix) {
		if (_writingStarted) {
			throw new RuntimeException(
					"Namespaces cannot be set after writing has started");
		}

		if (!_namespaceTable.containsKey(name)) {
			// Try to give it the specified prefix
			if (!_namespaceTable.containsValue(prefix)) {
				_namespaceTable.put(name, prefix);
			}
			else {
				// specified prefix is already used for another namespace
				if (fixedPrefix) {
					throw new IllegalArgumentException(
							"prefix already in use: " + prefix);
				}
				else {
					// specified prefix is already taken, append a number to
					// generate a unique prefix
					int number = 1;

					while (_namespaceTable.containsValue(prefix + number)) {
						number++;
					}

					_namespaceTable.put(RDF.NAMESPACE, prefix + number);
				}
			}
		}
	}

	public void startDocument()
		throws IOException
	{
		if (_writingStarted) {
			throw new RuntimeException("Document writing has already started");
		}

		// This export format needs the RDF namespace to be defined:
		_setNamespace("rdf", RDF.NAMESPACE, false);

		_writingStarted = true;

		_out.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
		
		//Print processing instructions related to stylesheets
		Iterator sheetIterator = _stylesheets.iterator();
		while (sheetIterator.hasNext()) {
			StyleSheet nextSheet = (StyleSheet) sheetIterator.next();
			_out.write("<?xml-stylesheet " +
					   "href=\"" + nextSheet.getHref() + "\" " +
					   "type=\"" + nextSheet.getType() + "\" ");
			if (nextSheet.getTitle() != null && !nextSheet.getTitle().equals("")) {
				_out.write("title=\"" + nextSheet.getTitle() + "\" ");
			}
			if (nextSheet.getMedia() != null && !nextSheet.getMedia().equals("")) {
				_out.write("media=\"" + nextSheet.getMedia() + "\" ");
			}
			if (nextSheet.getCharset() != null && !nextSheet.getCharset().equals("")) {
				_out.write("charset=\"" + nextSheet.getCharset() + "\" ");
			}
			if (nextSheet.isAlternate()) {
				_out.write("alternate=\"yes\" ");
			}
			_out.write("?>\n");
		}
		

		_writeStartOfStartTag(RDF.NAMESPACE, "RDF");

		Iterator nameIterator = _namespaceTable.keySet().iterator();
		while (nameIterator.hasNext()) {
			String name = (String)nameIterator.next();
			String prefix = (String)_namespaceTable.get(name);

			_writeNewLine();
			_writeIndent();
			_out.write("xmlns");
			if (prefix.length() > 0) {
				_out.write(':');
				_out.write(prefix);
			}
			_out.write("=\"");
			_out.write(XmlUtil.escapeDoubleQuotedAttValue(name));
			_out.write("\"");
		}

		_writeEndOfStartTag();

		_writeNewLine();
	}

	public void endDocument()
		throws IOException
	{
		if (!_writingStarted) {
			throw new RuntimeException("Document writing has not yet started");
		}

		try {
			if (_lastWrittenSubject != null) {
				// The last statement still has to be closed:
				_writeEndTag(RDF.NAMESPACE, "Description");
				_writeNewLine();

				_lastWrittenSubject = null;
			}

			_writeNewLine();
			_writeEndTag(RDF.NAMESPACE, "RDF");

			_out.flush();
		}
		finally {
			_writingStarted = false;
		}
	}

	public void writeStatement(Resource subj, URI pred, Value obj)
		throws IOException
	{
		if (!_writingStarted) {
			throw new RuntimeException("Document writing has not yet been started");
		}

		// SUBJECT
		if (!subj.equals(_lastWrittenSubject)) {
			if (_lastWrittenSubject != null) {
				// The previous statement still has to be closed:
				_writeEndTag(RDF.NAMESPACE, "Description");
				_writeNewLine();
			}

			// Write new subject:
			_writeNewLine();
			_writeStartOfStartTag(RDF.NAMESPACE, "Description");
			if (subj instanceof BNode) {
				BNode bNode = (BNode)subj;
				_writeAttribute(RDF.NAMESPACE, "nodeID", bNode.getID());
			}
			else {
				URI uri = (URI)subj;
				_writeAttribute(RDF.NAMESPACE, "about", uri.getURI());
			}
			_writeEndOfStartTag();
			_writeNewLine();

			_lastWrittenSubject = subj;
		}

		// PREDICATE
		_writeIndent();
		_writeStartOfStartTag(pred.getNamespace(), pred.getLocalName());

		// OBJECT
		if (obj instanceof Resource) {
			Resource objRes = (Resource)obj;

			if (objRes instanceof BNode) {
				BNode bNode = (BNode)objRes;
				_writeAttribute(RDF.NAMESPACE, "nodeID", bNode.getID());
			}
			else {
				URI uri = (URI)objRes;
				_writeAttribute(RDF.NAMESPACE, "resource", uri.getURI());
			}

			_writeEndOfEmptyTag();
		}
		else if (obj instanceof Literal) {
			Literal objLit = (Literal)obj;

			// language attribute
			if (objLit.getLanguage() != null) {
				_writeAttribute("xml:lang", objLit.getLanguage());
			}

			// datatype attribute
			boolean isXmlLiteral = false;
			URI datatype = objLit.getDatatype();
			if (datatype != null) {
				// Check if datatype is rdf:XMLLiteral
				String datatypeString = datatype.getURI();
				isXmlLiteral = datatypeString.equals(RDF.XMLLITERAL);

				if (isXmlLiteral) {
					_writeAttribute(RDF.NAMESPACE, "parseType", "Literal");
				}
				else {
					_writeAttribute(RDF.NAMESPACE, "datatype", objLit.getDatatype().getURI());
				}
			}

			_writeEndOfStartTag();

			// label
			if (isXmlLiteral) {
				// Write XML literal as plain XML
				_out.write(objLit.getLabel());
			}
			else {
				_writeCharacterData(objLit.getLabel());
			}

			_writeEndTag(pred.getNamespace(), pred.getLocalName());
		}

		_writeNewLine();

		// Don't write </rdf:Description> yet, maybe the next statement
		// has the same subject.
	}

	public void writeComment(String comment)
		throws IOException
	{
		if (_lastWrittenSubject != null) {
			// The last statement still has to be closed:
			_writeEndTag(RDF.NAMESPACE, "Description");
			_writeNewLine();

			_lastWrittenSubject = null;
		}

		_out.write("<!-- ");
		_out.write(comment);
		_out.write(" -->");
		_writeNewLine();
	}

/*---------------------------------+
| Other methods                    |
+---------------------------------*/

	protected void _writeStartOfStartTag(String namespace, String localName)
		throws IOException
	{
		String prefix = (String)_namespaceTable.get(namespace);

		if (prefix == null) {
			_out.write("<");
			_out.write(localName);
			_out.write(" xmlns=\"");
			_out.write( XmlUtil.escapeDoubleQuotedAttValue(namespace) );
			_out.write("\"");
		}
		else if (prefix.length() == 0) {
			// default namespace
			_out.write("<");
			_out.write(localName);
		}
		else {
			_out.write("<");
			_out.write(prefix);
			_out.write(":");
			_out.write(localName);
		}
	}

	protected void _writeAttribute(String attName, String value)
		throws IOException
	{
		_out.write(" ");
		_out.write(attName);
		_out.write("=\"");
		_out.write( XmlUtil.escapeDoubleQuotedAttValue(value) );
		_out.write("\"");
	}

	protected void _writeAttribute(String namespace, String attName, String value)
		throws IOException
	{
		String prefix = (String)_namespaceTable.get(namespace);

		if (prefix == null || prefix.length() == 0) {
			throw new RuntimeException(
					"No prefix has been declared for the namespace used in this attribute: " + namespace);
		}

		_out.write(" ");
		_out.write(prefix);
		_out.write(":");
		_out.write(attName);
		_out.write("=\"");
		_out.write( XmlUtil.escapeDoubleQuotedAttValue(value) );
		_out.write("\"");
	}

	protected void _writeEndOfStartTag()
		throws IOException
	{
		_out.write(">");
	}

	protected void _writeEndOfEmptyTag()
		throws IOException
	{
		_out.write("/>");
	}

	protected void _writeEndTag(String namespace, String localName)
		throws IOException
	{
		String prefix = (String)_namespaceTable.get(namespace);

		if (prefix == null || prefix.length() == 0) {
			_out.write("</");
			_out.write(localName);
			_out.write(">");
		}
		else {
			_out.write("</");
			_out.write(prefix);
			_out.write(":");
			_out.write(localName);
			_out.write(">");
		}
	}

	protected void _writeCharacterData(String chars)
		throws IOException
	{
		_out.write( XmlUtil.escapeCharacterData(chars) );
	}

	protected void _writeIndent()
		throws IOException
	{
		_out.write("\t");
	}

	protected void _writeNewLine()
		throws IOException
	{
		_out.write("\n");
	}
	
	private static class StyleSheet {
		
		private String href = null;
		private String type = null;
		private String title = null;
		private String media = null;
		private String charset = null;
		private boolean alternate = false;
		
		public StyleSheet(String href, String type, String title, String media, String charset, boolean alternate) {
			if (href != null && type != null) {
				this.href = href;
				this.type = type;
			}
			else {
				throw new NullPointerException("href and type must not be null");
			}
			this.title = title;
			this.media = media;
			this.charset = charset;
			this.alternate = alternate; 
		}
		
		/**
		 * @return Returns the alternate.
		 */
		public boolean isAlternate() {
			return alternate;
		}
		/**
		 * @param alternate The alternate to set.
		 */
		public void setAlternate(boolean alternate) {
			this.alternate = alternate;
		}
		/**
		 * @return Returns the charset.
		 */
		public String getCharset() {
			return charset;
		}
		/**
		 * @param charset The charset to set.
		 */
		public void setCharset(String charset) {
			this.charset = charset;
		}
		/**
		 * @return Returns the href.
		 */
		public String getHref() {
			return href;
		}
		/**
		 * @param href The href to set.
		 */
		public void setHref(String href) {
			this.href = href;
		}
		/**
		 * @return Returns the media.
		 */
		public String getMedia() {
			return media;
		}
		/**
		 * @param media The media to set.
		 */
		public void setMedia(String media) {
			this.media = media;
		}
		/**
		 * @return Returns the title.
		 */
		public String getTitle() {
			return title;
		}
		/**
		 * @param title The title to set.
		 */
		public void setTitle(String title) {
			this.title = title;
		}
		/**
		 * @return Returns the type.
		 */
		public String getType() {
			return type;
		}
		/**
		 * @param type The type to set.
		 */
		public void setType(String type) {
			this.type = type;
		}
	}
	
	/** Associate a stylesheet with this XML document.
	 * See the relevant <a href="http://www.w3.org/TR/xml-stylesheet/"> 
	 * W3C Recommendation</a> for an explanation of the parameters. 
	 * 
	 * @param href Required stylesheet attribute
	 * @param type Required stylesheet attribute
	 * @param title Optional stylesheet attribute
	 * @param media Optional stylesheet attribute
	 * @param charset Optional stylesheet attribute
	 * @param alternate Optional stylesheet attribute
	 */
	public void addStyleSheet(String href, String type, String title, String media, String charset, boolean alternate) {
		if (_writingStarted) {
			throw new RuntimeException(
					"Stylesheets cannot be added after writing has started");
		}
		_stylesheets.add(new StyleSheet(href, type, title, media, charset, alternate));
	}
	
	public void addStyleSheet(String href, String type) {
		_stylesheets.add(new StyleSheet(href, type, null, null, null, false));
	}
}
