/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.rio.turtle;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.PushbackReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import org.openrdf.util.ASCIIUtil;
import org.openrdf.util.xml.XmlDatatypeUtil;
import org.openrdf.vocabulary.RDF;
import org.openrdf.vocabulary.XmlSchema;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.impl.ValueFactoryImpl;

import org.openrdf.rio.NamespaceListener;
import org.openrdf.rio.ParseErrorListener;
import org.openrdf.rio.ParseException;
import org.openrdf.rio.ParseLocationListener;
import org.openrdf.rio.Parser;
import org.openrdf.rio.StatementHandler;
import org.openrdf.rio.StatementHandlerException;

/**
 * Parser for Turtle files. A specification of Turtle can be found
 * <a href="http://www.dajobe.org/2004/01/turtle/">in this document</a>. This
 * parser is not thread-safe, therefore its public methods are synchronized.
 * <p>
 * This implementation is based on the 2006/01/02 version of the Turtle
 * specification, with slight deviations:
 * <ul>
 * <li>Normalization of integer, floating point and boolean values is dependent
 * on the specified datatype handling. According to the specification, integers
 * and booleans should be normalized, but floats don't.</li>
 * <li>Comments can be used anywhere in the document, and extend to the end of
 * the line. The Turtle grammar doesn't allow comments to be used inside triple
 * constructs that extend over multiple lines, but the author's own parser
 * deviates from this too.</li>
 * </ul>
 **/
public class TurtleParser implements Parser {

/*--------------+
| Variables     |
+--------------*/

	private StatementHandler _statementHandler;

	private NamespaceListener _nsListener;

	private ParseErrorListener _errListener;

	private ParseLocationListener _locListener;

	/** The base URI for resolving relative URIs. **/
	private org.openrdf.util.uri.URI _baseURI;

	private LineNumberReader _lineReader;
	private PushbackReader _reader;
	
	private ValueFactory _valFactory;

	/**
	 * Mapping from bNode ID's as used in the RDF document to the
	 * object created for it by the ValueFactory.
	 **/
	private Map _bNodeIdMap;

	private Map _namespaceTable;

	private Resource _subject;
	private URI _predicate;
	private Value _object;

	/** Flag indicating whether the parser should check the data it parses. **/
	boolean _verifyData = true;

	/** 
	 * Flag indicating whether the parser should preserve bnode identifiers
	 * specified in the source.
	 */
	boolean _preserveBNodeIds = false;
	
	/**
	 * Indicates how datatyped literals should be handled. Legal
	 * values are <tt>DT_IGNORE</tt>, <tt>DT_VERIFY</tt> and
	 * <tt>DT_NORMALIZE</tt>.
	 **/
	private int _datatypeHandling;

/*--------------+
| Constructors  |
+--------------*/

	/**
	 * Creates a new TurtleParser that will use a <tt>ValueFactoryImpl</tt> to
	 * create object for resources, bNodes and literals.
	 * @see org.openrdf.model.impl.ValueFactoryImpl
	 **/
	public TurtleParser() {
		this(new ValueFactoryImpl());
	}

	/**
	 * Creates a new TurtleParser that will use the supplied ValueFactory to
	 * create objects for resources, bNodes and literals.
	 *
	 * @param valueFactory A ValueFactory.
	 **/
	public TurtleParser(ValueFactory valueFactory) {
		_valFactory = valueFactory;
		_bNodeIdMap = new HashMap();
		_datatypeHandling = DT_VERIFY;
		_namespaceTable = new HashMap(16);
	}

/*--------------+
| Methods       |
+--------------*/

	// Implements Parser.setStatementHandler(StatementHandler)
	public synchronized void setStatementHandler(StatementHandler sh) {
		_statementHandler = sh;
	}

	// Implements Parser.setNamespaceListener(NamespaceListener)
	public void setNamespaceListener(NamespaceListener nl) {
		_nsListener = nl;
	}

	// Implements Parser.setParseErrorListener(ParseErrorListener)
	public synchronized void setParseErrorListener(ParseErrorListener el) {
		_errListener = el;
	}

	// Implements Parser.setParseLocationListener(ParseLocationListener)
	public synchronized void setParseLocationListener(ParseLocationListener el) {
		_locListener = el;
	}

	// Implements Parser.setVerifyData(boolean)
	public synchronized void setVerifyData(boolean verifyData) {
		_verifyData = verifyData;
	}

	// Implements Parser.setPreserveBNodeIds(boolean)
	public void setPreserveBNodeIds(boolean preserveBNodeIds) {
		_preserveBNodeIds = preserveBNodeIds;
	}

	// Implements Parser.setStopAtFirstError(boolean)
	public synchronized void setStopAtFirstError(boolean stopAtFirstError) {
		// ignore
	}

	// Implements Parser.setDatatypeHandling(int)
	public void setDatatypeHandling(int datatypeHandling) {
		_datatypeHandling = datatypeHandling;
	}

	/**
	 * Implementation of the <tt>parse(InputStream, String)</tt> method defined
	 * in the Parser interface. 	
	 * 
	 * @param in The InputStream from which to read the data. The InputStream is
	 * supposed to contain UTF-8 encoded Unicode characters, as per the Turtle
	 * specification.
	 * @param baseURI The URI associated with the data in the InputStream.
	 * @exception IOException If an I/O error occurred while data was read
	 * from the InputStream.
	 * @exception ParseException If the parser has found an unrecoverable
	 * parse error.
	 * @exception StatementHandler If the configured statement handler
	 * encountered an unrecoverable error.
	 * @exception IllegalArgumentException If the supplied input stream or
	 * base URI is <tt>null</tt>.
	 **/
	public synchronized void parse(InputStream in, String baseURI)
		throws IOException, ParseException, StatementHandlerException
	{
		if (in == null) {
			throw new IllegalArgumentException("Input stream can not be 'null'");
		}
		// Note: baseURI will be checked in parse(Reader, String)

		try {
			parse(new InputStreamReader(in, "UTF-8"), baseURI);
		}
		catch (UnsupportedEncodingException e) {
			// Every platform should support the UTF-8 encoding...
			throw new RuntimeException(e);
		}
	}

	/**
	 * Implementation of the <tt>parse(Reader, String)</tt> method defined in
	 * the Parser interface. 	
	 * 
	 * @param reader The Reader from which to read the data.
	 * @param baseURI The URI associated with the data in the Reader.
	 * @exception IOException If an I/O error occurred while data was read
	 * from the InputStream.
	 * @exception ParseException If the parser has found an unrecoverable
	 * parse error.
	 * @exception StatementHandler If the configured statement handler
	 * encountered an unrecoverable error.
	 * @exception IllegalArgumentException If the supplied reader or base URI
	 * is <tt>null</tt>.
	 **/
	public synchronized void parse(Reader reader, String baseURI)
		throws IOException, ParseException, StatementHandlerException
	{
		if (reader == null) {
			throw new IllegalArgumentException("Reader can not be 'null'");
		}
		if (baseURI == null) {
			throw new IllegalArgumentException("base URI can not be 'null'");
		}

		_lineReader = new LineNumberReader(reader);
		// Start counting lines at 1:
		_lineReader.setLineNumber(1);

		// Allow at most 2 characters to be pushed back:
		_reader = new PushbackReader(_lineReader, 2);

		// Store normalized base URI
		_baseURI = new org.openrdf.util.uri.URI(baseURI);
		_baseURI.normalize();

		_reportLocation();

		try {
			while (true) {
				int c = _skipWSC();

				if (c == -1) {
					break;
				}
				else if (c == '@') {
					_parsePrefix();
				}
				else {
					_parseTriple();
				}
			}
		}
		finally {
			_bNodeIdMap.clear();
			_namespaceTable.clear();
		}
	}

	/**
	 * Parses a <tt>prefix</tt> directive.
	 */
	private void _parsePrefix()
		throws IOException, ParseException
	{
		// Verify that the first characters form the string "prefix"
		_verifyCharacter(_read(), "@");
		_verifyCharacter(_read(), "p");
		_verifyCharacter(_read(), "r");
		_verifyCharacter(_read(), "e");
		_verifyCharacter(_read(), "f");
		_verifyCharacter(_read(), "i");
		_verifyCharacter(_read(), "x");

		_skipWSC();

		// Read prefix ID (e.g. "rdf:" or ":")
		StringBuffer prefixID = new StringBuffer(8);

		while (true) {
			int c = _read();
			
			if (c == ':') {
				// marks the end of the prefix
				_read();
				break;
			}
			else if (c == -1) {
				_throwEOFException();
			}

			prefixID.append( (char)c );
		}

		_skipWSC();

		// Read the namespace URI
		URI namespace = _parseUriRef();

		_skipWSC();

		// Read closing dot
		_verifyCharacter(_read(), ".");

		// Store and report this namespace mapping
		String prefixStr = prefixID.toString();
		String namespaceStr = namespace.getURI();

		_namespaceTable.put(prefixStr, namespaceStr);

		if (_nsListener != null) {
			_nsListener.handleNamespace(prefixStr, namespaceStr);
		}
	}

	private void _parseTriple()
		throws IOException, ParseException, StatementHandlerException
	{
		_parseSubject();

		_skipWSC();

		_parsePredicateObjectList();

		_skipWSC();

		_verifyCharacter(_read(), ".");

		_subject = null;
		_predicate = null;
		_object = null;
	}

	private void _parsePredicateObjectList()
		throws IOException, ParseException, StatementHandlerException
	{
		_predicate = _parsePredicate();

		_skipWSC();

		_parseObjectList();

		while (_skipWSC() == ';') {
			_read();

			int c = _skipWSC();

			if (c == '.' || // end of triple
				c == ']') // end of predicateObjectList inside blank node
			{
				break;
			}

			_predicate = _parsePredicate();

			_skipWSC();

			_parseObjectList();
		}
	}

	private void _parseObjectList()
		throws IOException, ParseException, StatementHandlerException
	{
		_parseObject();
		
		while (_skipWSC() == ',') {
			_read();

			_skipWSC();

			_parseObject();
		}
	}

	private void _parseSubject()
		throws IOException, ParseException, StatementHandlerException
	{
		int c = _peek();

		if (c == '(') {
			_subject = _parseCollection();
		}
		else if (c == '[') {
			_subject = _parseImplicitBlank();
		}
		else {
			Value value = _parseValue();

			if (value instanceof Resource) {
				_subject = (Resource)value;
			}
			else {
				_throwParseException("Illegal subject value: " + value);
			}
		}
	}

	private URI _parsePredicate()
		throws IOException, ParseException
	{
		// Check if the short-cut 'a' is used
		int c1 = _read();

		if (c1 == 'a') {
			int c2 = _read();

			if (TurtleUtil.isWhitespace(c2)) {
				// Short-cut is used, return the rdf:type URI
				return _createURI(RDF.TYPE);
			}

			// Short-cut is not used, unread all characters
			_unread(c2);
		}
		_unread(c1);

		// Predicate is a normal resource
		Value predicate = _parseValue();
		if (predicate instanceof URI) {
			return (URI)predicate;
		}
		else {
			_throwParseException("Illegal predicate value: " + predicate);
			return null;
		}
	}

	private void _parseObject()
		throws IOException, ParseException, StatementHandlerException
	{
		int c = _peek();

		if (c == '(') {
			_object = _parseCollection();
		}
		else if (c == '[') {
			_object = _parseImplicitBlank();
		}
		else {
			_object = _parseValue();
		}

		_statementHandler.handleStatement(_subject, _predicate, _object);
	}

	/**
	 * Parses a collection, e.g. <tt>( item1 item2 item3 )</tt>.
	 */
	private Resource _parseCollection()
		throws IOException, ParseException, StatementHandlerException
	{
		_verifyCharacter(_read(), "(");

		int c = _skipWSC();

		if (c == ')') {
			// Empty list
			_read();
			return _createURI(RDF.NIL);
		}
		else {
			BNode listRoot = _createBNode();

			// Remember current subject and predicate
			Resource oldSubject = _subject;
			URI oldPredicate = _predicate;

			// Create URIs for rdf:first and rdf:rest
			URI rdfFirst = _createURI(RDF.FIRST);
			URI rdfRest = _createURI(RDF.REST);
			URI rdfNil = _createURI(RDF.NIL);

			// generated bNode becomes subject, predicate becomes rdf:first
			_subject = listRoot;
			_predicate = rdfFirst;

			_parseObject();

			BNode bNode = listRoot;

			while (_skipWSC() != ')') {
				// Create another list node and link it to the previous
				BNode newNode = _createBNode();
				_statementHandler.handleStatement(bNode, rdfRest, newNode);

				// New node becomes the current
				_subject = bNode = newNode;

				_parseObject();
			}

			// Skip ')'
			_read();

			// Close the list
			_statementHandler.handleStatement(bNode, rdfRest, rdfNil);

			// Restore previous subject and predicate
			_subject = oldSubject;
			_predicate = oldPredicate;

			return listRoot;
		}
	}

	/**
	 * Parses an implicit blank node. This method parses the token
	 * <tt>[]</tt> and predicateObjectLists that are surrounded by square
	 * brackets.
	 */
	private Resource _parseImplicitBlank()
		throws IOException, ParseException, StatementHandlerException
	{
		_verifyCharacter(_read(), "[");

		BNode bNode = _createBNode();

		int c = _read();
		if (c != ']') {
			_unread(c);

			// Remember current subject and predicate
			Resource oldSubject = _subject;
			URI oldPredicate = _predicate;

			// generated bNode becomes subject
			_subject = bNode;

			// Enter recursion with nested predicate-object list
			_skipWSC();

			_parsePredicateObjectList();

			_skipWSC();

			// Read closing bracket
			_verifyCharacter(_read(), "]");

			// Restore previous subject and predicate
			_subject = oldSubject;
			_predicate = oldPredicate;
		}

		return bNode;
	}

	/**
	 * Parses an RDF value. This method parses uriref, qname, node ID, quoted
	 * literal, integer, double and boolean.
	 */
	private Value _parseValue()
		throws IOException, ParseException
	{
		int c = _peek();

		if (c == '<') {
			// uriref, e.g. <foo://bar>
			return _parseUriRef();
		}
		else if (c == ':' || TurtleUtil.isPrefixStartChar(c)) {
			// qname or boolean
			return _parseQNameOrBoolean();
		}
		else if (c == '_') {
			// node ID, e.g. _:n1
			return _parseNodeID();
		}
		else if (c == '"') {
			// quoted literal, e.g. "foo" or """foo"""
			return _parseQuotedLiteral();
		}
		else if (ASCIIUtil.isNumber(c) || c == '.' || c == '+' || c == '-') {
			// integer or double, e.g. 123 or 1.2e3
			return _parseNumber();
		}
		else if (c == -1) {
			_throwEOFException();
			return null;
		}
		else {
			_throwParseException("Expected an RDF value here, found '" + (char)c + "'");
			return null;
		}
	}

	/**
	 * Parses a quoted string, optionally followed by a language tag or
	 * datatype.
	 */
	private Literal _parseQuotedLiteral()
		throws IOException, ParseException
	{
		String label = _parseQuotedString();

		// Check for presence of a language tag or datatype
		int c = _peek();

		if (c == '@') {
			_read();

			// Read language
			StringBuffer lang = new StringBuffer(8);

			c = _read();
			if (c == -1) {
				_throwEOFException();
			}
			if (!TurtleUtil.isLanguageStartChar(c)) {
				_throwParseException("Expected a letter, found '" + (char)c + "'");
			}

			lang.append( (char)c );

			c = _read();
			while (TurtleUtil.isLanguageChar(c)) {
				lang.append( (char)c );
				c = _read();
			}

			_unread(c);

			return _createLiteral(label, lang.toString(), null);
		}
		else if (c == '^') {
			_read();

			// next character should be another '^'
			_verifyCharacter(_read(), "^");

			// Read datatype
			Value datatype = _parseValue();
			if (datatype instanceof URI) {
				return _createLiteral(label, null, (URI)datatype);
			}
			else {
				_throwParseException("Illegal datatype value: " + datatype);
				return null;
			}
		}
		else {
			return _createLiteral(label, null, null);
		}
	}

	/**
	 * Parses a quoted string, which is either a "normal string" or a
	 * """long string""".
	 */
	private String _parseQuotedString()
		throws IOException, ParseException
	{
		String result = null;

		// First character should be '"'
		_verifyCharacter(_read(), "\"");

		// Check for long-string, which starts and ends with three double quotes
		int c2 = _read();
		int c3 = _read();

		if (c2 == '"' && c3 == '"') {
			// Long string
			result = _parseLongString();
		}
		else {
			// Normal string
			_unread(c3);
			_unread(c2);

			result = _parseString();
		}

		// Unescape any escape sequences
		try {
			result = TurtleUtil.decodeString(result);
		}
		catch (IllegalArgumentException e) {
			_throwParseException(e.getMessage());
		}

		return result;
	}

	/**
	 * Parses a "normal string". This method assumes that the first double quote
	 * has already been parsed.
	 */
	private String _parseString()
		throws IOException, ParseException
	{
		StringBuffer stringBuf = new StringBuffer(32);

		while (true) {
			int c = _read();

			if (c == '"') {
				break;
			}
			else if (c == -1) {
				_throwEOFException();
			}

			stringBuf.append( (char)c );

			if (c == '\\') {
				// This escapes the next character, which might be a '"'
				c = _read();
				if (c == -1) {
					_throwEOFException();
				}
				stringBuf.append( (char)c );
			}
		}

		return stringBuf.toString();
	}

	/**
	 * Parses a """long string""". This method assumes that the first three
	 * double quotes have already been parsed.
	 */
	private String _parseLongString()
		throws IOException, ParseException
	{
		StringBuffer stringBuf = new StringBuffer(1024);

		int doubleQuoteCount = 0;
		int c;

		while (doubleQuoteCount < 3) {
			c = _read();

			if (c == -1) {
				_throwEOFException();
			}
			else if (c == '"') {
				doubleQuoteCount++;
			}
			else {
				doubleQuoteCount = 0;
			}

			stringBuf.append( (char)c );

			if (c == '\\') {
				// This escapes the next character, which might be a '"'
				c = _read();
				if (c == -1) {
					_throwEOFException();
				}
				stringBuf.append( (char)c );
			}
		}

		return stringBuf.substring(0, stringBuf.length() - 3);
	}

	private Literal _parseNumber()
		throws IOException, ParseException
	{
		StringBuffer value = new StringBuffer(8);
		String datatype = XmlSchema.INTEGER;

		int c = _read();

		// read optional sign character
		if (c == '+' || c == '-') {
			value.append( (char)c );
			c = _read();
		}
		
		while (ASCIIUtil.isNumber(c)) {
			value.append( (char)c );
			c = _read();
		}

		if (c == '.' || c == 'e' || c == 'E') {
			// We're parsing a decimal or a double
			datatype = XmlSchema.DECIMAL;

			// read optional fractional digits
			if (c == '.') {
				value.append( (char)c );

				c = _read();
				while (ASCIIUtil.isNumber(c)) {
					value.append( (char)c );
					c = _read();
				}

				if (value.length() == 1) {
					// We've only parsed a '.'
					_throwParseException("Object for statement missing");
				}
			}
			else {
				if (value.length() == 0) {
					// We've only parsed an 'e' or 'E'
					_throwParseException("Object for statement missing");
				}
			}

			// read optional exponent
			if (c == 'e' || c == 'E') {
				datatype = XmlSchema.DOUBLE;
				value.append( (char)c );

				c = _read();
				if (c == '+' || c == '-') {
					value.append( (char)c );
					c = _read();
				}

				if (!ASCIIUtil.isNumber(c)) {
					_throwParseException("Exponent value missing");
				}

				value.append( (char)c );

				c = _read();
				while (ASCIIUtil.isNumber(c)) {
					value.append( (char)c );
					c = _read();
				}
			}
		}

		// Unread last character, it isn't part of the number
		_unread(c);

		// Return result as a typed literal
		URI dtURI = _createURI(datatype);
		return _createLiteral(value.toString(), null, dtURI);
	}

	private URI _parseUriRef()
		throws IOException, ParseException
	{
		StringBuffer uriRef = new StringBuffer(100);

		// First character should be '<'
		int c = _read();
		_verifyCharacter(c, "<");

		// Read up to the next '>' character
		while (true) {
			c = _read();

			if (c == '>') {
				break;
			}
			else if (c == -1) {
				_throwEOFException();
			}

			uriRef.append( (char)c );

			if (c == '\\') {
				// This escapes the next character, which might be a '>'
				c = _read();
				if (c == -1) {
					_throwEOFException();
				}
				uriRef.append( (char)c );
			}
		}

		// Unescape any escape sequences
		String uri = null;
		try {
			uri = TurtleUtil.decodeString(uriRef.toString());
		}
		catch (IllegalArgumentException e) {
			_throwParseException(e.getMessage());
		}

		// Resolve relative URIs against base URI
		org.openrdf.util.uri.URI relUri = new org.openrdf.util.uri.URI(uri);

		if (_verifyData) {
			if (relUri.isRelative() && !relUri.isSelfReference() && // Relative URI that is not a self-reference
				_baseURI.isOpaque())
			{
				_throwParseException("Relative URI '" + uri +
						"' cannot be resolved using the opaque base URI '" +
						_baseURI + "'");
			}
		}

		uri = _baseURI.resolve(relUri).toString();

		return _createURI(uri);
	}

	/**
	 * Parses qnames and boolean values, which have equivalent starting
	 * characters.
	 */
	private Value _parseQNameOrBoolean()
		throws IOException, ParseException
	{
		// First character should be a ':' or a letter
		int c = _read();
		if (c == -1) {
			_throwEOFException();
		}
		if (c != ':' && !TurtleUtil.isPrefixStartChar(c)) {
			_throwParseException("Expected a ':' or a letter, found '" + (char)c + "'");
		}

		String namespace = null;

		if (c == ':') {
			// qname using default namespace
			namespace = (String)_namespaceTable.get("");
			if (namespace == null) {
				_throwParseException("Default namespace used but not defined");
			}
		}
		else {
			// c is the first letter of the prefix
			StringBuffer prefix = new StringBuffer(8);
			prefix.append( (char)c );

			c = _read();
			while (TurtleUtil.isPrefixChar(c)) {
				prefix.append( (char)c );
				c = _read();
			}

			if (c != ':') {
				// prefix may actually be a boolean value
				String value = prefix.toString();

				if (value.equals("true") || value.equals("false")) {
					URI dtURI = _createURI(XmlSchema.BOOLEAN);
					return _createLiteral(value, null, dtURI);
				}
			}

			_verifyCharacter(c, ":");

			namespace = (String)_namespaceTable.get(prefix.toString());
			if (namespace == null) {
				_throwParseException("Namespace prefix '" + prefix.toString() + "' used but not defined");
			}
		}

		// c == ':', read optional local name
		StringBuffer localName = new StringBuffer(16);
		c = _read();
		if (TurtleUtil.isNameStartChar(c)) {
			localName.append( (char)c );

			c = _read();
			while (TurtleUtil.isNameChar(c)) {
				localName.append( (char)c );
				c = _read();
			}
		}

		// Unread last character
		_unread(c);

		// Note: namespace has already been resolved
		return _createURI( namespace + localName.toString() );
	}

	/**
	 * Parses a blank node ID, e.g. <tt>_:node1</tt>.
	 */
	private BNode _parseNodeID()
		throws IOException, ParseException
	{
		// Node ID should start with "_:"
		_verifyCharacter(_read(), "_");
		_verifyCharacter(_read(), ":");

		// Read the node ID
		int c = _read();
		if (c == -1) {
			_throwEOFException();
		}
		else if (!TurtleUtil.isNameStartChar(c)) {
			_throwParseException("Expected a letter, found '" + (char)c + "'");
		}

		StringBuffer name = new StringBuffer(32);
		name.append( (char)c );

		// Read all following letter and numbers, they are part of the name
		c = _read();
		while (TurtleUtil.isNameChar(c)) {
			name.append( (char)c );
			c = _read();
		}

		_unread(c);

		return _createBNode( name.toString() );
	}

	private URI _createURI(String uri)
		throws ParseException
	{
		try {
			return _valFactory.createURI(uri);
		}
		catch (Exception e) {
			_throwParseException(e);
			return null;
		}
	}

	private BNode _createBNode()
		throws ParseException
	{
		try {
			return _valFactory.createBNode();
		}
		catch (Exception e) {
			_throwParseException(e);
			return null;
		}
	}

	private BNode _createBNode(String nodeID)
		throws ParseException
	{
		// Maybe the node ID has been used before:
		BNode result = (BNode)_bNodeIdMap.get(nodeID);

		if (result == null) {
			// This is a new node ID, create a new BNode object for it
			try {
				if (_preserveBNodeIds) {
					result = _valFactory.createBNode(nodeID);
				}
				else {
					result = _valFactory.createBNode();
				}
			}
			catch (Exception e) {
				_throwParseException(e);
			}

			// Remember it, the nodeID might occur again.
			_bNodeIdMap.put(nodeID, result);
		}

		return result;
	}

	private Literal _createLiteral(String label, String lang, URI datatype)
		throws ParseException
	{
		if (datatype != null) {
			if (_datatypeHandling == DT_VERIFY) {
				if (!XmlDatatypeUtil.isValidValue(label, datatype.getURI())) {
					_throwParseException("'" + label + "' is not a valid value for datatype " + datatype);
				}
			}
			else if (_datatypeHandling == DT_NORMALIZE) {
				try {
					label = XmlDatatypeUtil.normalize(label, datatype.getURI());
				}
				catch (IllegalArgumentException e) {
					_throwParseException("'" + label + "' is not a valid value for datatype " + datatype + ": " + e.getMessage());
				}
			}
		}

		try {
			if (datatype != null) {
				return _valFactory.createLiteral(label, datatype);
			}
			else if (lang != null) {
				return _valFactory.createLiteral(label, lang);
			}
			else {
				return _valFactory.createLiteral(label);
			}
		}
		catch (Exception e) {
			_throwParseException(e);
			return null;
		}
	}

	/**
	 * Verifies that the supplied character <tt>c</tt> is one of the expected
	 * characters specified in <tt>expected</tt>. This method will throw a
	 * <tt>ParseException</tt> if this is not the case.
	 **/
	private void _verifyCharacter(int c, String expected)
		throws ParseException
	{
		if (c == -1) {
			_throwEOFException();
		}
		else if (expected.indexOf((char)c) == -1) {
			StringBuffer msg = new StringBuffer(32);
			msg.append("Expected ");
			for (int i = 0; i < expected.length(); i++) {
				if (i > 0) {
					msg.append(" or ");
				}
				msg.append('\'');
				msg.append(expected.charAt(i));
				msg.append('\'');
			}
			msg.append(", found '");
			msg.append( (char)c );
			msg.append("'");

			_throwParseException(msg.toString());
		}
	}

	/**
	 * Consumes any white space characters (space, tab, line feed, newline) and
	 * comments (#-style) from <tt>_reader</tt>. After this method has been
	 * called, the first character that is returned by <tt>_reader</tt> is
	 * either a non-ignorable character, or EOF. For convenience, this character
	 * is also returned by this method.
	 *
	 * @return The next character that will be returned by <tt>_reader</tt>.
	 **/
	private int _skipWSC()
		throws IOException
	{
		int c = _read();
		while (TurtleUtil.isWhitespace(c) || c == '#') {
			if (c == '#') {
				_skipLine();
			}

			c = _read();
		}

		_unread(c);

		return c;
	}

	/**
	 * Consumes characters from _reader until the first EOL has been read.
	 **/
	private void _skipLine()
		throws IOException
	{
		int c = _read();
		while (c != -1 && c != 0xD && c != 0xA) {
			c = _read();
		}

		// c is equal to -1, \r or \n.
		// In case c is equal to \r, we should also read a following \n.
		if (c == 0xD) {
			c = _read();

			if (c != 0xA) {
				_unread(c);
			}
		}

		_reportLocation();
	}

	private int _read()
		throws IOException
	{
		return _reader.read();
	}

	private void _unread(int c)
		throws IOException
	{
		if (c != -1) {
			_reader.unread(c);
		}
	}

	private int _peek()
		throws IOException
	{
		int result = _read();
		_unread(result);
		return result;
	}

	private void _reportLocation() {
		if (_locListener != null) {
			_locListener.parseLocationUpdate(_lineReader.getLineNumber(), -1);
		}
	}

	private void _throwParseException(String msg)
		throws ParseException
	{
		if (_errListener != null) {
			_errListener.fatalError(msg, _lineReader.getLineNumber(), -1);
		}

		throw new ParseException(msg, _lineReader.getLineNumber(), -1);
	}

	private void _throwParseException(Exception e)
		throws ParseException
	{
		if (e instanceof ParseException) {
			throw (ParseException)e;
		}
		else {
			if (_errListener != null) {
				_errListener.fatalError(e.getMessage(), _lineReader.getLineNumber(), -1);
			}

			throw new ParseException(e, _lineReader.getLineNumber(), -1);
		}
	}

	private void _throwEOFException()
		throws ParseException
	{
		_throwParseException("Unexpected end of file");
	}
}
