/*  Sesame - Storage and Querying architecture for RDF and RDF Schema
 *  Copyright (C) 2001-2006 Aduna
 *
 *  Contact: 
 *  	Aduna
 *  	Prinses Julianaplein 14 b
 *  	3817 CS Amersfoort
 *  	The Netherlands
 *  	tel. +33 (0)33 465 99 87
 *  	fax. +33 (0)33 465 99 87
 *
 *  	http://aduna-software.com/
 *  	http://www.openrdf.org/
 *  
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openrdf.rio.turtle;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;

import org.openrdf.vocabulary.RDF;

import org.openrdf.model.BNode;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.Value;

import org.openrdf.rio.RdfDocumentWriter;

/**
 * An implementation of the RdfDocumentWriter interface that writes RDF
 * documents in Turtle format. The Turtle format is defined in
 * <a href="http://www.dajobe.org/2004/01/turtle/">in this document</a>.
 **/
public class TurtleWriter implements RdfDocumentWriter {

/*---------------------------------+
| Variables                        |
+---------------------------------*/

	private Writer _out;

	private Map _namespaceTable;

	private boolean _writingStarted;

	/** Flag indicating whether the last written statement has been closed. **/
	private boolean _statementClosed;

	private Resource _lastWrittenSubject;
	private URI _lastWrittenPredicate;

/*---------------------------------+
| Constructors                     |
+---------------------------------*/

	protected TurtleWriter() {
		_namespaceTable = new HashMap();
		_writingStarted = false;
		_statementClosed = true;
		_lastWrittenSubject = null;
		_lastWrittenPredicate = null;
	}

	/**
	 * Creates a new TurtleWriter that will write to the supplied OutputStream.
	 * 
	 * @param out The OutputStream to write the Turtle document to.
	 **/
	public TurtleWriter(OutputStream out) {
		this();

		try {
			_out = new OutputStreamWriter(out, "UTF-8");
		}
		catch (java.io.UnsupportedEncodingException e) {
			// UTF-8 is required to be supported on all JVMs...
			throw new RuntimeException(e);
		}
	}

	/**
	 * Creates a new TurtleWriter that will write to the supplied Writer.
	 * 
	 * @param out The Writer to write the Turtle document to.
	 **/
	public TurtleWriter(Writer out) {
		this();
		_out = out;
	}

/*---------------------------------+
| Methods from interface RdfWriter |
+---------------------------------*/

	public void setNamespace(String prefix, String name)
		throws IOException
	{
		if (!_namespaceTable.containsKey(name)) {
			_namespaceTable.put(name, prefix);
			
			if (_writingStarted) {
				_closePreviousStatement();
				
				_writeNamespace(prefix, name);
				_writeNewLine();
			}
		}
	}

	public void startDocument()
		throws IOException
	{
		if (_writingStarted) {
			throw new RuntimeException("Document writing has already started");
		}

		_writingStarted = true;

		// Write namespace declarations
		Iterator nameIterator = _namespaceTable.keySet().iterator();
		while (nameIterator.hasNext()) {
			String name = (String)nameIterator.next();
			String prefix = (String)_namespaceTable.get(name);

			_writeNamespace(prefix, name);
		}

		if (!_namespaceTable.isEmpty()) {
			_writeNewLine();
		}
	}

	public void endDocument()
		throws IOException
	{
		if (!_writingStarted) {
			throw new RuntimeException("Document writing has not yet started");
		}

		try {
			_closePreviousStatement();
			_out.flush();
		}
		finally {
			_writingStarted = false;
		}
	}

	public void writeStatement(Resource subj, URI pred, Value obj)
		throws IOException
	{
		if (!_writingStarted) {
			throw new RuntimeException("Document writing has not yet been started");
		}

		if (subj.equals(_lastWrittenSubject)) {
			if (pred.equals(_lastWrittenPredicate)) {
				// Identical subject and predicate
				_out.write(" , ");
			}
			else {
				// Identical subject, new predicate
				_out.write(" ;");
				_writeNewLine();
				_writeIndent();

				// Write new predicate
				_writePredicate(pred);
				_out.write(" ");
				_lastWrittenPredicate = pred;
			}
		}
		else {
			// New subject
			_closePreviousStatement();

			// Write new subject:
			_writeResource(subj);
			_out.write(" ");
			_lastWrittenSubject = subj;

			// Write new predicate
			_writePredicate(pred);
			_out.write(" ");
			_lastWrittenPredicate = pred;

			_statementClosed = false;
		}

		_writeValue(obj);

		// Don't close the line just yet. Maybe the next
		// statement has the same subject and/or predicate.
	}

	public void writeComment(String comment)
		throws IOException
	{
		_closePreviousStatement();

		if (comment.indexOf('\r') != -1 || comment.indexOf('\n') != -1) {
			// Comment is not allowed to contain newlines or line feeds.
			// Split comment in individual lines and write comment lines
			// for each of them.
			StringTokenizer st = new StringTokenizer(comment, "\r\n");
			while (st.hasMoreTokens()) {
				_writeCommentLine(st.nextToken());
			}
		}
		else {
			_writeCommentLine(comment);
		}
	}

/*---------------------------------+
| Other methods                    |
+---------------------------------*/

	private void _writeNamespace(String prefix, String name)
		throws IOException
	{
		_out.write("@prefix ");
		_out.write(prefix);
		_out.write(": <");
		_out.write(TurtleUtil.encodeURIString(name));
		_out.write("> .");
		_writeNewLine();
	}

	private void _writeCommentLine(String line)
		throws IOException
	{
		_out.write("# ");
		_out.write(line);
		_writeNewLine();
	}

	private void _writePredicate(URI predicate)
		throws IOException
	{
		if (predicate.getURI().equals(RDF.TYPE)) {
			// Write short-cut for rdf:type
			_out.write("a");
		}
		else {
			_writeURI(predicate);
		}
	}

	private void _writeValue(Value val)
		throws IOException
	{
		if (val instanceof Resource) {
			_writeResource( (Resource)val );
		}
		else {
			_writeLiteral( (Literal)val );
		}
	}

	private void _writeResource(Resource res)
		throws IOException
	{
		if (res instanceof URI) {
			_writeURI( (URI)res );
		}
		else {
			_writeBNode( (BNode)res );
		}
	}

	private void _writeURI(URI uri)
		throws IOException
	{
		String namespace = uri.getNamespace();
		String localName = uri.getLocalName();
		String prefix = (String)_namespaceTable.get(namespace);

		if (prefix != null && TurtleUtil.isLegalName(localName)) {
			// Namespace is mapped to a prefix and localName is
			// a legal Turtle name: write abbreviated URI
			_out.write(prefix);
			_out.write(":");
			_out.write(localName);
		}
		else {
			// Write full URI
			_out.write("<");
			_out.write( TurtleUtil.encodeURIString(uri.getURI()) );
			_out.write(">");
		}
	}

	private void _writeBNode(BNode bNode)
		throws IOException
	{
		_out.write("_:");
		_out.write(bNode.getID());
	}

	private void _writeLiteral(Literal lit)
		throws IOException
	{
		if (lit.getLabel().indexOf('\n') >= 0 ||
			lit.getLabel().indexOf('\r') >= 0 ||
			lit.getLabel().indexOf('\t') >= 0)
		{
			// Write label as long string
			_out.write("\"\"\"");
			_out.write(TurtleUtil.encodeLongString(lit.getLabel()));
			_out.write("\"\"\"");
		}
		else {
			// Write label as normal string
			_out.write("\"");
			_out.write(TurtleUtil.encodeString(lit.getLabel()));
			_out.write("\"");
		}
	
		if (lit.getDatatype() != null) {
			// Append the literal's datatype (possibly written as an abbreviated URI)
			_out.write("^^");
			_writeURI(lit.getDatatype());
		}
		else if (lit.getLanguage() != null) {
			// Append the literal's language
			_out.write("@");
			_out.write(lit.getLanguage());
		}
	}

	private void _writeIndent()
		throws IOException
	{
		_out.write("\t");
	}

	private void _writeNewLine()
		throws IOException
	{
		_out.write("\n");
	}

	private void _closePreviousStatement()
		throws IOException
	{
		if (!_statementClosed) {
			// The previous statement still needs to be closed:
			_out.write(" .");
			_writeNewLine();
			_writeNewLine();

			_statementClosed = true;
			_lastWrittenSubject = null;
			_lastWrittenPredicate = null;
		}
	}
}
